package com.kidoz.events;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.res.Configuration;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.DisplayMetrics;
import android.util.SparseArray;
import android.view.WindowManager;

import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.general.utils.SdkDeviceUtils;
import com.kidoz.sdk.api.general.utils.Utils;

import org.json.JSONObject;

import java.util.Calendar;
import java.util.Locale;

public class EventManager
{
    public static int LOG_CRITICAL_LEVEL = 0;
    public static int LOG_NORMAL_LEVEL = 1;

    private static final int NIGHT_TIME_HOUR_VALUE_START = 21;
    private static final int NIGHT_TIME_HOUR_VALUE_END = 8;
    private static final int DAY_TIME_AMOUNT = 10;
    private static final int NIGHT_TIME_AMOUNT_EXTRA = 50;
    private static final int THREE_G_CONNECTON_TYPE_AMOUNT = 10;
    private static final int WIFI_CONNECTON_TYPE_AMOUNT_EXTRA = 20;
    private static final int SCREEN_STATE_ON_AMOUNT = 10;
    private static final int SCREEN_STATE_OFF_AMOUNT_EXTRA = 20;
    private static final String OS_TYPE = "Android";

    public static final String EVENTS_SHARED_PREFERENCES_KEY = "events_shared_preferences";
    public static final String EVENTS_LAST_SESSION_ID_KEY = "last_session_id";
    public static final String LOG_LEVEL_KEY = "log_level";

    // Variables Definitions
    public static int SDK_STYLE_VERSION_NUM = 0;

    private static EventManager mEventManager;
    private int mLogLevel;
    private long mSessionID;

    private String mDeveloperID;
    private SparseArray<Event> mOpenedEventsSet;
    private EventLogDatabaseManager mDatabaseManager;

    private static Utils.StaticHandler mSyncEventHandler;
    private EventSyncAsyncTask mEventSyncAsyncTask;
    private String mUserRefferal;

    private EventManager()
    {
        mOpenedEventsSet = new SparseArray<>();
    }

    public static EventManager getInstance(Context context)
    {
        if (mEventManager == null)
        {

            mEventManager = new EventManager();
            mEventManager.init(context, null);
        }
        return mEventManager;
    }

    public static EventManager getInstance(Context context,JSONObject params)
    {
        if (mEventManager == null)
        {
            mEventManager = new EventManager();
            mEventManager.init(context, params);
        }
        return mEventManager;
    }

    private EventLogDatabaseManager getEventDB(Context context)
    {
        if (mDatabaseManager == null)
        {
            mDatabaseManager = new EventLogDatabaseManager(context);
        }
        return mDatabaseManager;
    }

    /**
     * Get and initiate handler instance if needed
     */
    private Handler getMyHandler()
    {
        if (mSyncEventHandler == null)
        {
            mSyncEventHandler = new Utils.StaticHandler(Looper.getMainLooper())
            {
                @Override
                public void handleMessage(Message message)
                {
                    // This is where you do your work in the UI thread.
                    // Your worker tells you in the message what to do.
                }
            };
        }
        return mSyncEventHandler;
    }

    /**
     * this function init the event manager.
     *
     * @param params
     */
    public void init(Context context, JSONObject params)
    {
        mUserRefferal = SdkDeviceUtils.getDeviceReferral(context);

        try
        {
            /** Used specific for kidoz os and its descendants to retrieve the correct referral if exists */
            int resId = context.getResources().getIdentifier("PROPERTY_APPLICATION_REFFERAL", "string", context.getPackageName());
            if(resId != 0) {
                mUserRefferal = context.getString(resId);
            }
        } catch (Exception e)
        {

        }

        SharedPreferences eventsPref = context.getSharedPreferences(EVENTS_SHARED_PREFERENCES_KEY, 0);
        try
        {
            if (params != null)
            {
                mLogLevel = params.optInt(LOG_LEVEL_KEY, EventManager.LOG_CRITICAL_LEVEL);
                mDeveloperID = params.optString(EventParameters.DEVELOPER_ID, "");
                mSessionID = params.optInt(EventParameters.SESSION_ID, 0);
                //todo: save new values to preference and reload them
                SharedPreferences.Editor editor = eventsPref.edit();
                editor.putInt(LOG_LEVEL_KEY, mLogLevel);
                editor.putString(EventParameters.DEVELOPER_ID, mDeveloperID);
                editor.apply();
            } else
            {
                mLogLevel = eventsPref.getInt(LOG_LEVEL_KEY, EventManager.LOG_CRITICAL_LEVEL);
                mDeveloperID = eventsPref.getString(EventParameters.DEVELOPER_ID, "");
                mSessionID = -1;
            }

            if (mDatabaseManager == null)
            {
                mDatabaseManager = new EventLogDatabaseManager(context);
            }
            com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog("init Event manager");
        } catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager init " + ex.getMessage());
        }
    }//end init function

    /**
     * This method return wherever a log can be sent.
     *
     * @return is can send log.
     */
    public boolean getIsNeedEventsSync(Context context)
    {
        if (getEventDB(context).getEventsTable().isDBempty())
        {
            return false;
        } else
        {
            return true;
        }
    }

    public EventBulk getNextBulk(Context context)
    {
        EventBulk result = new EventBulk();
        result.setDeviceParamsJSONObject(getBulkDeviceParamsJsonObject(context));
        result.setEventArrayList(getEventDB(context).getEventsTable().getEvents(calculateBulkSize(context)));

        return result;
    }

    public void removeBulkFromDB(Context context, EventBulk eventBulk)
    {
        getEventDB(context).getEventsTable().removeLogEventsFromDB(eventBulk.getEventArrayList());
    }

    private int calculateBulkSize(Context context)
    {
        // The minimum bulk is 30, the maximum is 120.
        int result = DAY_TIME_AMOUNT + THREE_G_CONNECTON_TYPE_AMOUNT + SCREEN_STATE_ON_AMOUNT;
        Calendar calendar = Calendar.getInstance();
        int currentHour = calendar.get(Calendar.HOUR_OF_DAY);
        if (currentHour >= NIGHT_TIME_HOUR_VALUE_START || currentHour <= NIGHT_TIME_HOUR_VALUE_END)
        {
            result += NIGHT_TIME_AMOUNT_EXTRA;
        }

        /*if (BaseConnectionClient.getConnectionType(context) == ConnectivityManager.TYPE_WIFI)
        {
            result += WIFI_CONNECTON_TYPE_AMOUNT_EXTRA;
        }*/

        if (ScreenUtils.getIsScreenOff(context) == true)
        {
            result += SCREEN_STATE_OFF_AMOUNT_EXTRA;
        }

        return result;
    }

    /**
     * This method create basic event
     */
    public int logClickEvent(Context context, String widgetType, String styleId, String action, String label, String itemID, int position)
    {
        Event event = new Event();
        event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
        event.addParameterToJsonObject(EventParameters.ITEM_INDEX, position);
        internalLogEvent(context, widgetType, styleId, EventManager.LOG_NORMAL_LEVEL, event, EventParameters.CATEGORY_CONTENT_CLICK, action, label);

        return 0;
    }//end logEvent function

    /**
     * This method create basic event
     */
    public int logEvent(Context context, String widgetType, String styleId, int logLevel, String category, String action, String label)
    {
        internalLogEvent(context, widgetType, styleId, logLevel, null, category, action, label);
        return 0;
    }//end logEvent function

    /**
     * This method create event with duration. if a log entry with the same resource key ends
     * the previous event and start a new one. the return value is the resource id.
     */
    public int logEventWithDurationStart(String widgetType, String styleId, int logLevel, int resourceID, String category, String action, String label, String itemID)
    {
        try
        {
            Event openedEvent = new Event();
            fillBaseEventParameters(openedEvent, category, action, label, widgetType, styleId);

            openedEvent.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
            openedEvent.setLogLevel(logLevel);

            mOpenedEventsSet.put(resourceID, openedEvent);
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog("Error on EventManager logEventWithDuration " + ex.getMessage());
        }

        return 0;
    }//end logEvent function

    /**
     * This method ends the event with duration based on the resource ID
     */
    public int logEventWithDurationEnd(Context context, String widgetType, String styleId, int resourceID)
    {
        try
        {
            Event openedEvent = mOpenedEventsSet.get(resourceID);
            if (openedEvent != null)
            {
                int duration = (int) (System.currentTimeMillis() - openedEvent.getEventCreationTime());
                openedEvent.addParameterToJsonObject(EventParameters.DURATION, duration);
                internalLogEvent(context, widgetType, styleId, openedEvent.getLogLevel(), openedEvent, null, null, null);
                mOpenedEventsSet.remove(resourceID);
            }
            SDKLogger.printInfoLog("activity end event called");
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog("Error on EventManager logEventWithDuration " + ex.getMessage());
        }
        return 0;
    }//end logEvent function

    public int logSponsoredContentImpressionEvent(Context context, String widgetType, String styleId, String action, String label, String advertiserID, String itemID, int positionIndex)
    {
        try
        {
            Event event = new Event();
            event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
            event.addParameterToJsonObject(EventParameters.ADVERTISER_ID, advertiserID);
            event.addParameterToJsonObject(EventParameters.ITEM_INDEX, positionIndex);
            internalLogEvent(context, widgetType, styleId, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_SPONSORED_CONTENT, action, label);
            return 0;
        } catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logImpressionEvent " + ex.getMessage());
        }
        return -1;
    }

    public int logOrganicContentImpressionEvent(Context context, String widgetType, String styleId, String action, String label, String itemID, int positionIndex)
    {
        try
        {
            Event event = new Event();
            event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
            event.addParameterToJsonObject(EventParameters.ITEM_INDEX, positionIndex);
            internalLogEvent(context, widgetType, styleId, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_ORGANIC_CONTENT, action, label);
            return 0;
        } catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logImpressionEvent " + ex.getMessage());
        }
        return -1;
    }

    public int logSponsoredContentClickEvent(Context context, String widgetType, String styleId, String label, String advertiserID, String itemID, int positionIndex)
    {
        try
        {
            Event event = new Event();
            event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
            event.addParameterToJsonObject(EventParameters.ADVERTISER_ID, advertiserID);
            event.addParameterToJsonObject(EventParameters.ITEM_INDEX, positionIndex);
            internalLogEvent(context, widgetType, styleId, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_CLICK, label);
            return 0;
        } catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logImpressionEvent " + ex.getMessage());
        }
        return -1;
    }

    public int logSponsoredConversionEvent(Context context, String widgetType, String styleId, Event event, String label, String advertiserID, String itemID, int positionIndex, int rewardKey, int rewardValue)
    {
        try
        {

            if (event == null)
            {
                event = new Event();
            }
            event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
            event.addParameterToJsonObject(EventParameters.ADVERTISER_ID, advertiserID);
            event.addParameterToJsonObject(EventParameters.ITEM_INDEX, positionIndex);
            event.addParameterToJsonObject(EventParameters.INT_FIELD, rewardKey);
            event.addParameterToJsonObject(EventParameters.INT_FIELD_2, rewardValue);

            internalLogEvent(context, widgetType, styleId, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_CONVERSION, label);
            return 0;
        } catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logSponsoredConversionEvent " + ex.getMessage());
        }
        return -1;
    }

    public int logSponsoredContentInstallEvent(Context context, String widgetType, String styleId, String label, String advertiserID, String itemID, int positionIndex)
    {
        try
        {
            Event event = new Event();
            event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
            event.addParameterToJsonObject(EventParameters.ADVERTISER_ID, advertiserID);
            event.addParameterToJsonObject(EventParameters.ITEM_INDEX, positionIndex);
            internalLogEvent(context, widgetType, styleId, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_PROMOTED_APP_INSTALL_COMPLETE, label);
            return 0;
        } catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logImpressionEvent " + ex.getMessage());
        }
        return -1;
    }

    /**
     * This method is used to log basic event
     */
    public int LogEventWithIntField(Context context, String widgetType, String styleId, int logLevel, String category, String action, String label, int intField)
    {
        Event event = new Event();
        event.addParameterToJsonObject(EventParameters.INT_FIELD, intField);
        return (internalLogEvent(context, widgetType, styleId, logLevel, event, category, action, label));
    }

    /**
     * This method is used to log basic event
     */
    public int logEvent(Context context, String widgetType, String styleId, int logLevel, Event event, String category, String action, String label)
    {
        return (internalLogEvent(context, widgetType, styleId, logLevel, event, category, action, label));
    }

    /**
     * This method create create the events and add it to que
     */
    private int internalLogEvent(Context context, String widgetType, String styleId, int logLevel, Event event, String category, String action, String label)
    {
        Event tempEvent;

        if (logLevel <= mLogLevel)
        {
            if (event == null)
            {
                tempEvent = new Event();
            } else
            {
                tempEvent = event;
            }
            fillBaseEventParameters(tempEvent, category, action, label, widgetType, styleId);

            //SDKLogger.printInfoLog("Event sent to DB: " + tempEvent.getJSONObject().toString());

            try
            {
                EventLogDatabaseManager manager = getEventDB(context);
                if (manager != null)
                {
                    manager.getEventsTable().insertLogEventToDB(tempEvent);
                }
            } catch (Exception e)
            {
            }
        }
        return 0;
    }

    private int fillBaseEventParameters(Event event, String category, String action, String label, String widgetType, String styleId)
    {
        if (category != null)
        {
            event.addParameterToJsonObject(EventParameters.CATEGORY, category);
        }
        if (action != null)
        {
            event.addParameterToJsonObject(EventParameters.ACTION, action);
        }
        if (label != null)
        {
            event.addParameterToJsonObject(EventParameters.LABEL, label);
        }

        event.addParameterToJsonObject(EventParameters.SESSION_ID, mSessionID);

        if (widgetType != null)
        {
            event.addParameterToJsonObject(EventParameters.WIDGET_TYPE, widgetType);
        }

        if (styleId != null)
        {
            event.addParameterToJsonObject(EventParameters.STYLE_ID, styleId);
        }

        event.addParameterToJsonObject(EventParameters.ACTUAL_SDK_VERSION, ConstantDef.ACTUAL_SDK_VERSION);

        return 0;
    }

    private JSONObject getBulkDeviceParamsJsonObject(Context context)
    {
        JSONObject result = new JSONObject();
        if (context != null)
        {
            try
            {
                if (mDeveloperID == null)
                {
                    SharedPreferences eventsPref = context.getSharedPreferences(EVENTS_SHARED_PREFERENCES_KEY, 0);
                    mDeveloperID = eventsPref.getString(EventParameters.DEVELOPER_ID, "");
                }

                result.put(EventParameters.DEVICE_HASH, Utils.generateUniqeDeviceID(context.getPackageName(), mDeveloperID));

                result.put(EventParameters.USER_REFFERAL,mUserRefferal);

                result.put(EventParameters.KIDOZ_APP_PACKAGE_NAME, context.getPackageName());
                PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
                result.put(EventParameters.APP_VERSION, String.valueOf(pInfo.versionName));
                result.put(EventParameters.APP_ID, String.valueOf(pInfo.packageName));

                result.put(EventParameters.DEVELOPER_ID, mDeveloperID);
                result.put("EventVersion", 0);

                //todo: this data should be sent only once
                result.put(EventParameters.DEVICE_TYPE, String.valueOf(Build.DEVICE));
                result.put(EventParameters.DEVICE_BRAND, String.valueOf(Build.MANUFACTURER));
                result.put(EventParameters.DEVICE_MODEL, String.valueOf(Build.MODEL));
                result.put(EventParameters.SCREEN_TYPE, String.valueOf((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK)));
                result.put(EventParameters.SCREEN_SIZE, String.valueOf(ScreenUtils.getDeviceScreenSizeInInches(context)));

                WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
                DisplayMetrics metrics = new DisplayMetrics();
                windowManager.getDefaultDisplay().getMetrics(metrics);

                result.put(EventParameters.DPI_FACTOR, String.valueOf(metrics.densityDpi));
                result.put(EventParameters.SCREEN_WIDTH, String.valueOf(SdkDeviceUtils.getScreenSize(context, true)));
                result.put(EventParameters.SCREEN_HEIGHT, String.valueOf(SdkDeviceUtils.getScreenSize(context, false)));
                result.put(EventParameters.OS_TYPE, String.valueOf(OS_TYPE));
                result.put(EventParameters.OS_VERSION, String.valueOf(Build.VERSION.SDK_INT));
                //result.put(EventParameters.CONNECTION_TYPE, EventParameters.convertConnectionTypeToString(BaseConnectionClient.getConnectionType(context)));
                result.put(EventParameters.DEVICE_LANGUAGE, String.valueOf(Locale.getDefault().getLanguage()));
                result.put(EventParameters.TIME_ZONE, EventParameters.getCurrentTimezoneOffset());
                result.put(EventParameters.COUNTRY, String.valueOf(Locale.getDefault().getCountry()));
                result.put(EventParameters.EVENT_TYPE, EventParameters.EVENT_TYPE_DEVICE_INFO);

                result.put(EventParameters.ACTUAL_SDK_VERSION, ConstantDef.ACTUAL_SDK_VERSION);
                result.put(EventParameters.SDK_EXTENSION_TYPE, ConstantDef.SDK_EXTENSION_TYPE);
            } catch (Exception ex)
            {
                SDKLogger.printErrorLog("Error when trying to create device params: " + ex.getMessage());
            }
        }

        return result;
    }

    /**
     * This method simply start the SyncEventService.
     *
     * @param context
     */
    public void startEventsSync(final Context context)
    {
        getMyHandler().post(new Runnable()
        {
            @Override
            public void run()
            {
                if (mEventSyncAsyncTask == null || mEventSyncAsyncTask.isCancelled() == true || mEventSyncAsyncTask.getStatus() == AsyncTask.Status.FINISHED)
                {
                    mEventSyncAsyncTask = new EventSyncAsyncTask(context);
                    if (Build.VERSION.SDK_INT < 11)
                    {
                        mEventSyncAsyncTask.execute();
                    } else
                    {
                        mEventSyncAsyncTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
                    }
                }
            }
        });
    }

    /**
     * This method simply start the SyncEventService.
     *
     * @param context
     */
    public void startEventsSync(final Context context, int posDelayMillis)
    {
        getMyHandler().postDelayed(new Runnable()
        {
            @Override
            public void run()
            {
                if (mEventSyncAsyncTask == null || mEventSyncAsyncTask.isCancelled() == true || mEventSyncAsyncTask.getStatus() == AsyncTask.Status.FINISHED)
                {
                    mEventSyncAsyncTask = new EventSyncAsyncTask(context);
                    if (Build.VERSION.SDK_INT < 11)
                    {
                        mEventSyncAsyncTask.execute();
                    } else
                    {
                        mEventSyncAsyncTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
                    }
                }
            }
        }, posDelayMillis);
    }

}
