package com.kidoz.sdk.api.ui_views.one_item_view;

import android.content.Context;
import android.graphics.Color;
import android.graphics.Point;
import android.os.Looper;
import android.support.v4.view.ViewPager;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.assets_handling.AssetUtil;
import com.kidoz.sdk.api.general.assets_handling.StyleParser;
import com.kidoz.sdk.api.general.custom_views.CustomCardView.KidozCardView;
import com.kidoz.sdk.api.general.enums.ContentType;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.interfaces.AnotherInterface;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.html_view.HtmlManager;
import com.kidoz.sdk.api.ui_views.html_view.HtmlViewWrapper;
import com.kidoz.sdk.api.ui_views.html_view.IOnHtmlWebViewInterface;
import com.kidoz.sdk.api.ui_views.html_view.MaximizedViewDialog;
import com.kidoz.sdk.api.ui_views.loading_progress_view.LoadingProgressView;
import com.kidoz.sdk.api.ui_views.panel_view.OneContentItemView;
import com.kidoz.sdk.api.ui_views.parental_lock.AssetView;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

public class OneItemFamilyView extends RelativeLayout
{
    public enum PlaybackState
    {
        PLAYING, STOPED
    }

    public enum OneItemAction
    {
        STOP_ANY_ANIMATION, RESTART_ANY_ANIMATION, ON_RESET_AUTO_LAUNCH
    }

    private Lock mLock;

    protected static final float LOADING_PROGRESS_DEFAULT_RATIO = 0.35f;
    protected static final int ITEM_CONTAINER_STROKE_WIDTH = 4;
    protected int AUTO_SCROLL_DELAY = 12000;

    public AutoScrollViewPager mViewPager;
    protected LoadingProgressView mLoadingProgressView;
    protected AssetView mMaximizeBtn;
    // Default html view with interface handling
    protected HtmlViewWrapper mHtmlDefaultView;
    // Special html view for loading full pages without any interactions
    protected HtmlViewWrapper mHtmlSpecialView;
    public KidozCardView mViewPagerCardContainerView;
    private Utils.StaticHandler staticHandler;

    protected ItemViewPagerAdapter mAdapter;

    protected HashMap<String, Integer> mContentDisplayed = new HashMap<>();
    protected ItemViewPagerAdapter.ViewPagerItemClickListener mViewPagerItemClickListener;

    protected String mMaximizeBtnUrl;
    protected int mThumbContainerBgColor;

    protected JSONObject mProperties;
    protected String mWidgetType;
    protected String mStyleId;

    protected boolean mIsEnabled = false;
    protected PlaybackState mPlaybackState = PlaybackState.STOPED;

    protected ContentItem mCurrentItemInView;
    protected int mCurrentViewPagerIndexInFocus = 0;
    protected int mPreviousViewPagerIndexInFocus = -1;

    protected MaximizedViewDialog mMaximizedViewDialog;
    protected IOnOneItemActionListener mOneItemActionListener;

    protected int mWidth;
    protected int mHeight;

    private IOnRefreshListener mRefreshListener;

    public OneItemFamilyView(Context context, String widgetType, String styleId, JSONObject properties, IOnOneItemActionListener listener)
    {
        super(context);
        staticHandler = new Utils.StaticHandler(Looper.getMainLooper());
        mLock = new ReentrantLock();

        mProperties = properties;
        mWidgetType = widgetType;
        mStyleId = styleId;
        mOneItemActionListener = listener;

        applyStyleProperties();

        initView();

        Utils.setOnGlobalLayoutFinishListener(this, new Utils.OnGlobalLayoutFinishedListener()
        {
            @Override
            public void onLayoutFinished()
            {
                mWidth = getMeasuredWidth();
                mHeight = getMeasuredHeight();

                mAdapter = new ItemViewPagerAdapter(mProperties, mWidth, mHeight);
                mAdapter.setViewPagerItemClickListener(mViewPagerItemClickListener);
                mViewPager.setAdapter(mAdapter);
                mAdapter.setContent(new ArrayList<ContentItem>());
                mAdapter.notifyDataSetChanged();
                mAdapter.setOnItemInstantiateFinishListener(new ItemViewPagerAdapter.IOnItemInstaniateFinishListener()
                {
                    @Override
                    public void onFirstItemReady()
                    {
                        // Used only for family feed because it openes starin on initiation, as for panel the items instaniated while panel not visible
                        if (mWidgetType.equals(WidgetType.WIDGET_TYPE_FEED.getStringValue()))
                        {
                            refreshNextViewItem(0);
                        }
                    }
                });

                int size = (int) (mWidth * 0.22f);

                mMaximizeBtn.getLayoutParams().width = size;
                mMaximizeBtn.getLayoutParams().height = size;
            }
        });
    }

    protected void applyStyleProperties()
    {
        if (mProperties != null)
        {
            mMaximizeBtnUrl = mProperties.optString(StyleParser.HTML_MAXIMIZE_BTN, null);
            AUTO_SCROLL_DELAY = mProperties.optInt(StyleParser.SWAP_CONTENT_REFRESH_RATE_SECONDS, 12) * 1000;
            mThumbContainerBgColor = Color.parseColor(mProperties.optString(StyleParser.THUMBNAIL_BG_COLOR, "#ccffffff"));
        }
    }

    protected void initView()
    {
        initPagerContainerCard();
        initViewPager();
        initHtmlDefaultWebView();
        initHtmlSpecialWebView();

        initLoadingProgressView();
        initMaximizeButton();
    }

    private void initPagerContainerCard()
    {
        mViewPagerCardContainerView = new KidozCardView(getContext());
        mViewPagerCardContainerView.setId(Utils.generateViewId());

        mViewPagerCardContainerView.setRadius(Utils.dpTOpx(getContext(), 4));
        mViewPagerCardContainerView.setCardBackgroundColor(mThumbContainerBgColor);

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
        addView(mViewPagerCardContainerView, params);
    }

    private void initHtmlDefaultWebView()
    {
        mHtmlDefaultView = HtmlManager.getDefaultHtmlView(getContext());
        if (mHtmlDefaultView != null)
        {
            mHtmlDefaultView.setInFocusActivityContext(getContext());
            mHtmlDefaultView.setWidgetType(mWidgetType);
            mHtmlDefaultView.setStyleID(mStyleId);

            applyListenerToDefaultHtmlView();
        }
    }

    // We set listener to the HTML view
    private void applyListenerToDefaultHtmlView()
    {
        if (mHtmlDefaultView != null)
        {
            mHtmlDefaultView.setHtmlWebViewListener(new IOnHtmlWebViewInterface()
            {
                @Override
                public void toggleWidgetState(boolean isEnabled)
                {
                    if (mAnotherInterface != null)
                    {
                        mAnotherInterface.toggleWidgetState(isEnabled);
                    }
                }

                @Override
                public void onOpenMaximized()
                {
                    openMaximizedDialogVew(mCurrentItemInView);
                }

                @Override
                public void onDonePlayBack()
                {
                    startAutoScroll();
                    nextPagerItem();
                }

                @Override
                public void onHtmlFinishedLoading()
                {
                    super.onHtmlFinishedLoading();

                    staticHandler.postDelayed(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            if (mRefreshListener != null)
                            {
                                mRefreshListener.onRefreshDrawRequest();
                            }
                        }
                    }, 100);
                }
            });
        }
    }

    /**
     * Loadable web view that loads special HTML's and not using an preloaded one
     */
    private void initHtmlSpecialWebView()
    {
        mHtmlSpecialView = new HtmlViewWrapper(getContext(), false);
        mHtmlSpecialView.setInFocusActivityContext(getContext());
        mHtmlSpecialView.setWidgetType(mWidgetType);
        mHtmlSpecialView.setStyleID(mStyleId);

        applyListenerToSpecialHtmlView();

        mHtmlSpecialView.setVisibility(View.GONE);
    }

    // We set listener to the HTML view
    private void applyListenerToSpecialHtmlView()
    {
        if (mHtmlSpecialView != null)
        {
            mHtmlSpecialView.setHtmlWebViewListener(new IOnHtmlWebViewInterface()
            {
                @Override
                public void toggleWidgetState(boolean isEnabled)
                {
                    if (mAnotherInterface != null)
                    {
                        mAnotherInterface.toggleWidgetState(isEnabled);
                    }
                }

                @Override
                public void onHtmlStartLoading()
                {
                    if (mHtmlSpecialView.getVisibility() == View.VISIBLE)
                    {
                        mHtmlSpecialView.showLoadingProgressView();
                    }

                    mMaximizeBtn.setVisibility(View.INVISIBLE);
                }

                @Override
                public void onHtmlFinishedLoading()
                {
                    if (mHtmlSpecialView.getVisibility() == View.VISIBLE)
                    {
                        mHtmlSpecialView.hideLoadingProgressView();
                    }

                    if (mCurrentItemInView != null)
                    {
                        if (mCurrentItemInView.getShouldDisplayMaximizeButton())
                        {
                            mMaximizeBtn.setVisibility(View.VISIBLE);
                        } else
                        {
                            mMaximizeBtn.setVisibility(View.INVISIBLE);
                        }
                    }
                }
            });
        }
    }

    private void initMaximizeButton()
    {
        mMaximizeBtn = new AssetView(getContext());

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        params.addRule(RelativeLayout.ALIGN_RIGHT, mViewPagerCardContainerView.getId());
        params.addRule(RelativeLayout.ALIGN_BOTTOM, mViewPagerCardContainerView.getId());
        params.setMargins(0, 0, Utils.dpTOpx(getContext(), 5), Utils.dpTOpx(getContext(), 5));

        addView(mMaximizeBtn, params);
        mMaximizeBtn.setVisibility(View.INVISIBLE);

        mMaximizeBtn.setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View v)
            {
                GenAnimator.clickItemAnimation(v, 70, new GenAnimator.ViewAnimationListener()
                {
                    @Override
                    public void onAnimationEnd()
                    {
                        openMaximizedDialogVew(mCurrentItemInView);
                    }

                    @Override
                    public void onAnimationStart()
                    {
                    }
                });
            }
        });

        mMaximizeBtn.loadAsset(AssetUtil.getAssetFile(getContext(), mMaximizeBtnUrl), null);
    }

    /**
     * Refresh and render next item
     */
    private void refreshNextViewItem(int viewPagerIndex)
    {
        if(mAdapter.getRealCount() > 0) {
            int realContentPosition = viewPagerIndex % mAdapter.getRealCount();
            //SDKLogger.printWarningLog("###@@@###  Refresh View Pager Item Index : " + viewPagerIndex);
            if (mIsEnabled)
            {
                stopAutoScroll();

                mCurrentViewPagerIndexInFocus = viewPagerIndex;
                sendImpressionLog(realContentPosition);

                // Report Focus removed on HTML view that was swapped form the screen
                if (mPreviousViewPagerIndexInFocus != -1 && mPreviousViewPagerIndexInFocus != mCurrentViewPagerIndexInFocus)
                {
                    ContentItem prevItem = mAdapter.getContentItem(mPreviousViewPagerIndexInFocus % mAdapter.getRealCount());
                    if (prevItem != null && prevItem.getContentType() == ContentType.HTML)
                    {
                        if (prevItem.getIsNonDefaultHtml())
                        {
                            mHtmlSpecialView.clearHtmlView();
                        } else
                        {
                            mHtmlDefaultView.requestFocusOff();
                        }
                    }

                    View view = mViewPager.findViewWithTag(mPreviousViewPagerIndexInFocus);
                    if (view != null)
                    {
                        if (view != null)
                        {
                            ((OneContentItemView) view).clearActiveWebView();
                        }
                    }

                    mPreviousViewPagerIndexInFocus = -1;
                }

                mCurrentItemInView = mAdapter.getContentItem(realContentPosition);
                if (mCurrentItemInView != null && mCurrentItemInView.getContentType() == ContentType.HTML)
                {
                    View view = mViewPager.findViewWithTag(mCurrentViewPagerIndexInFocus);
                    if (view != null)
                    {
                        final OneContentItemView panelContentItem = (OneContentItemView) view;

                        // MEans its not a default loaded HTML
                        if (mCurrentItemInView.getIsNonDefaultHtml())
                        {
                            if (panelContentItem != null)
                            {
                                panelContentItem.setActiveWebView(mHtmlSpecialView);
                            }

                            mHtmlDefaultView.setVisibility(INVISIBLE);
                            mHtmlSpecialView.setVisibility(VISIBLE);
                            mHtmlSpecialView.clearHtmlView();
                            mHtmlSpecialView.setData(mCurrentItemInView);
                            mHtmlSpecialView.reloadHtml();
                            if (mOneItemActionListener != null)
                            {
                                mOneItemActionListener.onAction(OneItemAction.RESTART_ANY_ANIMATION);
                            }
                        } // The items i sued ny default loaded HTML
                        else
                        {
                            mHtmlSpecialView.setVisibility(INVISIBLE);
                            if (mHtmlDefaultView != null)
                            {
                                mHtmlDefaultView.setData(mCurrentItemInView);
                                if (panelContentItem != null)
                                {
                                    panelContentItem.setActiveWebView(mHtmlDefaultView);
                                }
                                mHtmlDefaultView.setVisibility(VISIBLE);

                                staticHandler.postDelayed(new Runnable()
                                {
                                    @Override
                                    public void run()
                                    {
                                        // The delay is added due to prevent inner crush of webview when switching fast on html items
                                        // Somewhere in webview there is incosistency in calls of video play witch lead to crush
                                        // The delay helps to separate the calls of focusOff and focusOn and to avoid the problem(At least try)
                                        if (mRefreshListener != null)
                                        {
                                            mRefreshListener.onRefreshDrawRequest();
                                        }
                                        mHtmlDefaultView.requestFocusOn(false);
                                    }
                                },100);
                            }

                            if (mOneItemActionListener != null)
                            {
                                mOneItemActionListener.onAction(OneItemAction.STOP_ANY_ANIMATION);
                            }
                        }

                        if (mCurrentItemInView.getShouldDisplayMaximizeButton())
                        {
                            mMaximizeBtn.setVisibility(View.VISIBLE);
                        } else
                        {
                            mMaximizeBtn.setVisibility(View.INVISIBLE);
                        }

                        mPreviousViewPagerIndexInFocus = mCurrentViewPagerIndexInFocus;
                    }
                } else
                {
                    startAutoScroll();

                    if (mHtmlDefaultView != null)
                    {
                        //mHtmlDefaultView.requestFocusOff();
                        mHtmlDefaultView.setVisibility(INVISIBLE);
                    }

                    //mHtmlSpecialView.clearHtmlView();
                    mHtmlSpecialView.setVisibility(INVISIBLE);
                    mMaximizeBtn.setVisibility(INVISIBLE);

                    if (mOneItemActionListener != null)
                    {
                        mOneItemActionListener.onAction(OneItemAction.RESTART_ANY_ANIMATION);
                    }
                }
            }
        }
    }

    private void initLoadingProgressView()
    {
        mLoadingProgressView = new LoadingProgressView(getContext());
        mLoadingProgressView.setVisibility(View.GONE);

        Point point = Utils.getScreenSize(getContext());
        int size = (int) (Math.min(point.x, point.y) * LOADING_PROGRESS_DEFAULT_RATIO);

        LinearLayout layout = new LinearLayout(getContext());
        layout.setGravity(Gravity.CENTER);
        layout.addView(mLoadingProgressView, new LinearLayout.LayoutParams(size, size));

        RelativeLayout.LayoutParams prParams = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
        prParams.addRule(RelativeLayout.ALIGN_LEFT, mViewPagerCardContainerView.getId());
        prParams.addRule(RelativeLayout.ALIGN_RIGHT, mViewPagerCardContainerView.getId());
        addView(layout, prParams);
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh)
    {
        super.onSizeChanged(w, h, oldw, oldh);

        int size = (int) (Math.min(w, h) * LOADING_PROGRESS_DEFAULT_RATIO);
        mLoadingProgressView.getLayoutParams().height = size;
        mLoadingProgressView.getLayoutParams().width = size;
        mLoadingProgressView.setCircleWidthRelativeToSize(w);
    }

    private void initViewPager()
    {
        mViewPager = new AutoScrollViewPager(getContext());
        mViewPager.setInterval(AUTO_SCROLL_DELAY);
        mViewPager.addOnPageChangeListener(new ViewPager.OnPageChangeListener()
        {
            @Override
            public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels)
            {
                if (mRefreshListener != null)
                {
                    mRefreshListener.onRefreshDrawRequest();
                }
            }

            @Override
            public void onPageSelected(final int position)
            {
                if (mAdapter.getRealCount() > 0)
                {
                    mAdapter.toggleIsClickable(false);
                    staticHandler.removeCallbacks(null);

                    staticHandler.postDelayed(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            mAdapter.toggleIsClickable(true);
                        }
                    }, 750);

                    refreshNextViewItem(position);
                }
            }

            @Override
            public void onPageScrollStateChanged(int state)
            {
            }
        });

        mViewPager.setId(Utils.generateViewId());
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);

        mViewPager.setPageMargin(Utils.dpTOpx(getContext(), 3));
        mViewPager.setClipToPadding(false);

        int margin = Utils.dpTOpx(getContext(), ITEM_CONTAINER_STROKE_WIDTH);
        mViewPagerCardContainerView.setContentPadding(margin, margin, margin, margin);
        mViewPagerCardContainerView.addView(mViewPager, params);
    }

    private void sendImpressionLog(int position)
    {
        ContentItem contentItem = mAdapter.getContentItem(position);
        if (contentItem != null && mContentDisplayed != null)
        {
            if (mContentDisplayed.containsKey(contentItem.getId()) == false)
            {
                if (contentItem.getIsAvoidAutoImpression() == false)
                {
                    if (contentItem.isPromoted() == true)
                    {
                        EventManager.getInstance(getContext()).logSponsoredContentImpressionEvent(getContext(), mWidgetType, mStyleId, EventParameters.ACTION_IMPRESSION, contentItem.getName(), contentItem.getAdvertiserID(), contentItem.getId(), position);
                    } else
                    {
                        EventManager.getInstance(getContext()).logOrganicContentImpressionEvent(getContext(), mWidgetType, mStyleId, EventParameters.ACTION_IMPRESSION, contentItem.getName(), contentItem.getId(), position);
                    }
                    mContentDisplayed.put(contentItem.getId(), position);
                }
            }
        }
    }

    /**
     * Show or Hide loading progress view
     *
     * @param show
     */
    public void showLoadingProgressView(boolean show)
    {
        if (show)
        {
            mLoadingProgressView.startLoadingAnimation();
        } else
        {
            mLoadingProgressView.stopLoadingAnimation();
        }
    }

    public void setViewPagerItemClickListener(ItemViewPagerAdapter.ViewPagerItemClickListener viewPagerItemClickListener)
    {
        mViewPagerItemClickListener = viewPagerItemClickListener;
        mHtmlDefaultView.setViewPagerItemClickListener(mViewPagerItemClickListener);
        mHtmlSpecialView.setViewPagerItemClickListener(mViewPagerItemClickListener);
    }

    public void setContent(ArrayList<ContentItem> content)
    {
        if (mLock.tryLock())
        {
            mCurrentViewPagerIndexInFocus = 0;
            mPreviousViewPagerIndexInFocus = -1;
            try
            {
                // Reset Impression map
                mContentDisplayed.clear();
                if (mAdapter != null)
                {
                    mAdapter.setContent(content);
                    mViewPager.setCurrentItem(0);
                }
            } catch (Exception e)
            {
            } finally
            {
                mLock.unlock();
            }
        }
    }

    public void clearContent()
    {
        mContentDisplayed.clear();
        if (mAdapter != null)
        {
            mAdapter.clearContent();
        }
    }

    public void startAutoScroll()
    {
        if (mViewPager.isAutoScrollOn() == false)
        {
            mViewPager.startAutoScroll(AUTO_SCROLL_DELAY);
        }
    }

    public void stopAutoScroll()
    {
        mViewPager.stopAutoScroll();
    }

    public boolean isEmpty()
    {
        if (mAdapter == null)
        {
            return true;
        } else
        {
            return mAdapter.isAdapterEmpty();
        }
    }

    @Subscribe
    public void onHandleEvent(EventMessage event)
    {
        if (event != null)
        {
            if (event.getMessageType() == EventMessage.MessageType.HTML_FULL_VIEW_CLOSE && event.getAdditionalParam() != null && event.getAdditionalParam().equals(mWidgetType))
            {
                onMaximizedScreenPlayerViewClosed();
            }
        }
    }

    /**
     * Open maximized dialog view
     */
    protected void openMaximizedDialogVew(ContentItem contentItem)
    {
        stopAutoScroll();

        if (contentItem != null)
        {
            if (contentItem.getIsNonDefaultHtml())
            {
                mMaximizedViewDialog = new MaximizedViewDialog(getContext(), mWidgetType, mStyleId, mHtmlSpecialView);
            } else
            {
                mMaximizedViewDialog = new MaximizedViewDialog(getContext(), mWidgetType, mStyleId, mHtmlDefaultView);
            }

            mMaximizedViewDialog.loadHtmlView(contentItem);
            mMaximizedViewDialog.openDialog();
        }

        if (mOneItemActionListener != null)
        {
            mOneItemActionListener.onAction(OneItemAction.STOP_ANY_ANIMATION);
        }
    }

    /**
     * Invoked when a player of maximized view has been closed
     */
    protected void onMaximizedScreenPlayerViewClosed()
    {
        if (mCurrentItemInView.getIsNonDefaultHtml())
        {
            // Restore listener to html view
            applyListenerToSpecialHtmlView();

            mHtmlSpecialView.reloadHtml();

            if (mOneItemActionListener != null)
            {
                mOneItemActionListener.onAction(OneItemAction.RESTART_ANY_ANIMATION);
            }
        } else
        {
            // Restore listener to html view
            applyListenerToDefaultHtmlView();

            if (mCurrentViewPagerIndexInFocus != -1)
            {
                View view = mViewPager.findViewWithTag(mCurrentViewPagerIndexInFocus);
                if (view != null)
                {
                    OneContentItemView panelContentItem = (OneContentItemView) view;
                    panelContentItem.setActiveWebView(mHtmlDefaultView);
                    mHtmlDefaultView.requestMinimize();
                    if (mCurrentItemInView != null && mCurrentItemInView.getContentType() != ContentType.HTML)
                    {
                        if (mOneItemActionListener != null)
                        {
                            mOneItemActionListener.onAction(OneItemAction.RESTART_ANY_ANIMATION);
                        }
                    }
                }
            } else
            {
                if (mOneItemActionListener != null)
                {
                    mOneItemActionListener.onAction(OneItemAction.RESTART_ANY_ANIMATION);
                }
            }
        }
    }

    @Override
    protected void onAttachedToWindow()
    {
        super.onAttachedToWindow();

        if (EventBus.getDefault().isRegistered(this) == false)
        {
            EventBus.getDefault().register(this);
        }
    }

    @Override
    protected void onDetachedFromWindow()
    {
        super.onDetachedFromWindow();

        if (EventBus.getDefault().isRegistered(this))
        {
            EventBus.getDefault().unregister(this);
        }

        if (mHtmlSpecialView != null)
        {
            mHtmlSpecialView.stopAndReleaseWebView();
        }

        staticHandler.removeCallbacksAndMessages(null);
    }

    /**
     * Go to next pager item
     */
    public void nextPagerItem()
    {
        mViewPager.setCurrentItem(mViewPager.getCurrentItem() + 1);
    }

    @Override
    protected void onVisibilityChanged(View changedView, int visibility)
    {
        super.onVisibilityChanged(changedView, visibility);

        handleVisibilityChange(visibility);
    }

    /**
     * On Visibility change logic
     */
    public synchronized void handleVisibilityChange(int visibility)
    {
        if (visibility == View.VISIBLE)
        {
            // Restore current in focus activity to be used by html wrappers
            mHtmlSpecialView.setInFocusActivityContext(getContext());
            mHtmlDefaultView.setInFocusActivityContext(getContext());

            startPlaybackAction();
        } else
        {
            stopPlaybackAction(true);
            staticHandler.removeCallbacksAndMessages(null);
        }
    }

    /**
     * Go to previous pager item
     */
    public void previousItem()
    {
        mViewPager.setCurrentItem(mViewPager.getCurrentItem() - 1);
    }

    public synchronized void stopPlaybackAction(boolean focusOff)
    {
        stopAutoScroll();
        if (mIsEnabled)
        {
            if (mPlaybackState == PlaybackState.PLAYING)
            {
                mPlaybackState = PlaybackState.STOPED;
                if(mAdapter.getRealCount() > 0) {
                    ContentItem item = mAdapter.getContentItem(mCurrentViewPagerIndexInFocus % mAdapter.getRealCount());
                    if (item != null)
                    {
                        if (item.getContentType() == ContentType.HTML)
                        {
                            if (item.getIsNonDefaultHtml())
                            {
                                mHtmlSpecialView.clearHtmlView();
                            } else
                            {
                                if (focusOff)
                                {
                                    mHtmlDefaultView.requestFocusOff();
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    public synchronized void startPlaybackAction()
    {
        if (getVisibility() == View.VISIBLE)
        {
            if (mIsEnabled)
            {
                if (mPlaybackState == PlaybackState.STOPED)
                {
                    mPlaybackState = PlaybackState.PLAYING;
                    //SDKLogger.printWarningLog("###@@@###  Resume Playback Call");
                    if (mAdapter != null)
                    {
                        if (mAdapter.getRealCount() > 0)
                        {
                            refreshNextViewItem(mCurrentViewPagerIndexInFocus);
                        }
                    }
                }
            }
        }
    }

    public interface IOnOneItemActionListener
    {
        void onAction(OneItemAction action);
    }

    public void setOnRefreshDrawRequestListener(IOnRefreshListener requestListener)
    {
        mRefreshListener = requestListener;
    }

    public interface IOnRefreshListener
    {
        void onRefreshDrawRequest();
    }

    public void invokeWidgetOpen()
    {
        if (mHtmlDefaultView != null)
        {
            mHtmlDefaultView.requestWidgetOpen(mWidgetType);
        }
    }

    public void invokeWidgetClose()
    {
        if (mHtmlDefaultView != null)
        {
            mHtmlDefaultView.requestWidgetClose(mWidgetType);
        }
    }

    private AnotherInterface mAnotherInterface;

    public void setAnotherInterface(AnotherInterface anotherInterface)
    {
        mAnotherInterface = anotherInterface;
    }

    public void setOneItemEnableState(boolean enableState)
    {
        mIsEnabled = enableState;
    }
}
