package com.kidoz.sdk.api.general.assets_handling;

import android.content.Context;
import android.os.AsyncTask;
import android.os.Build;

import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.SdkDeviceUtils;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.server_connect.BaseConnectionClient;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.lang.ref.WeakReference;
import java.util.ArrayList;

public class AssetUtil
{
    private static final String TAG = AssetUtil.class.getSimpleName();

    private static final String PANEL_FAMILY_ASSETS = "panelFamily";
    private static final String FEED_FAMILY_ASSETS = "feedFamily";
    private static final String FLEXI_VIEW_ASSETS = "flexiView";
    private static final String INTERSTITIAL_ASSETS = "interstitial";
    private static final String VIDEO_UNIT_ASSETS = "videoUnit";
    private static final String GLOBAL_STYLE_PARAMS = "globalStyleParams";

    public static boolean parseAssets(Context context, String dataResponse)
    {
        boolean result = false;
        if (context != null && dataResponse != null)
        {
            try
            {
                JSONObject rootObject = new JSONObject(dataResponse);

                if (rootObject.has(GLOBAL_STYLE_PARAMS) == true)
                {
                    result = StyleParser.parseGlobalStyleData(context, rootObject.getJSONObject(GLOBAL_STYLE_PARAMS));
                } else if (rootObject.has(FEED_FAMILY_ASSETS) == true)
                {
                    result = StyleParser.parseFamilyFeed(context, rootObject.getJSONObject(FEED_FAMILY_ASSETS));
                } else if (rootObject.has(PANEL_FAMILY_ASSETS) == true)
                {
                    result = StyleParser.parseFamilyPanelData(context, rootObject.getJSONObject(PANEL_FAMILY_ASSETS));
                } else if (rootObject.has(FLEXI_VIEW_ASSETS) == true)
                {
                    result = StyleParser.parseFlexiViewData(context, rootObject.getJSONObject(FLEXI_VIEW_ASSETS));
                } else if (rootObject.has(INTERSTITIAL_ASSETS) == true)
                {
                    result = StyleParser.parseInterstitialData(context, rootObject.getJSONObject(INTERSTITIAL_ASSETS));
                } else if (rootObject.has(VIDEO_UNIT_ASSETS) == true)
                {
                    result = StyleParser.parseVideoUnit(context, rootObject.getJSONObject(VIDEO_UNIT_ASSETS));
                }
            } catch (Exception ex)
            {
                SDKLogger.printErrorLog(TAG, "Error when trying to parse assets: " + ex.getMessage());
            }
        }
        return result;
    }

    private static String createAssetFileName(String assetURL)
    {
        String result = null;
        if (assetURL != null)
        {
            result = assetURL.substring(assetURL.lastIndexOf("/") + 1, assetURL.length());
        }
        return result;
    }

    private static String createAssetFileName2(String assetURL)
    {
        String result = null;
        if (assetURL != null)
        {
            result = assetURL.replace("/", "");
            result += assetURL.substring(assetURL.lastIndexOf("/") + 1, assetURL.length());
        }
        return result;
    }

    public static boolean createAssetFile(Context context, String assetURL)
    {
        boolean result = true;
        if (assetURL != null && !assetURL.equals(""))
        {
            // 1. Extract the asset file name
            String fileName = createAssetFileName(assetURL);
            File storageLocation = SdkDeviceUtils.getInnerStoragePathIfPosible(context);
            File newFile = new File(storageLocation, fileName);
            if (newFile.exists() == false)
            {
                result = BaseConnectionClient.downloadAndSaveFile(assetURL, newFile);
            }
        }
        return result;
    }

    public static boolean createAssetFile2(Context context, String assetURL)
    {
        boolean result = true;
        if (assetURL != null && !assetURL.equals(""))
        {
            // 1. Extract the asset file name
            String fileName = createAssetFileName2(assetURL);
            File storageLocation = SdkDeviceUtils.getInnerStoragePathIfPosible(context);
            File newFile = new File(storageLocation, fileName);
            if (newFile.exists() == false)
            {
                result = BaseConnectionClient.downloadAndSaveFile(assetURL, newFile);
            }
        }
        return result;
    }

    public static void deleteAssetFile(Context context, String assetURL)
    {
        if (assetURL != null)
        {
            // 1. Extract the asset file name
            String fileName = createAssetFileName(assetURL);
            File storageLocation = SdkDeviceUtils.getInnerStoragePathIfPosible(context);
            File fileToDelete = new File(storageLocation, fileName);
            if (fileToDelete.exists() == true)
            {
                fileToDelete.delete();
            }
        }
    }

    public static void deleteAssetFile2(Context context, String assetURL)
    {
        if (assetURL != null)
        {
            // 1. Extract the asset file name
            String fileName = createAssetFileName2(assetURL);
            File storageLocation = SdkDeviceUtils.getInnerStoragePathIfPosible(context);
            File fileToDelete = new File(storageLocation, fileName);
            if (fileToDelete.exists() == true)
            {
                fileToDelete.delete();
            }
        }
    }

    public static File getAssetFile(Context context, String assetURL)
    {
        File result = null;
        try
        {
            if (assetURL != null)
            {
                // 1. Extract the asset file name
                String fileName = createAssetFileName(assetURL);
                File storageLocation = SdkDeviceUtils.getInnerStoragePathIfPosible(context);
                if (!assetURL.equals(""))
                {
                    result = new File(storageLocation, fileName);
                    if (result.exists() == false)
                    {
                        result = null;
                    }
                }
            }
        } catch (Exception ex)
        {

        }
        return result;
    }

    public static File getAssetFile2(Context context, String assetURL)
    {
        File result = null;
        try
        {
            if (assetURL != null)
            {
                // 1. Extract the asset file name
                String fileName = createAssetFileName2(assetURL);
                File storageLocation = SdkDeviceUtils.getInnerStoragePathIfPosible(context);
                if (!assetURL.equals(""))
                {
                    result = new File(storageLocation, fileName);
                    if (result.exists() == false)
                    {
                        result = null;
                    }
                }
            }
        } catch (Exception ex)
        {

        }
        return result;
    }

    /**
     * Prepare and download assets
     */
    public static boolean loadAsset(Context context, JSONObject oldProperties, JSONObject newProperties, String assetKey) throws JSONException
    {
        boolean result = false;
        //String newAssetUrl = newProperties.optString(assetKey);
        ArrayList<String> imgList = generateAssetFileImageList(newProperties, assetKey);
        if (imgList.isEmpty() == false)
        {
            for (int i = 0; i < imgList.size(); i++)
            {
                if (oldProperties != null)
                {
                    result = true;
                    String oldAssetURL = oldProperties.optString(assetKey);

                    File oldFile = AssetUtil.getAssetFile(context, oldAssetURL);
                    if (oldAssetURL == null
                            || oldAssetURL.equals(imgList.get(i)) == false
                            || oldFile == null
                            || oldFile.exists() == false
                            || oldFile.length() == 0)
                    {
                        if (oldAssetURL != null && !oldAssetURL.equals(""))
                        {
                            if (oldFile != null && oldFile.exists() == true)
                            {
                                AssetUtil.deleteAssetFile(context, oldAssetURL);
                            }
                        }

                        result = AssetUtil.createAssetFile(context, imgList.get(i));
                    }
                } else
                {
                    result = AssetUtil.createAssetFile(context, imgList.get(i));
                }

                if (result == true)
                {
                    newProperties.put(assetKey, imgList.get(i));
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Generate valid asset files list
     */
    public static ArrayList<String> generateAssetFileImageList(JSONObject jsonObject, String key)
    {
        ArrayList<String> list = new ArrayList<>();
        String link = jsonObject.optString(key);
        if (jsonObject.has(key + "Arr"))
        { // Check if multiple files exists
            JSONArray jsonArray = jsonObject.optJSONArray(key + "Arr");
            if (jsonArray != null && jsonArray.length() > 0)
            {
                if (link != null)
                {
                    // Remove suffix
                    String stripedLink = link.substring(0, link.lastIndexOf(".") + 1);
                    for (int i = 0; i < jsonArray.length(); i++)
                    {
                        String suffix = jsonArray.optString(i, "");
                        // Check if has webp file and check compatibility
                        if (suffix.equals("webp") || suffix.equals("webP") || suffix.equals("WEBP"))
                        {
                            //If compatible download the webp anim
                            if (Utils.isWebViewSupportWebP())
                            {
                                list.add(stripedLink + suffix);
                            }
                        } else
                        {
                            //Otherwise download next file in array
                            list.add(stripedLink + suffix);
                        }
                    }
                }
            }
        }
        //No array exists or something went wrong , download the file from original link
        if (list.isEmpty())
        {
            list.add(link);
        }
        return list;
    }

    /**
     * Background Async Task to parse style and loaded resources if needed
     */
    public static class ParserAsyncTask extends AsyncTask<Void, Void, Boolean>
    {
        public enum StyleType
        {
            PANEL_STYLE, FEED_STYLE, FLEXI_STYLE, INTERSTITIAL_STYLE,GLOBAL_PARAMS_STYLE
        }

        private WeakReference<Context> mContextRef;
        private StyleType mType;
        private IOnStyleParseListener mStyleParseListener;

        public ParserAsyncTask(Context context, StyleType type, IOnStyleParseListener resultCallback)
        {
            mContextRef = new WeakReference<>(context);
            mType = type;
            mStyleParseListener = resultCallback;
        }

        @Override
        protected Boolean doInBackground(Void... params)
        {
            boolean res = false;

            if (mContextRef.get() != null)
            {
                String data = null;
                PropertiesObj appConfig = DatabaseManager.getInstance(mContextRef.get()).getConfigTable().loadAppProperties();
                if(appConfig != null) {
                    switch (mType)
                    {

                        case GLOBAL_PARAMS_STYLE:
                        {
                            data = appConfig.getGlobalParamsStyle();
                            break;
                        }
                        case INTERSTITIAL_STYLE:
                        {
                            data = appConfig.getInterstitialViewStyle();
                            break;
                        }
                        case PANEL_STYLE:
                        {
                            data = appConfig.getPanelViewStyle();
                            break;
                        }
                        case FLEXI_STYLE:
                        {
                            data = appConfig.getFlexiViewStyle();
                            break;
                        }
                        case FEED_STYLE:
                        {
                            data = appConfig.getFeedViewStyle();
                            break;
                        }
                    }
                }

                if (data != null)
                {
                    res = AssetUtil.parseAssets(mContextRef.get(), data);
                }
            }
            return res;
        }

        @Override
        protected void onPostExecute(Boolean result)
        {
            if (mStyleParseListener != null)
            {
                mStyleParseListener.onParseFinished(result);
            }
        }

        @Override
        protected void onCancelled()
        {
            super.onCancelled();
        }

        public interface IOnStyleParseListener
        {
            void onParseFinished(boolean success);
        }
    }

    /**
     * Parse style in background thread (asynchronously)
     */
    public static void parseStyleAsync(Context context, ParserAsyncTask.StyleType type, ParserAsyncTask.IOnStyleParseListener listener)
    {
        ParserAsyncTask task = new ParserAsyncTask(context, type, listener);
        if (Build.VERSION.SDK_INT < 11)
        {
            task.execute();
        } else
        {
            task.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
        }
    }
}
