package com.kidoz.sdk.api;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.os.Build;
import android.util.Log;
import android.widget.Toast;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.assets_handling.AssetUtil;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.SdkCookieManager;
import com.kidoz.sdk.api.general.utils.SdkDeviceUtils;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.general.utils.StorageLife;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.receivers.SdkReceiver;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.kidoz.sdk.api.server_connect.StreamToStringConverter;
import com.kidoz.sdk.api.ui_views.html_view.HtmlManager;
import com.kidoz.sdk.api.ui_views.html_view.HtmlViewWrapper;
import com.kidoz.sdk.api.ui_views.interstitial.IntrstWrapper;
import com.kidoz.sdk.api.ui_views.video_unit.VideoUnitActivity;

import org.greenrobot.eventbus.EventBus;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.util.List;

/**
 * KIDOZ SDK initiator class
 */
public final class KidozSDK
{
    public static final String TAG = KidozSDK.class.getSimpleName();

    private static boolean isInitialized = false;
    private static String mPublisherId = null;
    private static String mSecurityKey = null;

    private static boolean isDeveloperLoggingON = false;

    static
    {
        try
        {
            InputStream is = KidozSDK.class.getClassLoader().getResourceAsStream("build_properties.txt");
            if (is != null)
            {
                String value = StreamToStringConverter.readStream(new BufferedInputStream(is), null, false);
                ConstantDef.SDK_EXTENSION_TYPE = Integer.parseInt(value);
            }
        } catch (Exception e)
        {
            ConstantDef.SDK_EXTENSION_TYPE = ConstantDef.EXTENSION_TYPE_ANDROID;
        }
    }

    private static void validateParameters(Context context, String publisher_id, String token)
    {
        SDKLogger.printDebbugLog(TAG, "validateParameters");

        if (context == null)
        {
            // Validate context.
            throw new RuntimeException("Context can't be null!");
        } else if (token == null || token.length() > 48)
        {
            // Validate token.
            throw new RuntimeException("Invalid Security Token! Please recheck you security token..");
        } else if (publisher_id == null)
        {
            // Validate publisherID.
            throw new RuntimeException("Invalid publisher id! Please check you publisher id..");
        } else
        {
            try
            {
                Long.parseLong(publisher_id);
            } catch (NumberFormatException e)
            {
                throw new RuntimeException("Invalid publisher id! Please check you publisher id..");
            }
        }
    }

    private static void notifyUserUsingDemoPublisherID(final Context context, String publisher_id)
    {
        SDKLogger.printDebbugLog(TAG, "notifyUserUsingDemoPublisherID");

        if (publisher_id.equals("5") || publisher_id.equals("7") || publisher_id.equals("8"))
        {
            if (context instanceof Activity)
            {
                ((Activity) context).runOnUiThread(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        Toast.makeText(context, "Notice ! You are using Demo PUBLISHER ID ! " + context.getPackageName(), Toast.LENGTH_LONG).show();
                    }
                });
            } else
            {
                Log.e(SDKLogger.GENERAL_TAG, "Notice ! You are using Demo PUBLISHER ID ! , Package Name :" + context.getPackageName());
            }
        }
    }

    /**
     * Initialize KIDOZ Sdk
     *
     * @param context      activity context
     * @param publisher_id valid publisher id
     * @param token        valid security token
     * @throws RuntimeException in case of invalid or missing <b>publisher_id</b> , <b>Security Token </b> or missing <b>Manifest.xml</b> definitions
     */
    public static void initialize(final Context context, String publisher_id, String token)
    {
        SDKLogger.printDebbugLog(TAG, "initialize : " + isInitialized);
        long startTime = System.currentTimeMillis();

        // 1. Validate parameters.
        validateParameters(context, publisher_id, token);
        SDKLogger.printDebbugLog(TAG, "CT validateParameters time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");

        // 2. Notify user if using demo publisherID.
        notifyUserUsingDemoPublisherID(context, publisher_id);
        SDKLogger.printDebbugLog(TAG, "CT notifyUserUsingDemoPublisherID time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");

        // 3. Check manifest declaration.
        checkManifestDeclarations(context);
        SDKLogger.printDebbugLog(TAG, "CT checkManifestDeclarations time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");

        // 4. Load Youtube player HTML and webView.
        //        VideoManager.getInstance(context);
        SDKLogger.printDebbugLog(TAG, "CT VideoManager.getInstance time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");

        // 5. Init other relevant parameters.
        try
        {
            // Init google ID.
            SdkDeviceUtils.getGoogleAdvertisingID(context);
            SDKLogger.printDebbugLog(TAG, "CT getGoogleAdvertisingID time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");

            // Init web related stuff.
            Utils.preInitiate(context);
            SDKLogger.printDebbugLog(TAG, "CT preInitiate time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");

            // Init Shared preferences stuff.
            SharedPreferences eventsPref = context.getSharedPreferences(EventManager.EVENTS_SHARED_PREFERENCES_KEY, 0);
            long sessionID = eventsPref.getLong(EventManager.EVENTS_LAST_SESSION_ID_KEY, 0);
            sessionID += 1;
            SharedPreferences.Editor editor = eventsPref.edit();
            editor.putLong(EventManager.EVENTS_LAST_SESSION_ID_KEY, sessionID);
            editor.apply();

            SDKLogger.printDebbugLog(TAG, "CT SharedPreferences time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");

            // Init event stuff.
            JSONObject eventParams = new JSONObject();
            eventParams.put(EventParameters.DEVELOPER_ID, publisher_id);
            eventParams.put(EventManager.LOG_LEVEL_KEY, EventManager.LOG_NORMAL_LEVEL);
            eventParams.put(EventParameters.SESSION_ID, sessionID);
            EventManager.getInstance(context).init(context, eventParams);
            EventManager.getInstance(context).logEvent(context, null, null, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SESSION, EventParameters.ACTION_SESSION_START, publisher_id);
            if (isInitialized)
            {
                EventManager.getInstance(context).logEvent(context, null, null, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_SDK_ALREADY_INIT, publisher_id);
            }
            EventManager.getInstance(context).startEventsSync(context);

            SDKLogger.printDebbugLog(TAG, "CT EventManager time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");

            // Reset The storage of the impressions for the HTML content
            SdkCookieManager.resetStorage(context, null, StorageLife.SESSION);

            // Init static parameters.
            SharedPreferencesUtils.saveSharedPreferencesData(context, "PUBLISHER_ID", publisher_id);
            mPublisherId = publisher_id;
            mSecurityKey = token;

            // Init api.
            SdkAPIManager.init(context, publisher_id, token, isDeveloperLoggingON);

            // Validate SDK.
            validateSDK(context, publisher_id);

            SDKLogger.printDebbugLog(TAG, "CT initialize time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to init SDK: " + ex.getMessage());
        }
    }

    /**
     * Validate SDK configuration (Configuration retrieved from remote server)
     */
    private static void validateSDK(final Context context, final String publisher_id)
    {
        SDKLogger.printDebbugLog(TAG, "validateSDK");
        final long startTime = System.currentTimeMillis();

        try
        {
            if (context != null)
            {
                // Update Current Sdk style version for Event Log
                PropertiesObj appConfig = DatabaseManager.getInstance(context).getConfigTable().loadAppProperties();
                if (appConfig != null)
                {
                    EventManager.SDK_STYLE_VERSION_NUM = appConfig.getSdkStyleVersion();
                }

                // Validate the SDK again the server.
                getApiManager(context).initSdk(context, new ApiResultCallback<PropertiesObj>()
                {
                    @Override
                    public void onServerResult(ResultData<?> result)
                    {
                        if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true && result.getData() != null && result.getData() instanceof PropertiesObj)
                        {
                            PropertiesObj serverConfig = (PropertiesObj) result.getData();
                            // Update Current Sdk style version for Event Log
                            EventManager.SDK_STYLE_VERSION_NUM = serverConfig.getSdkStyleVersion();

                            // Update main (feed) api domain name
                            SdkAPIManager.updateDomain(serverConfig.getFeedApiDomain());

                            PropertiesObj savedConfig = DatabaseManager.getInstance(context).getConfigTable().loadAppProperties();

                            boolean clearHtmlCache = true;
                            if (savedConfig != null && savedConfig.getHtmlLoaderDefaultLink() != null && savedConfig.getHtmlLoaderDefaultLink().equals(serverConfig.getHtmlLoaderDefaultLink()))
                            {
                                clearHtmlCache = false;
                            }

                            // Load Default Html Player url and webView
                            initHtmlWrapper(context, serverConfig, clearHtmlCache);
                            // Save and update properties to inner db
                            DatabaseManager.getInstance(context).getConfigTable().insertAppProperties(serverConfig);

                            // Parse video unit style
                            VideoUnitActivity.parseVideoUnitStyle(context, serverConfig.getProperties());

                            // Parse global style params
                            AssetUtil.parseStyleAsync(context, AssetUtil.ParserAsyncTask.StyleType.GLOBAL_PARAMS_STYLE,null);


                            // Notify validate finished.
                            isInitialized = true;
                            EventManager.getInstance(context).logEvent(context, null, null, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_SDK_INIT, publisher_id);
                            EventMessage eventMessage = new EventMessage(EventMessage.MessageType.INIT_SDK);
                            EventBus.getDefault().post(eventMessage);

                            SDKLogger.printDebbugLog(TAG, "CT validateSDK time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");

                            if (isDeveloperLoggingON)
                            {
                                Log.d(SDKLogger.GENERAL_TAG, "Kidoz SDK has been successfully Initialized !");
                            }
                        }
                    }

                    @Override
                    public void onFailed()
                    {

                    }
                });
            }
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to validateSDK: " + ex.getMessage());
        }
    }

    /**
     * Initiate HTML view wrapper
     */
    private static void initHtmlWrapper(final Context context, PropertiesObj serverConfig, boolean clearHtmlCache)
    {
        SDKLogger.printDebbugLog(TAG, "initHtmlWrapper");
        final long startTime = System.currentTimeMillis();

        HtmlViewWrapper viewWrapper = HtmlManager.initDefaultHtmlView(context, clearHtmlCache);
        viewWrapper.setSdkInitListener(new HtmlViewWrapper.IOnInitFinishedListener()
        {
            @Override
            public void onInitFinished()
            {
                IntrstWrapper.initInstance(context);
                SDKLogger.printDebbugLog(TAG, "CT initHtmlWrapper time = " + (System.currentTimeMillis() - startTime) / 1000.0 + " seconds");
            }
        });


        viewWrapper.loadHtml(serverConfig.getHtmlLoaderDefaultLink());
        //viewWrapper.loadHtml("https://s3.amazonaws.com/kidoz.test.bucket/ronybrosh/rovio/rovio_loader_2.html");
    }

    private static SdkAPIManager getApiManager(Context context)
    {
        SdkAPIManager.init(context, mPublisherId, mSecurityKey, isDeveloperLoggingON);
        return SdkAPIManager.getSdkApiInstance(context);
    }

    /**
     * Check if Kidoz SDK has been initialised
     *
     * @return initialized
     */
    public static boolean isInitialised()
    {
        return isInitialized;
    }

    // Check necessary manifest declarations that mast be present
    private static void checkManifestDeclarations(Context context)
    {
        SDKLogger.printDebbugLog(TAG, "checkManifestDeclarations");

        if (ConstantDef.SDK_EXTENSION_TYPE == ConstantDef.EXTENSION_TYPE_ANDROID || ConstantDef.SDK_EXTENSION_TYPE == ConstantDef.EXTENSION_TYPE_ECLIPSE || ConstantDef.SDK_EXTENSION_TYPE == ConstantDef.EXTENSION_TYPE_COCOS_DX)
        {
            PackageManager packageManager = context.getPackageManager();

            // Check Sdk receiver declaration exists in the manifest
            Intent intent = new Intent(context, SdkReceiver.class);

            int flag = PackageManager.GET_META_DATA;
            if (Build.VERSION.SDK_INT >= 23)
            {
                flag = PackageManager.MATCH_ALL;
            }

            List resolveInfo = packageManager.queryBroadcastReceivers(intent, flag);
            if (resolveInfo == null || resolveInfo.size() == 0)
            {
                throw new RuntimeException("Missing Android Manifest declaration ! \n" +
                        "Please add the following lines to your Manifest.xml file..\n...\n" +
                        "<receiver android:name=\"com.kidoz.sdk.api.receivers.SdkReceiver\">\n" +
                        "    <intent-filter>\n" +
                        "         <action android:name=\"android.intent.action.PACKAGE_ADDED\" />\n" +
                        "         <data android:scheme=\"package\" />\n" +
                        "    </intent-filter>\n" +
                        "</receiver> \n...\n");
            }
        }
    }

    /**
     * Set developer logging enabled
     * (Must be set before initialization of the SDK)
     *
     * @param enabled
     */
    public static void setLoggingEnabled(boolean enabled)
    {
        isDeveloperLoggingON = enabled;
    }

    public static String getPublisherID()
    {
        return mPublisherId;
    }

    public static String getAuthToken()
    {
        return mSecurityKey;
    }

    public static String getSDKVersion()
    {
        return ConstantDef.ACTUAL_SDK_VERSION;
    }
}
