package com.kidoz.sdk.api;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.view.View;

import com.kidoz.sdk.api.general.ContentLogicLoader;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.assets_handling.AssetUtil;
import com.kidoz.sdk.api.general.assets_handling.StyleParser;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.SdkCookieManager;
import com.kidoz.sdk.api.general.utils.StorageLife;
import com.kidoz.sdk.api.interfaces.IOnFeedViewEventListener;
import com.kidoz.sdk.api.interfaces.KidozPlayerEventHelper;
import com.kidoz.sdk.api.interfaces.KidozPlayerListener;
import com.kidoz.sdk.api.structure.ContentData;
import com.kidoz.sdk.api.ui_views.feed_view.FeedFamilyViewDialog;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.json.JSONObject;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * KIDOZ FeedView widget class
 */
public class FeedView
{
    // Properties keys
    public static final String KIDOZ_FEED_BUTTON = "feedButton";

    public static final String TAG = FeedView.class.getSimpleName();

    private Builder mBuilder;
    private Lock raceConditionLock = new ReentrantLock();

    private FeedFamilyViewDialog mFamilyViewDialog;
    protected ContentLogicLoader mContentLogicLoader;

    protected IOnFeedViewEventListener mOnFeedViewEventListener;
    protected boolean mIsAutomaticInvocation = false;
    protected boolean mIsShowing = false;

    protected boolean mShowWhenInitiated = false;
    protected JSONObject mJsonObjectProperties;

    private KidozPlayerEventHelper mKidozPlayerEventHelper;
    private Dialog.OnDismissListener dismissListener;
    private KidozPlayerListener mKidozPlayerListener;

    private boolean mIsViewReadyCalled = false; // This flag used to know if the call of onViewReady() has been made before set of FeedView listener (Asynchronous Bug in FeedButton case)
    // In that case we will launch onViewReady() as soon as listener is set
    private boolean mIsReadyToShowCalled = false; // The same issue with the above flag applies to onReadyToShow() call

    /**
     * Set kidoz players event listeners
     * Used for listening to invoked Player state events (Applied for single view)
     *
     * @param kidozPlayerListener event listener
     */
    public void setKidozPlayerListener(KidozPlayerListener kidozPlayerListener)
    {
        mKidozPlayerListener = kidozPlayerListener;
    }

    protected FeedView(Builder builder)
    {
        mBuilder = builder;

        initContentLogicLoader();

        if (EventBus.getDefault().isRegistered(this) == false)
        {
            EventBus.getDefault().register(this);
        }

        // Has been initialised
        if (KidozSDK.isInitialised() == true)
        {
            validateLayoutAndAssets();
        }

        mKidozPlayerEventHelper = new KidozPlayerEventHelper(new KidozPlayerEventHelper.IPlayerListener()
        {
            @Override
            public void onOpen(boolean isMaximizedPlayer)
            {
                if (mKidozPlayerListener != null)
                {
                    mKidozPlayerListener.onPlayerOpen();
                }

                if (mFamilyViewDialog != null && mFamilyViewDialog.isShowing())
                {
                    mFamilyViewDialog.pauseAction(isMaximizedPlayer);
                }
            }

            @Override
            public void onClose(boolean isMaximizedPlayer)
            {
                if (mKidozPlayerListener != null)
                {
                    mKidozPlayerListener.onPlayerClose();
                }

                if (mFamilyViewDialog != null && mFamilyViewDialog.isShowing())
                {
                    mFamilyViewDialog.resumeAction(isMaximizedPlayer);
                }
            }
        }, WidgetType.WIDGET_TYPE_FEED);

        dismissListener = new DialogInterface.OnDismissListener()
        {
            @Override
            public void onDismiss(DialogInterface dialog)
            {
                if (mKidozPlayerEventHelper != null)
                {
                    mKidozPlayerEventHelper.unRegister();
                }

                if (mOnFeedViewEventListener != null)
                {
                    mOnFeedViewEventListener.onDismissView();
                }

                mIsShowing = false;
            }
        };
    }

    // Event bus callback
    @Subscribe
    public void onHandleEvent(EventMessage event)
    {
        if (event.getMessageType() == EventMessage.MessageType.INIT_SDK)
        {
            validateLayoutAndAssets();
        }
    }

    private void unregisterEventBus()
    {
        if (EventBus.getDefault().isRegistered(this))
        {
            EventBus.getDefault().unregister(this);
        }
    }

    private void validateLayoutAndAssets()
    {
        unregisterEventBus();

        if (raceConditionLock.tryLock())
        {
            try
            {
                AssetUtil.parseStyleAsync(mBuilder.mContext, AssetUtil.ParserAsyncTask.StyleType.FEED_STYLE, new AssetUtil.ParserAsyncTask.IOnStyleParseListener()
                {
                    @Override
                    public void onParseFinished(boolean success)
                    {
                        if (success)
                        {
                            mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
                            if (mJsonObjectProperties != null)
                            {
                                EventBus.getDefault().post(new EventMessage(EventMessage.MessageType.INIT_FEED_BUTTON));

                                if (mOnFeedViewEventListener != null)
                                {
                                    mOnFeedViewEventListener.onViewReady();
                                } else
                                {
                                    mIsViewReadyCalled = true;
                                }

                                if (mShowWhenInitiated)
                                {
                                    showView();
                                }
                            }
                        }
                    }
                });
            } finally
            {
                raceConditionLock.unlock();
            }
        }
    }

    /**
     * Initiate  content loader logic ,that handles all the logic for loading content
     **/
    private void initContentLogicLoader()
    {
        mContentLogicLoader = new ContentLogicLoader(new ContentLogicLoader.IOnContentDataReadyCallback()
        {
            @Override
            public void onDataReady(ContentData contentData)
            {
                mFamilyViewDialog.setContentData(contentData);

                // Reset The storage of the impressions for the HTML content
                SdkCookieManager.resetStorage(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getStringValue(), StorageLife.WIDGET);
            }

            @Override
            public void onLoadContentFailed()
            {
                // TODO
            }
        });
    }

    /**
     * Show feed interstitial view (Will be shown <b>Before</b> content data is ready)
     **/
    public void showView()
    {
        showViewInner();
    }

    private void showViewInner()
    {
        if (KidozSDK.isInitialised())
        {
            mIsAutomaticInvocation = true;

            if (mJsonObjectProperties == null)
            {
                mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
            }

            if (mJsonObjectProperties != null)
            {
                openSimpleDialog();
            } else
            {
                mShowWhenInitiated = true;
            }
        } else
        {
            mShowWhenInitiated = true;
        }
    }

    protected synchronized void openSimpleDialog()
    {
        if (mKidozPlayerEventHelper != null)
        {
            mKidozPlayerEventHelper.register();
        }
        if (mJsonObjectProperties != null && mBuilder.mContext != null && mBuilder.mContext instanceof Activity && ((Activity) mBuilder.mContext).isFinishing() == false)
        {
            if (mFamilyViewDialog == null || (mFamilyViewDialog.isShowing() == false && mFamilyViewDialog.isWaitingToOpen() == false))
            {
                mFamilyViewDialog = new FeedFamilyViewDialog(mBuilder.mContext, mOnFeedViewEventListener, mIsAutomaticInvocation, mJsonObjectProperties);
                mFamilyViewDialog.setOnDismissListener(dismissListener);
                mFamilyViewDialog.setOnCloseButtonClickListener(new View.OnClickListener()
                {
                    @Override
                    public void onClick(View view)
                    {
                        dismissView();
                        unregisterEventBus();
                    }
                });

                if (mOnFeedViewEventListener != null)
                {
                    mOnFeedViewEventListener.onReadyToShow();
                } else
                {
                    mIsReadyToShowCalled = true;
                }

                mFamilyViewDialog.openDialog();

                if (mJsonObjectProperties != null)
                {
                    mContentLogicLoader.loadContent(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getStringValue(), mJsonObjectProperties.optString(StyleParser.STYLE_ID));
                }
            }

            mShowWhenInitiated = false;
        }
    }

    /**
     * Dismiss feed view
     **/
    public void dismissView()
    {
        if (mFamilyViewDialog != null)
        {
            mFamilyViewDialog.closeDialog();
        }
    }

    /**
     * Set on view event listener
     *
     * @param listener
     **/
    public void setOnFeedViewEventListener(IOnFeedViewEventListener listener)
    {
        mOnFeedViewEventListener = listener;

        if (mFamilyViewDialog != null)
        {
            mFamilyViewDialog.setOnInterstitialViewEventListener(mOnFeedViewEventListener);
        }

        if (mIsViewReadyCalled)
        {
            mIsViewReadyCalled = false;
            mOnFeedViewEventListener.onViewReady();
        }

        if (mIsReadyToShowCalled)
        {
            mIsReadyToShowCalled = false;
            mOnFeedViewEventListener.onReadyToShow();
        }
    }

    /**
     * Builder class that handles view initiation
     **/
    public static class Builder
    {
        private Context mContext;

        public Builder(Context context)
        {
            mContext = context;
        }

        public FeedView build()
        {
            return new FeedView(this);
        }
    }
}
