package com.kidoz.sdk.api.ui_views.interstitial;

import android.content.Context;
import android.content.Intent;
import android.util.Log;

import com.kidoz.sdk.api.KidozInterstitial;
import com.kidoz.sdk.api.KidozSDK;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.WidgetEventMessage;
import com.kidoz.sdk.api.general.assets_handling.AssetUtil;
import com.kidoz.sdk.api.general.assets_handling.StyleParser;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SdkCookieManager;
import com.kidoz.sdk.api.general.utils.StorageLife;
import com.kidoz.sdk.api.ui_views.html_view.HtmlManager;
import com.kidoz.sdk.api.ui_views.html_view.HtmlViewWrapper;
import com.kidoz.sdk.api.ui_views.html_view.IOnHtmlWebViewInterface;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.json.JSONObject;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

public class IntrstWrapper
{
    private static final int REQUEST_FILTER_TIMEOUT = 1000 * 5; //5 seconds

    public enum InterstitialEvents
    {
        OPENED, CLOSED, READY, LOAD_FAILED, REWARDED, REWARDED_VIDEO_STARTED
    }

    private static IntrstWrapper mInstance = null;

    public Context mContext;
    private HtmlViewWrapper webView;

    private String mStyleId;

    private boolean mIsInitializationFinished = false;
    private boolean mIsLoaded = false;

    private boolean mIsShowRequested = false; // Used for avoiding multiple calls of show for short amount of time
    private boolean mIsLoadRequested = false; // Used for avoiding multiple calls of load for short amount of time
    private long mLoadRequestTM = 0;

    private boolean mIsLoadRequestedWhileNotReady;

    private JSONObject mProperties;
    private Lock mLaunchLock;
    private InnerHelperInterface mInnerHelperInterface;
    private KidozInterstitial.AD_TYPE mAdRequestType = KidozInterstitial.AD_TYPE.INTERSTITIAL;
    private Lock raceConditionLock = new ReentrantLock();

    private JSONObject mVastProperties;

    //Flag used to differ the source of interstitial request developer or local
    private boolean mIsLocal = false;

    public static IntrstWrapper initInstance(Context context)
    {
        if (mInstance == null)
        {
            return mInstance = new IntrstWrapper(context);
        } else
        {
            mInstance.validateIfPrepared();
            return mInstance;
        }
    }

    protected IntrstWrapper(Context context)
    {
        mContext = context;

        if (EventBus.getDefault().isRegistered(this) == false)
        {
            EventBus.getDefault().register(this);
        }

        mLaunchLock = new ReentrantLock();

        // Has been initialised
        if (KidozSDK.isInitialised() == true)
        {
            validateLayoutAndAssets();
        }

        mInnerHelperInterface = new InnerHelperInterface()
        {
            @Override
            public void onAdOpen()
            {
                if (!mIsLocal)
                {
                    EventBus.getDefault().post(InterstitialEvents.OPENED);
                }
            }

            @Override
            public void onAdClose()
            {
                if (!mIsLocal)
                {
                    EventBus.getDefault().post(InterstitialEvents.CLOSED);
                }
                mIsShowRequested = false;

                mIsLoaded = false;
                SdkCookieManager.resetStorage(mContext, WidgetType.WIDGET_TYPE_INTERSTITIAL.getStringValue(), StorageLife.WIDGET);
            }
        };
    }

    public InnerHelperInterface getHelperInterface()
    {
        return mInnerHelperInterface;
    }

    /**
     * Validate resources
     */
    private void validateLayoutAndAssets()
    {
        if (raceConditionLock.tryLock())
        {
            try
            {
                AssetUtil.parseStyleAsync(mContext, AssetUtil.ParserAsyncTask.StyleType.INTERSTITIAL_STYLE, new AssetUtil.ParserAsyncTask.IOnStyleParseListener()
                {
                    @Override
                    public void onParseFinished(boolean success)
                    {
                        if (success)
                        {
                            mProperties = DatabaseManager.getInstance(mContext).getConfigTable().loadProperties(KidozInterstitial.TAG);
                            if (mProperties != null)
                            {
                                mStyleId = mProperties.optString(StyleParser.STYLE_ID);
                            }
                            prepareWebView();
                        } else
                        {
                            EventBus.getDefault().post(InterstitialEvents.LOAD_FAILED);
                        }
                    }
                });
            } finally
            {
                raceConditionLock.unlock();
            }
        }
    }

    /**
     * Initiate and  Prepare webview for work
     */
    private synchronized void prepareWebView()
    {
        if (webView == null)
        {
            webView = HtmlManager.initIntrstDefaultHtmlView(mContext);
            webView.setHtmlWebViewListener(new IOnHtmlWebViewInterface()
            {
                @Override
                public void onHtmlFinishedLoading()
                {
                    super.onHtmlFinishedLoading();
                }

                @Override
                public void onNotifyVastReady(boolean isVastReady, JSONObject vast_properties)
                {
                    super.onNotifyVastReady(isVastReady, vast_properties);

                    mVastProperties = vast_properties;
                    mIsLoadRequested = false;
                    if (isVastReady)
                    {
                        mIsLoaded = true;

                        if (!mIsLocal)
                        {
                            EventBus.getDefault().post(InterstitialEvents.READY);
                        } else
                        {
                            show();
                        }
                    } else
                    {
                        mIsLoaded = false;
                        if (!mIsLocal)
                        {
                            EventBus.getDefault().post(InterstitialEvents.LOAD_FAILED);
                        }
                    }
                    webView.setWidgetType(WidgetType.WIDGET_TYPE_INTERSTITIAL.getStringValue());
                }

                @Override
                public void onViewReady()
                {
                    super.onViewReady();
                    mIsInitializationFinished = true;

                    if (mIsLoadRequestedWhileNotReady)
                    {
                        mIsLoadRequestedWhileNotReady = false;
                        load(mAdRequestType);
                    }
                }

                @Override
                public void onClose()
                {
                    super.onClose();
                    WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.INTERSTITIAL_AD_CLOSE, WidgetType.WIDGET_TYPE_INTERSTITIAL);
                    EventBus.getDefault().post(eventMessage);
                }

                @Override
                public void onRewarded()
                {
                    super.onRewarded();
                    EventBus.getDefault().post(InterstitialEvents.REWARDED);
                }

                @Override
                public void onRewardedVideoStarted()
                {
                    super.onRewardedVideoStarted();
                    EventBus.getDefault().post(InterstitialEvents.REWARDED_VIDEO_STARTED);
                }
            });
        }

        webView.setWidgetType(WidgetType.WIDGET_TYPE_INTERSTITIAL.getStringValue());

        PropertiesObj appConfig = DatabaseManager.getInstance(mContext).getConfigTable().loadAppProperties();
        if (appConfig != null)
        {
            webView.loadHtml(appConfig.getInterstitialHtmlLink());
        }
    }

    /**
     * Load interstitial content
     **/
    public synchronized void load(KidozInterstitial.AD_TYPE ad_type)
    {
        mIsShowRequested = false;
        mAdRequestType = ad_type;
        if (mIsInitializationFinished == true)
        {
            // Reset loading request flag in case something got wrong an we never accepted any answer from the HTML
            if (System.currentTimeMillis() - mLoadRequestTM > REQUEST_FILTER_TIMEOUT)
            {
                mIsLoadRequested = false;
            }

            if (mIsLoadRequested == false)
            {
                mIsLoaded = false;
                mIsLoadRequested = true;
                mLoadRequestTM = System.currentTimeMillis();
                if (webView != null)
                {
                    webView.requestVastAds(ad_type.getValue());
                }
            }
        } else
        {
            mIsLoadRequestedWhileNotReady = true;
        }
    }

    public synchronized void show()
    {
        if (mIsLoaded)
        {
            if (mIsShowRequested == false)
            {
                mIsShowRequested = true;
                webView.startVastAd();
                openActivity();
            }
        } else
        {
            Log.w("KidozInterstitial", "Interstitial Ad is not loaded! Please call loadAd() first.");
        }
    }

    public boolean isInterstitialLoaded()
    {
        return mIsLoaded;
    }

    /**
     * Prepare correct interstitial dialog depending on the type of the HTML used
     * I both cases of Default and non default HTMl we need to wait until loading is finished before showing
     * the interstitial view for correct flow
     */
    private synchronized void openActivity()
    {
        if (mLaunchLock.tryLock())
        {
            try
            {
                try
                {
                    Intent intent = new Intent(mContext, KidozAdActivity.class);
                    intent.putExtra(KidozAdActivity.STYLE_ID_KEY, mStyleId);

                    if (mVastProperties != null)
                    {
                        intent.putExtra(KidozAdActivity.DISABLE_BACK_KEY, mVastProperties.optBoolean("interstitial_lock", false));
                    }

                    if (mAdRequestType == KidozInterstitial.AD_TYPE.REWARDED_VIDEO)
                    {
                        intent.putExtra(KidozAdActivity.DISABLE_BACK_KEY, true);
                    }

                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_REORDER_TO_FRONT | Intent.FLAG_ACTIVITY_SINGLE_TOP);
                    mContext.startActivity(intent);
                } catch (Exception e)
                {
                }
            } finally
            {
                mLaunchLock.unlock();
            }
        }
    }

    // Event bus callback
    @Subscribe
    public void onHandleEvent(EventMessage event)
    {
        if (event.getMessageType() == EventMessage.MessageType.INIT_SDK)
        {
            validateLayoutAndAssets();
        }
    }

    protected void validateIfPrepared()
    {
        if (EventBus.getDefault().isRegistered(this) == false)
        {
            EventBus.getDefault().register(this);
        }

        if (mIsInitializationFinished == false)
        {
            prepareWebView();
        }
    }

    public interface InnerHelperInterface
    {
        void onAdOpen();

        void onAdClose();
    }

    public void setIsLocalRequest(boolean local)
    {
        mIsLocal = local;
    }

    public boolean getIsLocal()
    {
        return mIsLocal;
    }
}
