package com.kidoz.sdk.api.ui_views.video_unit;

import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.res.Configuration;
import android.os.Build;
import android.os.Bundle;
import android.support.v4.content.LocalBroadcastManager;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;

import com.kidoz.sdk.api.fullscreen_video_layer.FullScreenVideoEnabledWebView;
import com.kidoz.sdk.api.general.ContentExecutionHandler;
import com.kidoz.sdk.api.general.enums.ContentType;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.structure.ContentItem;

/**
 * Created by RonyBrosh on 8/24/2016.
 */
public class VideoUnitActivity extends Activity
{
    private final String TAG = VideoUnitActivity.class.getSimpleName();

    public static VideoUnitActivityListener sVideoUnitActivityListener;

    private FullScreenVideoEnabledWebView mWebView;
    private BroadcastReceiver mBroadcastReceiver;

    @Override
    public void onConfigurationChanged(Configuration newConfig)
    {
        super.onConfigurationChanged(newConfig);
    }

    @Override
    protected void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);
        initActivity();
        setContentView(mWebView, new ViewGroup.MarginLayoutParams(ViewGroup.MarginLayoutParams.MATCH_PARENT, ViewGroup.MarginLayoutParams.MATCH_PARENT));
        invokeJavaScript("javascript:onWidgetOpen();");
        if (VideoUnit.sVideoUnitListener != null)
        {
            VideoUnit.sVideoUnitListener.onOpen();
        }
    }

    @Override
    protected void onDestroy()
    {
        super.onDestroy();
        invokeJavaScript("javascript:onWidgetClose();");
        if (VideoUnit.sVideoUnitListener != null)
        {
            VideoUnit.sVideoUnitListener.onClose();
        }
        sVideoUnitActivityListener = null;
    }

    @Override
    protected void onResume()
    {
        super.onResume();
        initFullScreenFlags();
        LocalBroadcastManager.getInstance(VideoUnitActivity.this).registerReceiver(mBroadcastReceiver, new IntentFilter("CloseVideoUnit"));
        LocalBroadcastManager.getInstance(VideoUnitActivity.this).registerReceiver(mBroadcastReceiver, new IntentFilter("WebPlayerDialogClosed"));

        mWebView.bringToFront();
        invokeJavaScript("javascript:onWidgetResume();");
    }

    @Override
    protected void onPause()
    {
        super.onPause();
        LocalBroadcastManager.getInstance(VideoUnitActivity.this).unregisterReceiver(mBroadcastReceiver);
        invokeJavaScript("javascript:onWidgetPause();");
    }

    private void initActivity()
    {
        initFullScreenFlags();
        initWebView();
        initBroadcastReceiver();
        initVideoUnitActivityListener();
    }

    private void initVideoUnitActivityListener()
    {
        sVideoUnitActivityListener = new VideoUnitActivityListener()
        {
            @Override
            public void forwardToGooglePlay(final String advertiserID, final String appID, final String googlePlayLink, final String widgetType, final String styleID, final String itemIndex)
            {
                runOnUiThread(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        int itemIndexAsInteger = 0;
                        try
                        {
                            itemIndexAsInteger = Integer.parseInt(itemIndex);
                        }
                        catch (Exception ex)
                        {
                            SDKLogger.printErrorLog(TAG, "Error when trying to parse item index: " + ex.getMessage());
                        }
                        ContentItem contentItem = new ContentItem();
                        contentItem.setId(appID);
                        contentItem.setName("");
                        contentItem.setData(googlePlayLink);
                        contentItem.setContentType(ContentType.PROMOTED_PLAY_APPLICATION);
                        contentItem.setAdvertiserID(advertiserID);
                        contentItem.setIsPromoted(true);

                        ContentExecutionHandler.handleContentItemClick(VideoUnitActivity.this, contentItem, widgetType, styleID, itemIndexAsInteger, false, null);
                    }
                });
            }
        };
    }


    private void initWebView()
    {
        mWebView = VideoUnit.mWebView;
        if (mWebView.getParent() != null)
        {
            ((ViewGroup) mWebView.getParent()).removeView(mWebView);
        }
        if (Build.VERSION.SDK_INT >= 19 && ConstantDef.DGM == true)
        {
            mWebView.setWebContentsDebuggingEnabled(true);
        }
    }

    private void initBroadcastReceiver()
    {
        mBroadcastReceiver = new BroadcastReceiver()
        {
            @Override
            public void onReceive(Context context, Intent intent)
            {
                if (intent != null)
                {
                    if (intent.getAction().equals("CloseVideoUnit") == true)
                    {
                        finish();
                    }
                    else if (intent.getAction().equals("WebPlayerDialogClosed") == true)
                    {
                        mWebView.bringToFront();
                        invokeJavaScript("javascript:onWidgetResume();");
                    }
                }
            }
        };
    }

    private void initFullScreenFlags()
    {
        if (Build.VERSION.SDK_INT < 16)
        {
            getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
        }
        else
        {
            if (Build.VERSION.SDK_INT < 18)
            {
                View decorView = getWindow().getDecorView();
                int uiOptions = View.SYSTEM_UI_FLAG_FULLSCREEN;
                decorView.setSystemUiVisibility(uiOptions);
            }
            else
            {
                final View decorView = getWindow().getDecorView();
                decorView.setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_STABLE | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_FULLSCREEN | View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY);
                decorView.setOnSystemUiVisibilityChangeListener(new View.OnSystemUiVisibilityChangeListener()
                {
                    @Override
                    public void onSystemUiVisibilityChange(int visibility)
                    {
                        if ((visibility & View.SYSTEM_UI_FLAG_FULLSCREEN) == 0)
                        {
                            decorView.setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_STABLE | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_FULLSCREEN | View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY);
                        }
                    }
                });
            }
        }
    }

    private void invokeJavaScript(final String javascript)
    {
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                if (Build.VERSION.SDK_INT >= 19)
                {
                    mWebView.evaluateJavascript(javascript, null);
                }
                else
                {
                    mWebView.loadUrl(javascript);
                }
            }
        });
    }

    public interface VideoUnitActivityListener
    {
        void forwardToGooglePlay(final String advertiserID, final String appID, final String googlePlayLink, final String widgetType, final String styleID, final String itemIndex);
    }
}
