package com.kidoz.sdk.api.ui_views.parental_lock;

import android.content.Context;
import android.graphics.Color;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.widget.FrameLayout;
import android.widget.ImageView;

import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.picasso_related.PicassoOk;
import com.squareup.imagelib.Callback;
import com.squareup.imagelib.RequestCreator;
import com.squareup.imagelib.Transformation;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import pl.droidsonroids.gif.GifDrawable;

public class AssetView extends FrameLayout
{
    private boolean mIsAssetLoaded;
    private ArrayList<Transformation> mTransformations = new ArrayList<Transformation>();

    private GifDrawable mAnimatedGifDrawable = null;
    private ImageView mImageView;
    private MyWebView mWebPwebView;

    public AssetView(Context context)
    {
        super(context);

        initView();
    }

    public AssetView(Context context, AttributeSet attrs)
    {
        super(context, attrs);

        initView();
    }

    public AssetView(Context context, AttributeSet attrs, int defStyleAttr)
    {
        super(context, attrs, defStyleAttr);

        initView();
    }

    private void initView()
    {
        setDrawingCacheBackgroundColor(0x00000000);
        setId(Utils.generateViewId());
        mImageView = new ImageView(getContext());
        mImageView.setScaleType(ImageView.ScaleType.FIT_XY);
        addView(mImageView, new FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT));
    }

    public void loadAsset(File file, IOnAssetLoadedListener loadedListener)
    {
        loadCorrectAsset(file, null, loadedListener);
    }

    public void loadAsset(File file, Drawable defaultImage, IOnAssetLoadedListener loadedListener)
    {
        loadCorrectAsset(file, defaultImage, loadedListener);
    }

    public void loadAsset(final String url, final IOnAssetLoadedListener loadedListener)
    {
        //setAssetBackgroundDrawable(new ColorDrawable(Color.WHITE));

        RequestCreator requestCreator = PicassoOk.getPicasso(getContext()).load(url);
        if (mImageView != null && mImageView.getWidth() != 0 && mImageView.getHeight() != 0)
        {
            requestCreator.resize(mImageView.getWidth(), mImageView.getHeight());
            requestCreator.onlyScaleDown();
            requestCreator.centerCrop();
        }

        requestCreator.noFade();
        requestCreator.transform(mTransformations);
        requestCreator.fetch(new Callback()
        {
            @Override
            public void onSuccess()
            {
                //setAssetBackgroundDrawable(null);
                RequestCreator requestCreator = PicassoOk.getPicasso(getContext()).load(url);
                if (mImageView != null && mImageView.getWidth() != 0 && mImageView.getHeight() != 0)
                {
                    requestCreator.resize(mImageView.getWidth(), mImageView.getHeight());
                    requestCreator.onlyScaleDown();
                    requestCreator.centerCrop();
                }

                requestCreator.noFade();
                requestCreator.transform(mTransformations);
                requestCreator.into(mImageView);

                mIsAssetLoaded = true;
                if (loadedListener != null)
                {
                    loadedListener.onAssetLoaded(mIsAssetLoaded);
                }
            }

            @Override
            public void onError()
            {
                mIsAssetLoaded = false;
                if (loadedListener != null)
                {
                    loadedListener.onAssetLoaded(mIsAssetLoaded);
                }
            }
        });
    }

    public void prefetchAsset(final String url, final IOnAssetLoadedListener loadedListener)
    {
        //setAssetBackgroundDrawable(new ColorDrawable(Color.WHITE));
        RequestCreator requestCreator = PicassoOk.getPicasso(getContext()).load(url);
        if (mImageView.getWidth() != 0 && mImageView.getHeight() != 0)
        {
            requestCreator.resize(mImageView.getWidth(), mImageView.getHeight());
            requestCreator.onlyScaleDown();
            requestCreator.centerCrop();
        }

        requestCreator.noFade();
        requestCreator.transform(mTransformations);
        requestCreator.fetch(new Callback()
        {
            @Override
            public void onSuccess()
            {

                if (loadedListener != null)
                {
                    loadedListener.onAssetLoaded(true);
                }
            }

            @Override
            public void onError()
            {
                if (loadedListener != null)
                {
                    loadedListener.onAssetLoaded(false);
                }
            }
        });
    }

    public void loadCorrectAsset(File file, Drawable defaultImage, final IOnAssetLoadedListener loadedListener)
    {
        mIsAssetLoaded = false;
        if (defaultImage != null)
        {
            setAssetBackgroundDrawable(defaultImage);
        }

        if (file != null)
        {
            if (file.exists())
            {
                final String path = file.getPath();
                // Png or Jpeg image file
                if (path.contains(".png") || path.contains(".PNG") || path.contains(".jpg") || path.contains(".JPG"))
                {
                    RequestCreator requestCreator = PicassoOk.getPicasso(getContext()).load(file);
                    if (mImageView.getWidth() != 0 && mImageView.getHeight() != 0)
                    {
                        requestCreator.resize(mImageView.getWidth(), mImageView.getHeight());
                        requestCreator.onlyScaleDown();
                        requestCreator.centerCrop();
                    }

                    requestCreator.transform(mTransformations);
                    requestCreator.into(mImageView, new Callback()
                    {
                        @Override
                        public void onSuccess()
                        {
                            setAssetBackgroundDrawable(null);
                            mIsAssetLoaded = true;
                            if (loadedListener != null)
                            {
                                loadedListener.onAssetLoaded(mIsAssetLoaded);
                            }
                        }

                        @Override
                        public void onError()
                        {
                            mIsAssetLoaded = false;
                            if (loadedListener != null)
                            {
                                loadedListener.onAssetLoaded(mIsAssetLoaded);
                            }
                        }
                    });
                    // Gif image file
                } else if (path.contains(".gif") || path.contains(".GIF"))
                {
                    try
                    {
                        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
                        {
                            mAnimatedGifDrawable.stop();
                            mAnimatedGifDrawable.recycle();
                            mAnimatedGifDrawable = null;
                        }

                        mAnimatedGifDrawable = new GifDrawable(file.getAbsolutePath());
                        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
                        {
                            mImageView.setImageDrawable(mAnimatedGifDrawable);
                            setAssetBackgroundDrawable(null);
                            mAnimatedGifDrawable.start();
                        }
                        mIsAssetLoaded = true;

                        if (loadedListener != null)
                        {
                            loadedListener.onAssetLoaded(mIsAssetLoaded);
                        }
                    } catch (IOException e)
                    {
                        if (e != null)
                        {
                            SDKLogger.printErrorLog(e.getMessage() + "   path: " + path);
                        }

                        mIsAssetLoaded = false;
                        // If something went wrong delete the file
                        if (defaultImage != null)
                        {
                            setAssetBackgroundDrawable(defaultImage);
                        }
                        if (loadedListener != null)
                        {
                            loadedListener.onAssetLoaded(mIsAssetLoaded);
                        }
                    }
                    // WebP image file
                } else if (path.contains(".webp") || path.contains(".WEBP") || path.contains(".webP"))
                {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1)
                    {
                        if (Utils.isWebViewSupportWebP())
                        {
                            mWebPwebView = generateWebPAnimatableWebView();

                            addView(mWebPwebView, 0, new FrameLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
                            loadImageToWebPwebView("file://" + path);

                            mIsAssetLoaded = true;
                        }

                        if (mIsAssetLoaded == false)
                        {
                            setAssetBackgroundDrawable(defaultImage);
                        }

                        if (loadedListener != null)
                        {
                            loadedListener.onAssetLoaded(mIsAssetLoaded);
                        }
                    }
                } else
                {
                    mIsAssetLoaded = false;
                    setAssetBackgroundDrawable(defaultImage);
                    if (loadedListener != null)
                    {
                        loadedListener.onAssetLoaded(mIsAssetLoaded);
                    }
                }
            } else
            {

                mIsAssetLoaded = false;
                setAssetBackgroundDrawable(defaultImage);
                if (loadedListener != null)
                {
                    loadedListener.onAssetLoaded(mIsAssetLoaded);
                }
            }
        } else
        {
            mIsAssetLoaded = false;
            if (loadedListener != null)
            {
                loadedListener.onAssetLoaded(mIsAssetLoaded);
            }
        }
    }

    public void setAssetBackgroundDrawable(Drawable defaultImage)
    {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
        {
            mImageView.setBackground(defaultImage);
        } else
        {
            mImageView.setBackgroundDrawable(defaultImage);
        }
    }

    public boolean getIsAssetLoaded()
    {
        return mIsAssetLoaded;
    }

    /**
     * Generate WebView that plays an webP animations
     *
     * @return WebView
     */
    private MyWebView generateWebPAnimatableWebView()
    {
        MyWebView kidozWebView = new MyWebView(getContext());
        kidozWebView.setBackgroundColor(Color.TRANSPARENT);
        kidozWebView.setClipToPadding(false);
        kidozWebView.setDrawingCacheBackgroundColor(0x00000000);
        kidozWebView.getSettings().setLayoutAlgorithm(WebSettings.LayoutAlgorithm.SINGLE_COLUMN);
        kidozWebView.setScrollBarStyle(View.SCROLLBARS_INSIDE_OVERLAY);
        kidozWebView.setScrollbarFadingEnabled(true);
        kidozWebView.getSettings().setAppCacheEnabled(false);
        kidozWebView.getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);
        kidozWebView.getSettings().setDisplayZoomControls(false);
        kidozWebView.getSettings().setAllowContentAccess(true);
        kidozWebView.getSettings().setAllowFileAccess(true);
        kidozWebView.getSettings().setSupportZoom(false);
        kidozWebView.getSettings().setSupportMultipleWindows(false);
        kidozWebView.setVerticalScrollBarEnabled(false);
        kidozWebView.setHorizontalScrollBarEnabled(false);
        kidozWebView.setScrollContainer(false);
        kidozWebView.setPadding(0, 0, 0, 0);
        kidozWebView.setLayerType(View.LAYER_TYPE_HARDWARE, null);

        if (Build.VERSION.SDK_INT >= 21)
        {
            kidozWebView.getSettings().setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
        }

        return kidozWebView;
    }

    private void loadImageToWebPwebView(String path)
    {
        if (mWebPwebView != null)
        {
            String s = "<html>" +
                    "<head>" +
                    "<style type='text/css'> img {max-width: 100%;height:initial;} margin: 0; padding: 0 </style>" +
                    "</head>" +
                    "<body style='margin:0;padding:0;' >" + "<img src=\"" + path + "\"/></body>" +
                    "</html>";
            mWebPwebView.loadDataWithBaseURL("", s, "text/html", "utf-8", null);
        }
    }

    @Override
    protected void onFocusChanged(boolean gainFocus, int direction, Rect previouslyFocusedRect)
    {
        super.onFocusChanged(gainFocus, direction, previouslyFocusedRect);

        if (gainFocus)
        {
            if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled() && mAnimatedGifDrawable.isVisible())
            {
                if (!mAnimatedGifDrawable.isPlaying())
                {
                    mAnimatedGifDrawable.start();
                }
            }
        }
    }

    @Override
    protected void onDetachedFromWindow()
    {
        super.onDetachedFromWindow();
        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
        {
            mAnimatedGifDrawable.stop();
            mAnimatedGifDrawable.recycle();
            mAnimatedGifDrawable = null;

            if (mWebPwebView != null)
            {
                mWebPwebView.clearHistory();
                mWebPwebView.clearCache(true);
                mWebPwebView.loadUrl("about:blank");
                mWebPwebView.freeMemory();
                mWebPwebView.pauseTimers();
                mWebPwebView = null;
            }
        }
    }

    @Override
    protected void onVisibilityChanged(View changedView, int visibility)
    {
        super.onVisibilityChanged(changedView, visibility);

        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
        {
            if (visibility == View.VISIBLE)
            {
                if (!mAnimatedGifDrawable.isPlaying())
                {
                    mAnimatedGifDrawable.start();
                }
            } else
            {
                mAnimatedGifDrawable.stop();
            }
        }
    }

    public interface IOnAssetLoadedListener
    {
        void onAssetLoaded(boolean successful);
    }

    /**
     * Add transformation to be applied on loaded png / jpg image
     *
     * @param transformation
     */
    public void addTransformation(Transformation transformation)
    {
        if(mTransformations != null) {
            mTransformations.add(transformation);
        }
    }

    /**
     * Clear transformations
     */
    private void clearTrasnformations()
    {
        mTransformations.clear();
    }

    /**
     * Start gif animation if the AssetView is loaded the animated gif and the gif is animatable
     * Otherwise do nothing
     */
    public void startGifAnim()
    {
        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
        {
            if (!mAnimatedGifDrawable.isPlaying())
            {
                mAnimatedGifDrawable.start();
            }
        }
    }

    /**
     * Stop gif animation if the AssetView is loaded the animated gif and the gif is animatable
     * Otherwise do nothing
     */
    public void stopGifAnim()
    {
        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
        {
            mAnimatedGifDrawable.stop();
            mAnimatedGifDrawable.seekToFrame(0);
        }
    }

    /**
     * Pause gif animation if the AssetView is loaded the animated gif and the gif is animatable
     * Otherwise do nothing
     */
    public void pauseGifAnim()
    {
        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
        {
            if (mAnimatedGifDrawable.isPlaying())
            {
                mAnimatedGifDrawable.pause();
            }
        }
    }

    /**
     * Set asset view scale type
     */
    public void setScaleType(ImageView.ScaleType scaleType)
    {
        mImageView.setScaleType(scaleType);
    }

    class MyWebView extends WebView
    {

        public MyWebView(Context context)
        {
            super(context);
            if (Build.VERSION.SDK_INT >= 21)
            {
                getSettings().setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
            }
        }

        @Override
        public boolean onInterceptTouchEvent(MotionEvent ev)
        {
            return false;
        }

        @Override
        public boolean onTouchEvent(MotionEvent event)
        {
            return false;
        }
    }
}
