package com.kidoz.sdk.api.ui_views.custom_drawables;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.ColorFilter;
import android.graphics.CornerPathEffect;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PixelFormat;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.drawable.Drawable;

import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.ui_views.panel_view.PANEL_TYPE;

public class DefaultPanelDrawable extends Drawable
{
    private static final float SIZE_RATIO = 0.54f;
    private static final float LINE_WIDTH = 0.258f;

    public enum PanelDrawableType
    {
        OPEN_VIEW, CLOSE_VIEW
    }

    private Paint mXPaint;
    private Paint mBgPaint;
    private int mWidth;
    private int mHeight;

    private RectF mWhiteBackgroundRoundedRectF;
    private RectF mWhiteBackgroundRectF;
    private int mBaseColor;
    private int mXcolor;
    private PANEL_TYPE mPanelType;
    private PanelDrawableType mDrawableType;

    private Path mPath = new Path();
    private boolean mDrawXImg = true;

    public DefaultPanelDrawable(Context context, PANEL_TYPE panel_type, PanelDrawableType panelDrawableType)
    {
        float scaleFactor = ScreenUtils.getScaleFactor(context);

        mWidth *= scaleFactor;
        mHeight *= scaleFactor;

        mBaseColor = Color.parseColor("#ffffff");
        mXcolor = Color.parseColor("#cdffffff");

        mXPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mXPaint.setStyle(Paint.Style.FILL_AND_STROKE);
        mXPaint.setStrokeJoin(Paint.Join.ROUND);    // set the join to round you want
        mXPaint.setStrokeCap(Paint.Cap.ROUND);      // set the paint cap to round too
        mXPaint.setPathEffect(new CornerPathEffect(15));
        mXPaint.setColor(mXcolor);
        mXPaint.setShadowLayer(4.5f, 0, 5.0f, 0x45000000);
        mXPaint.setDither(true);

        mBgPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mBgPaint.setStyle(Paint.Style.FILL_AND_STROKE);
        mBgPaint.setColor(mBaseColor);
        mBgPaint.setDither(true);

        mPanelType = panel_type;

        mDrawableType = panelDrawableType;
        initRectF();
    }

    @Override
    public void setAlpha(int alpha)
    {
        mBgPaint.setAlpha(alpha);
        mXPaint.setAlpha(alpha);
    }

    @Override
    public void setColorFilter(ColorFilter cf)
    {
        mBgPaint.setColorFilter(cf);
        mXPaint.setColorFilter(cf);
    }

    @Override
    public int getOpacity()
    {
        return PixelFormat.TRANSLUCENT;
    }

    @Override
    protected void onBoundsChange(Rect bounds)
    {
        super.onBoundsChange(bounds);

        mWidth = bounds.width();
        mHeight = bounds.height();

        initRectF();
    }

    private void initRectF()
    {
        float left;
        float top;
        float right;
        float bottom;

        switch (mPanelType)
        {
            case LEFT:
            {
                left = 0f;
                top = 0f;
                right = mWidth / 2;
                bottom = mHeight;
                mWhiteBackgroundRectF = new RectF(left, top, right, bottom);
                break;
            }
            case TOP:
            {
                left = 0f;
                top = 0f;
                right = mWidth;
                bottom = mHeight / 2;
                mWhiteBackgroundRectF = new RectF(left, top, right, bottom);
                break;
            }
            case RIGHT:
            {
                left = mWidth / 2;
                top = 0f;
                right = mWidth;
                bottom = mHeight;
                mWhiteBackgroundRectF = new RectF(left, top, right, bottom);
                break;
            }
            case BOTTOM:
            {
                left = 0f;
                top = mHeight / 2;
                right = mWidth;
                bottom = mHeight;
                mWhiteBackgroundRectF = new RectF(left, top, right, bottom);
                break;
            }
        }
        left = 0f;
        top = 0f;
        right = mWidth;
        bottom = mHeight;
        mWhiteBackgroundRoundedRectF = new RectF(left, top, right, bottom);

        // REMOVE IF NEEDED

        if (mDrawXImg)
        {
            float centerX = mWidth / 2f;
            float centerY = mHeight / 2f;

            float size = mWidth * SIZE_RATIO;
            float width = mWidth * LINE_WIDTH;

            float offsetSize = size / 2f - width / 3.5f;
            float offsetWidth = width / 2f;

            mPath.moveTo(centerX - offsetWidth / 2f, centerY + offsetWidth / 2f);
            mPath.lineTo(centerX - offsetSize - offsetWidth / 2f, centerY + offsetWidth / 2f);
            mPath.lineTo(centerX - offsetSize - offsetWidth / 2f, centerY - offsetWidth / 2f);
            mPath.lineTo(centerX - offsetWidth / 2f, centerY - offsetWidth / 2f);
            mPath.lineTo(centerX - offsetWidth / 2f, centerY - offsetWidth / 2f - offsetSize);
            mPath.lineTo(centerX + offsetWidth / 2f, centerY - offsetWidth / 2f - offsetSize);
            mPath.lineTo(centerX + offsetWidth / 2f, centerY - offsetWidth / 2f);

            mPath.lineTo(centerX + offsetWidth / 2f + offsetSize, centerY - offsetWidth / 2f);
            mPath.lineTo(centerX + offsetWidth / 2f + offsetSize, centerY + offsetWidth / 2f);
            mPath.lineTo(centerX + offsetWidth / 2f, centerY + offsetWidth / 2f);

            mPath.lineTo(centerX + offsetWidth / 2f, centerY + offsetWidth / 2f + offsetSize);
            mPath.lineTo(centerX - offsetWidth / 2f, centerY + offsetWidth / 2f + offsetSize);
            mPath.lineTo(centerX - offsetWidth / 2f, centerY + offsetWidth / 2f);
            mPath.lineTo(centerX - offsetSize - offsetWidth / 2f, centerY + offsetWidth / 2f);
        }
    }

    @Override
    public void draw(Canvas canvas)
    {
        // Draw background shape
        float centerX = mWidth / 2f;
        float centerY = mHeight / 2f;

        mBgPaint.setColor(mBaseColor);
        mXPaint.setColor(mXcolor);

        float radius = (20f / 142) * mWidth;
        canvas.drawRect(mWhiteBackgroundRectF, mBgPaint);
        canvas.drawRoundRect(mWhiteBackgroundRoundedRectF, radius, radius, mBgPaint);

        switch (mDrawableType)
        {
            case OPEN_VIEW:
            {
                canvas.drawPath(mPath, mXPaint);
                break;
            }

            case CLOSE_VIEW:
            {
                int saveCount = canvas.save();
                canvas.rotate(45f, centerX, centerY);
                canvas.drawPath(mPath, mXPaint);
                canvas.restoreToCount(saveCount);
                break;
            }
        }
    }

    public void setPanelType(PANEL_TYPE panelType)
    {
        mPanelType = panelType;
        initRectF();
    }

    public void setBaseColor(int baseColor)
    {
        mBaseColor = baseColor;
        mBgPaint.setColor(baseColor);
    }

    public void setDrawXImage(boolean draw)
    {
        mDrawXImg = draw;
        invalidateSelf();
    }
}