package com.kidoz.sdk.api.players.web_player;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.graphics.Color;
import android.graphics.Point;
import android.os.Build;
import android.os.Bundle;
import android.os.Looper;
import android.support.v4.content.LocalBroadcastManager;
import android.view.View;
import android.webkit.CookieManager;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.BaseDialog;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.WidgetEventMessage;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.KidozTopBar;
import com.kidoz.sdk.api.ui_views.html_view.HtmlViewWrapper;
import com.kidoz.sdk.api.ui_views.html_view.IOnHtmlWebViewInterface;

import org.greenrobot.eventbus.EventBus;
import org.json.JSONArray;

public class WebPlayerDialog extends BaseDialog
{
    private static final int DEFAULT_TOP_BAR_COLOR = Color.parseColor("#36a7ec");

    private static final String TAG = WebPlayerDialog.class.getSimpleName();
    private static final int WEB_PLAYER_RESOURCE_ID = 1;

    private static final String CONTENT_ITEM_SAVE_STATE_KEY = "CONTENT_ITEM_SAVE_STATE_KEY_WP";
    private static final String WEB_PLAYER_TYPE_SAVE_STATE_KEY = "WEB_PLAYER_TYPE_SAVE_STATE_KEY_WP";

    private RelativeLayout mInnerContainer;
    private ContentItem mContentItem;
    private WebPlayerType mWebPlayerType;
    private KidozTopBar mKidozTopBar;

    private WebPlayerView mWebPlayerView;
    private HtmlViewWrapper mHtmlViewWrapper;


    private String mWidgetType = null;
    private String mStyleId = null;
    private Utils.StaticHandler staticHandler = new Utils.StaticHandler(Looper.getMainLooper());
    private BroadcastReceiver mBroadcastReceiver;

    public WebPlayerDialog(Context context, ContentItem contentItem, String widgetType, String styleId, WebPlayerType webPlayerType, boolean allowOrientationModification)
    {
        super(context, android.R.style.Theme_Translucent_NoTitleBar_Fullscreen,allowOrientationModification);
        if (webPlayerType != null && webPlayerType == WebPlayerType.EXTERNAL_LINK)
        {
            setCancelable(false);
        }
        else
        {
            setCancelable(true);
        }

        mContentItem = contentItem;

        mWebPlayerType = webPlayerType;
        this.mWidgetType = widgetType;
        mStyleId = styleId;
    }

    /**
     * Set timer to auto close this dialog
     */
    public void setAutoCloseTimer()
    {
        mBroadcastReceiver = new BroadcastReceiver()
        {
            @Override
            public void onReceive(Context context, Intent intent)
            {
                if (intent != null && intent.getAction().equals("CloseRedirectWebsiteDialog") == true)
                {
                    LocalBroadcastManager.getInstance(getContext()).unregisterReceiver(mBroadcastReceiver);
                    closeDialog();
                }
            }
        };
        LocalBroadcastManager.getInstance(getContext()).registerReceiver(mBroadcastReceiver, new IntentFilter("CloseRedirectWebsiteDialog"));
    }

    @Override
    protected void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        if (savedInstanceState != null)
        {
            if (savedInstanceState.containsKey(CONTENT_ITEM_SAVE_STATE_KEY))
            {
                mContentItem = (ContentItem) savedInstanceState.getSerializable(CONTENT_ITEM_SAVE_STATE_KEY);
            }
            mWebPlayerType = (WebPlayerType) savedInstanceState.getSerializable(WEB_PLAYER_TYPE_SAVE_STATE_KEY);
        }

        setOnDismissListener(new OnDismissListener()
        {
            @Override
            public void onDismiss(DialogInterface dialog)
            {
            }
        });

        try
        {
            initDialogView();
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error initDialog: " + ex.getMessage());
        }
    }

    private void initDialogView()
    {
        mInnerContainer = new RelativeLayout(getContext());
        setContentView(mInnerContainer, new RelativeLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT));

        if (mWebPlayerType == WebPlayerType.HTML_PLAYBACK)
        {
            addHtmlPlaybackView();
        }
        else
        {
            addWebPlayerView();
        }

        initTopBar();

        // 3. Create video layout.
        RelativeLayout videoLayout = new RelativeLayout(getContext());
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT);
        videoLayout.setId(Utils.generateViewId());
        mInnerContainer.addView(videoLayout, params);
        if (mWebPlayerView != null)
        {
            mWebPlayerView.setFullScreenWebChromeClient(videoLayout);
        }

        if (mWebPlayerType != null && mWebPlayerType == WebPlayerType.EXTERNAL_LINK)
        {
            mInnerContainer.setBackgroundColor(Color.parseColor("#77000000"));
            mKidozTopBar.setVisibility(View.INVISIBLE);
        }
        else
        {
            mInnerContainer.setBackgroundColor(Color.WHITE);
        }

        if (mContentItem.getName() != null)
        {
            mKidozTopBar.setTitle(mContentItem.getName());
        }
        else
        {
            mKidozTopBar.setTitle("");
        }

        if (mWebPlayerType != null && mWebPlayerType == WebPlayerType.ROVIO)
        {
            mKidozTopBar.setVisibility(View.GONE);
            CookieManager.getInstance().setAcceptCookie(true);
            if (mWebPlayerView != null && Build.VERSION.SDK_INT >= 21)
            {
                CookieManager.getInstance().setAcceptThirdPartyCookies(mWebPlayerView.getKidozWebView(), true);
            }
        }
    }

    private void addHtmlPlaybackView()
    {
        mHtmlViewWrapper = new HtmlViewWrapper(mCallingActivity, false);
        mHtmlViewWrapper.setInFocusActivityContext(mCallingActivity);
        mHtmlViewWrapper.setHtmlWebViewListener(new IOnHtmlWebViewInterface()
        {
            @Override
            public void onHtmlStartLoading()
            {
                super.onHtmlStartLoading();
                mHtmlViewWrapper.showLoadingProgressView();
            }

            @Override
            public void onHtmlFinishedLoading()
            {
                super.onHtmlFinishedLoading();
                mHtmlViewWrapper.hideLoadingProgressView();
            }

            @Override
            public void onClose()
            {
                super.onClose();
                closeDialog();
            }
        });

        mHtmlViewWrapper.setData(mContentItem);
        mHtmlViewWrapper.loadHtml();

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT);
        Point point = Utils.getScreenSize(getContext());
        int paddingTop = (int) (Math.max(point.x, point.y) * KidozTopBar.BACKGROUND_RATIO);
        params.topMargin = paddingTop;
        mInnerContainer.addView(mHtmlViewWrapper, params);
    }

    private void addWebPlayerView()
    {
        mWebPlayerView = new WebPlayerView(getWindow(), mCallingActivity, mWebPlayerType, mWidgetType);

        mWebPlayerView.getKidozWebView().addJavascriptInterface(new WebPlayerJavaInterface(), "KidozAndroid");
        if (mContentItem != null)
        {
            String[] allowedDomains = null;
            try
            {
                if (mContentItem.getExtraParameters() != null)
                {
                    JSONArray jsonArray = mContentItem.getExtraParameters().optJSONArray("allowed_domains");
                    if (jsonArray != null)
                    {
                        allowedDomains = new String[jsonArray.length()];
                        for (int keyIndex = 0; keyIndex < jsonArray.length(); keyIndex++)
                        {
                            allowedDomains[keyIndex] = jsonArray.getString(keyIndex);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                SDKLogger.printErrorLog(TAG, "Error when trying to gnerate website's allowed domains: " + ex.getMessage());
            }
            mWebPlayerView.loadWebUrlWithAllowedDomains(allowedDomains, mContentItem.getData(), mContentItem.getId());
        }

        mWebPlayerView.setOnRemoveViewRequestListener(new WebPlayerView.IOnRemoveViewRequestListener()
        {
            @Override
            public void onRemoveView()
            {
                closeDialog();
            }
        });

        mInnerContainer.addView(mWebPlayerView);
    }

    private void initTopBar()
    {
        //Create top header view
        mKidozTopBar = new KidozTopBar(getContext(), true, mWidgetType, null);
        mKidozTopBar.setId(Utils.generateViewId());
        mKidozTopBar.setTopBarBackgroundColor(DEFAULT_TOP_BAR_COLOR);
        mKidozTopBar.setKidozTopBarListener(new KidozTopBar.KidozTopBarListener()
        {
            @Override
            public void onExitClicked(View view)
            {
                closeDialog();
            }
        });
        mKidozTopBar.showParentalLockIcon(false);

        // Add header lay mo view
        mInnerContainer.addView(mKidozTopBar, new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.WRAP_CONTENT));
    }

    @Override
    protected void onStart()
    {
        super.onStart();
    }

    @Override
    protected void onStop()
    {
        super.onStop();
    }

    @Override
    public Bundle onSaveInstanceState()
    {
        Bundle bundle = super.onSaveInstanceState();
        if (bundle != null)
        {
            bundle.putSerializable(WEB_PLAYER_TYPE_SAVE_STATE_KEY, mWebPlayerType);
        }
        return bundle;
    }

    @Override
    public void closeDialog()
    {
        super.closeDialog();
        Intent intent = new Intent();
        intent.setAction("WebPlayerDialogClosed");
        LocalBroadcastManager.getInstance(getContext()).sendBroadcast(intent);
    }

    @Override
    public void dismiss()
    {
        super.dismiss();
        if (mWebPlayerType != null)
        {
            if (mWebPlayerType == WebPlayerType.ONLINE_GAME || mWebPlayerType == WebPlayerType.WEB_BROWSER || mWebPlayerType == WebPlayerType.HTML_PLAYBACK)
            {
                WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.PLAYER_CLOSE, WidgetType.createFromString(mWidgetType));
                EventBus.getDefault().post(eventMessage);
            }
        }
    }

    @Override
    public void openDialog()
    {
        super.openDialog();
        String action = "";
        switch (mWebPlayerType)
        {
            case EXTERNAL_LINK:
                action = EventParameters.ACTION_EXTERNAL_LINK_VIEW;
                break;
            case ONLINE_GAME:
            {
                WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.PLAYER_OPEN, WidgetType.createFromString(mWidgetType));
                EventBus.getDefault().post(eventMessage);
                action = EventParameters.ACTION_GAME_VIEW;
                break;
            }
            case WEB_BROWSER:
            {
                WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.PLAYER_OPEN, WidgetType.createFromString(mWidgetType));
                EventBus.getDefault().post(eventMessage);
                action = EventParameters.ACTION_WEBSITE_VIEW;
                break;
            }

            case HTML_PLAYBACK:
            {
                WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.PLAYER_OPEN, WidgetType.createFromString(mWidgetType));
                EventBus.getDefault().post(eventMessage);
                action = EventParameters.ACTION_HTML_VIEW;
                break;
            }
        }

        EventManager.getInstance(getContext()).logEventWithDuration(this.getContext(), mWidgetType, mStyleId, EventManager.LOG_NORMAL_LEVEL, WEB_PLAYER_RESOURCE_ID, EventManager.LOG_CLOSE_RESOURCE_LEVEL, EventParameters.CATEGORY_WEB_PLAYER, action, mContentItem.getName(), mContentItem.getId(), false);
    }

    @Override
    public void onDetachedFromWindow()
    {
        super.onDetachedFromWindow();
        staticHandler.removeCallbacksAndMessages(null);
        if (mWebPlayerView != null)
        {
            try
            {
                mWebPlayerView.releaseAndcloseView();
            }
            catch (Exception ex)
            {
                SDKLogger.printDebbugLog(TAG, "Error when trying to onDetach: " + ex.getMessage());
            }
        }
        if (mHtmlViewWrapper != null)
        {
            try
            {
                mHtmlViewWrapper.stopAndReleaseWebView();
            }
            catch (Exception ex)
            {
                SDKLogger.printDebbugLog(TAG, "Error when trying to mHtmlViewWrapper.stopAndReleaseWebView(): " + ex.getMessage());
            }
        }

        EventManager.getInstance(getContext()).logEventWithDurationEnd(getContext(), mWidgetType, mStyleId, WEB_PLAYER_RESOURCE_ID, false);
    }

    @Override
    protected void onDialogVisibilityChange(int isVisible)
    {
        super.onDialogVisibilityChange(isVisible);

        if (isVisible != View.VISIBLE)
        {
            staticHandler.removeCallbacksAndMessages(null);
        }
    }

    @Override
    protected void onDialogWindowVisibilityChange(boolean visible)
    {
        super.onDialogWindowVisibilityChange(visible);

        if (visible == false)
        {
            staticHandler.removeCallbacksAndMessages(null);
        }
    }

    class WebPlayerJavaInterface
    {
        @android.webkit.JavascriptInterface
        public void onInvokeCloseClick()
        {
            try
            {
                staticHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        closeDialog();
                    }
                });
            }
            catch (Exception ex)
            {
                SDKLogger.printErrorLog(TAG, "Error when trying to close web dialog: " + ex.getMessage());
            }
        }
    }
}
