package com.kidoz.sdk.api.dialogs.WebDialog;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Configuration;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.AttributeSet;
import android.webkit.WebChromeClient;
import android.webkit.WebSettings;
import android.webkit.WebView;

import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.Utils;

import java.util.Map;

/**
 * Created by roni on 10/13/2015.
 */
public class VideoEnabledWebView extends WebView
{
    private final String TAG = VideoEnabledWebView.class.getSimpleName();
    private ConfigurationChangeListener mConfigurationChangeListener;
    private Utils.StaticHandler staticHandler = new Utils.StaticHandler(Looper.getMainLooper());


    private WebChromeClient videoEnabledWebChromeClient;
    private boolean addedJavascriptInterface;

    public VideoEnabledWebView(Context context)
    {
        super(context);
        init();
        addedJavascriptInterface = false;
    }

    public VideoEnabledWebView(Context context, AttributeSet attrs)
    {
        super(context, attrs);
        init();
        addedJavascriptInterface = false;
    }

    public VideoEnabledWebView(Context context, AttributeSet attrs, int defStyle)
    {
        super(context, attrs, defStyle);
        init();
        addedJavascriptInterface = false;
    }

    private void init()
    {
        if (Build.VERSION.SDK_INT >= 21)
        {
            getSettings().setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
        }

        videoEnabledWebChromeClient  = new WebChromeClient();
        setWebChromeClient(videoEnabledWebChromeClient);
    }


    @Override
    public void loadData(String data, String mimeType, String encoding)
    {
        addJavascriptInterface();
        super.loadData(data, mimeType, encoding);
    }

    @Override
    public void loadDataWithBaseURL(String baseUrl, String data, String mimeType, String encoding, String historyUrl)
    {
        addJavascriptInterface();
        super.loadDataWithBaseURL(baseUrl, data, mimeType, encoding, historyUrl);
    }

    @Override
    public void loadUrl(String url)
    {
        addJavascriptInterface();
        super.loadUrl(url);
    }

    public void invokeJS(String url)
    {
        addJavascriptInterface();
        try
        {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT)
            {
                super.evaluateJavascript(url, null);
            } else
            {
                super.loadUrl(url);
            }
        } catch (Exception ex)
        {
        }
    }

    @Override
    public void loadUrl(String url, Map<String, String> additionalHttpHeaders)
    {
        addJavascriptInterface();
        super.loadUrl(url, additionalHttpHeaders);
    }

    @SuppressLint("JavascriptInterface")
    private void addJavascriptInterface()
    {
        if (!addedJavascriptInterface)
        {
            // Add javascript interface to be called when the video ends (must be done before page load)
            addJavascriptInterface(new VideoJavaScriptInterface(), "_VideoEnabledWebView"); // Must match Javascript interface name of VideoEnabledWebChromeClient
            addedJavascriptInterface = true;
        }
    }

    class VideoJavaScriptInterface
    {
        @android.webkit.JavascriptInterface
        public void notifyVideoEnd() // Must match Javascript interface method of VideoEnabledWebChromeClient
        {
            // This code is not executed in the UI thread, so we must force it to happen
            new Handler(Looper.getMainLooper()).post(new Runnable()
            {
                @Override
                public void run()
                {
                    if (videoEnabledWebChromeClient != null)
                    {
                        videoEnabledWebChromeClient.onHideCustomView();
                    }
                }
            });
        }
    }

    public void seekTo(int progress)
    {
        SDKLogger.printDebbugLog(TAG, ">>>>VIDEO TIME: Seek to (in seconds) = " + String.valueOf(progress / 1000));
        invokeJS("javascript:seek(\"" + (progress / 1000) + "\")");
    }

    public void loadVideo(String videoDomain, String videoURL)
    {
        SDKLogger.printDebbugLog(TAG, ">>>>loadVideo: Domain = " + videoDomain + "\nURL = " + videoURL);
        invokeJS("javascript:initVideoPlayer(" +
                "\"" + videoDomain + "\"," +
                "\"" + videoURL + "\")");
    }

    public void requestCurrentVideoTime()
    {
        invokeJS("javascript:requestCurrentVideoTime()");
    }

    public void pauseVideo()
    {
        invokeJS("javascript:pause()");
    }

    public void stopVideo()
    {
        invokeJS("javascript:stop()");
    }

    public void resumeVideo()
    {
        invokeJS("javascript:resume()");
    }

    public void refreshVideo()
    {
        invokeJS("javascript:refresh()");
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig)
    {
        super.onConfigurationChanged(newConfig);
        if (mConfigurationChangeListener != null)
        {
            mConfigurationChangeListener.onConfigChange();
        }
    }

    public void setOnConfigurationChangeListener(ConfigurationChangeListener configurationChangeListener)
    {
        mConfigurationChangeListener = configurationChangeListener;
    }

    public interface ConfigurationChangeListener
    {
        public void onConfigChange();
    }
}
