package com.kidoz.sdk.api;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.view.View;

import com.kidoz.sdk.api.general.ContentLogicLoader;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.CookieManager;
import com.kidoz.sdk.api.general.utils.StorageLife;
import com.kidoz.sdk.api.interfaces.IOnFeedViewEventListener;
import com.kidoz.sdk.api.interfaces.KidozPlayerEventHelper;
import com.kidoz.sdk.api.interfaces.KidozPlayerListener;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.kidoz.sdk.api.structure.ContentData;
import com.kidoz.sdk.api.ui_views.feed_view.FeedFamilyViewDialog;
import com.kidoz.sdk.api.ui_views.feed_view.FeedViewDialog;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.json.JSONObject;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * KIDOZ FeedView widget class
 */
public class FeedView
{
    // Properties keys
    public static final String KIDOZ_FEED_BUTTON = "feedButton";
    public static final String KIDOZ_FEED_BUTTON_SCALE = "feedButtonScale";
    public static final String OVERRIDE_USER_PROPERTIES = "overrideUserProperties";
    public static final String LOUNCH_INTERSTITIAL_ON_OPEN = "lounchInterstitialOnOpen";

    public static final String TAG = FeedView.class.getSimpleName();
    public static final String STYLE_ID = "style_id";

    private Builder mBuilder;
    private Lock raceConditionLock = new ReentrantLock();

    private FeedViewDialog mFeedViewDialog;
    private FeedFamilyViewDialog mFamilyViewDialog;
    protected ContentLogicLoader mContentLogicLoader;

    protected IOnFeedViewEventListener mOnFeedViewEventListener;
    protected boolean mIsAutomaticInvocation = false;
    protected boolean mIsShowing = false;
    protected String mFeedViewType = null;

    protected boolean mShowWhenInitiated = false;
    protected JSONObject mJsonObjectProperties;
    protected static boolean mOncePerSessionLoaded = false;

    private KidozPlayerEventHelper mKidozPlayerEventHelper;
    private Dialog.OnDismissListener dismissListener;
    private KidozPlayerListener mKidozPlayerListener;

    private boolean mIsViewReadyCalled = false; // This flag used to know if the call of onViewReady() has been made before set of FeedView listener (Asynchronous Bug in FeedButton case)
    // In that case we will launch onViewReady() as soon as listener is set
    private boolean mIsReadyToShowCalled = false; // The same issue with the above flag applies to onReadyToShow() call

    /**
     * Set kidoz players event listeners
     * Used for listening to invoked Player state events (Applied for single view)
     *
     * @param kidozPlayerListener event listener
     */
    public void setKidozPlayerListener(KidozPlayerListener kidozPlayerListener)
    {
        mKidozPlayerListener = kidozPlayerListener;
    }

    protected FeedView(Builder builder)
    {
        mBuilder = builder;

        initContentLogicLoader();

        if (EventBus.getDefault().isRegistered(this) == false)
        {
            EventBus.getDefault().register(this);
        }

        // Has been initialised
        if (KidozSDK.isInitialised() == true)
        {
            checkIfValidationNeededAndInit();
        }

        mKidozPlayerEventHelper = new KidozPlayerEventHelper(new KidozPlayerEventHelper.IPlayerListener()
        {
            @Override
            public void onOpen(boolean isMaximizedPlayer)
            {
                if (mKidozPlayerListener != null)
                {
                    mKidozPlayerListener.onPlayerOpen();
                }

                if (mFamilyViewDialog != null && mFamilyViewDialog.isShowing())
                {
                    mFamilyViewDialog.pauseAction(isMaximizedPlayer);
                }
            }

            @Override
            public void onClose(boolean isMaximizedPlayer)
            {
                if (mKidozPlayerListener != null)
                {
                    mKidozPlayerListener.onPlayerClose();
                }

                if (mFamilyViewDialog != null && mFamilyViewDialog.isShowing())
                {
                    mFamilyViewDialog.resumeAction(isMaximizedPlayer);
                }
            }
        }, WidgetType.WIDGET_TYPE_FEED);

        dismissListener = new DialogInterface.OnDismissListener()
        {
            @Override
            public void onDismiss(DialogInterface dialog)
            {

                if (mKidozPlayerEventHelper != null)
                {
                    mKidozPlayerEventHelper.unRegister();
                }

                if (mOnFeedViewEventListener != null)
                {
                    mOnFeedViewEventListener.onDismissView();
                }

                mIsShowing = false;
            }
        };
    }

    /**
     * Check if needed a validation
     */
    private void checkIfValidationNeededAndInit()
    {
        unregisterEventBus();

        if (raceConditionLock.tryLock())
        {
            try
            {
                if (mOncePerSessionLoaded == true)
                {
                    mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
                    if (mJsonObjectProperties != null)
                    {
                        mFeedViewType = mJsonObjectProperties.optString(FeedView.TAG);
                        EventBus.getDefault().post(new EventMessage(EventMessage.MessageType.INIT_FEED_BUTTON));

                        if (mOnFeedViewEventListener != null)
                        {
                            mOnFeedViewEventListener.onViewReady();
                        } else
                        {
                            mIsViewReadyCalled = true;
                        }

                        if (mShowWhenInitiated)
                        {
                            showView();
                        }
                    } else
                    {
                        validateLayoutAndAssets();
                    }
                } else
                {
                    validateLayoutAndAssets();
                }
            } finally
            {
                raceConditionLock.unlock();
            }
        }
    }

    // Event bus callback
    @Subscribe
    public void onHandleEvent(EventMessage event)
    {
        if (event.getMessageType() == EventMessage.MessageType.INIT_SDK)
        {
            checkIfValidationNeededAndInit();
        }
    }

    private void unregisterEventBus()
    {
        if (EventBus.getDefault().isRegistered(this))
        {
            EventBus.getDefault().unregister(this);
        }
    }

    private void validateLayoutAndAssets()
    {
        SdkAPIManager.getSdkApiInstance(mBuilder.mContext).getStyle(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getValue(), new ApiResultCallback<Boolean>()
        {
            @Override
            public void onServerResult(ResultData<?> result)
            {
                // If result success notify wrapper to build the banner.
                if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true)
                {
                    if (result.getData() != null)
                    {
                        boolean isAssetsReady = (Boolean) result.getData();
                        if (isAssetsReady == true)
                        {
                            mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
                            if (mJsonObjectProperties != null)
                            {
                                mFeedViewType = mJsonObjectProperties.optString(FeedView.TAG);
                                EventBus.getDefault().post(new EventMessage(EventMessage.MessageType.INIT_FEED_BUTTON));
                                mOncePerSessionLoaded = true;

                                if (mOnFeedViewEventListener != null)
                                {
                                    mOnFeedViewEventListener.onViewReady();
                                } else
                                {
                                    mIsViewReadyCalled = true;
                                }

                                if (mShowWhenInitiated)
                                {
                                    showView();
                                }
                            }
                        } else
                        {
                            onFailed();
                        }
                    }
                } else
                {
                    onFailed();
                }
            }

            @Override
            public void onFailed()
            {

            }
        });
    }

    /**
     * Initiate  content loader logic ,that handles all the logic for loading content
     **/
    private void initContentLogicLoader()
    {
        mContentLogicLoader = new ContentLogicLoader(new ContentLogicLoader.IOnContentDataReadyCallback()
        {
            @Override
            public void onDataReady(ContentData contentData)
            {
                if (mFeedViewType.equals(FeedViewDialog.TAG))
                {
                    mFeedViewDialog.setContentData(contentData);
                } else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
                {
                    mFamilyViewDialog.setContentData(contentData);
                }

                // Reset The storage of the impressions for the HTML content
                CookieManager.resetStorage(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getStringValue(), StorageLife.WIDGET);
            }

            @Override
            public void onLoadContentFailed()
            {

            }
        });
    }

    /**
     * Show feed interstitial view (Will be shown <b>Before</b> content data is ready)
     **/
    public void showView()
    {
        showViewInner();
    }

    private void showViewInner()
    {
        if (KidozSDK.isInitialised())
        {
            mIsAutomaticInvocation = true;

            if (mJsonObjectProperties == null)
            {
                mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
            }

            if (mJsonObjectProperties != null)
            {
                openSimpleDialog();
            } else
            {
                mShowWhenInitiated = true;
            }
        } else
        {
            mShowWhenInitiated = true;
        }
    }

    protected synchronized void openSimpleDialog()
    {
        if (mKidozPlayerEventHelper != null)
        {
            mKidozPlayerEventHelper.register();
        }
        if (mJsonObjectProperties != null && mBuilder.mContext != null && mBuilder.mContext instanceof Activity && ((Activity) mBuilder.mContext).isFinishing() == false)
        {
            mFeedViewType = mJsonObjectProperties.optString(FeedView.TAG);
            if (mFeedViewType != null)
            {
                if (mFeedViewType.equals(FeedViewDialog.TAG))
                {
                    if (mFeedViewDialog == null || mFeedViewDialog.isShowing() == false)
                    {
                        mFeedViewDialog = new FeedViewDialog(mBuilder.mContext, mOnFeedViewEventListener, mIsAutomaticInvocation, mJsonObjectProperties);
                        mFeedViewDialog.setOnDismissListener(dismissListener);
                        mFeedViewDialog.setOnCloseButtonClickListener(new View.OnClickListener()
                        {
                            @Override
                            public void onClick(View view)
                            {
                                dismissView();
                                unregisterEventBus();
                            }
                        });

                        if (mOnFeedViewEventListener != null)
                        {
                            mOnFeedViewEventListener.onReadyToShow();
                        } else
                        {
                            mIsReadyToShowCalled = true;
                        }

                        mFeedViewDialog.openDialog();

                        if (mJsonObjectProperties != null)
                        {
                            mContentLogicLoader.loadContent(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getStringValue(), mJsonObjectProperties.optString(STYLE_ID));
                        }
                    }
                } else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
                {
                    if (mFamilyViewDialog == null || (mFamilyViewDialog.isShowing() == false && mFamilyViewDialog.isWaitingToOpen() == false))
                    {
                        mFamilyViewDialog = new FeedFamilyViewDialog(mBuilder.mContext, mOnFeedViewEventListener, mIsAutomaticInvocation, mJsonObjectProperties);
                        mFamilyViewDialog.setOnDismissListener(dismissListener);
                        mFamilyViewDialog.setOnCloseButtonClickListener(new View.OnClickListener()
                        {
                            @Override
                            public void onClick(View view)
                            {
                                dismissView();
                                unregisterEventBus();
                            }
                        });

                        if (mOnFeedViewEventListener != null)
                        {
                            mOnFeedViewEventListener.onReadyToShow();
                        } else
                        {
                            mIsReadyToShowCalled = true;
                        }

                        mFamilyViewDialog.openDialog();

                        if (mJsonObjectProperties != null)
                        {
                            mContentLogicLoader.loadContent(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getStringValue(), mJsonObjectProperties.optString(STYLE_ID));
                        }
                    }
                }
                mShowWhenInitiated = false;
            }
        }
    }

    /**
     * Dismiss feed view
     **/
    public void dismissView()
    {
        if (mFeedViewType != null)
        {
            if (mFeedViewType.equals(FeedViewDialog.TAG))
            {
                if (mFeedViewDialog != null)
                {
                    mFeedViewDialog.closeDialog();
                }
            } else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
            {
                if (mFamilyViewDialog != null)
                {
                    mFamilyViewDialog.closeDialog();
                }
            }
        }
    }

    /**
     * Set on view event listener
     *
     * @param listener
     **/
    public void setOnFeedViewEventListener(IOnFeedViewEventListener listener)
    {
        mOnFeedViewEventListener = listener;
        if (mFeedViewType != null)
        {
            if (mFeedViewType.equals(FeedViewDialog.TAG))
            {
                if (mFeedViewDialog != null)
                {
                    mFeedViewDialog.setOnInterstitialViewEventListener(mOnFeedViewEventListener);
                }
            } else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
            {
                if (mFamilyViewDialog != null)
                {
                    mFamilyViewDialog.setOnInterstitialViewEventListener(mOnFeedViewEventListener);
                }
            }

            if (mIsViewReadyCalled)
            {
                mIsViewReadyCalled = false;
                mOnFeedViewEventListener.onViewReady();
            }

            if (mIsReadyToShowCalled)
            {
                mIsReadyToShowCalled = false;
                mOnFeedViewEventListener.onReadyToShow();
            }
        }
    }

    /**
     * Builder class that handles view initiation
     **/
    public static class Builder
    {
        private Context mContext;

        public Builder(Context context)
        {
            mContext = context;
        }

        public FeedView build()
        {
            return new FeedView(this);
        }
    }
}
