package com.kidoz.sdk.api.ui_views.html_view;

import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.net.Uri;
import android.os.Build;
import android.os.Looper;
import android.os.Message;
import android.webkit.WebChromeClient;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.RelativeLayout;

import com.kidoz.events.DeviceUtils;
import com.kidoz.events.Event;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.KidozSDK;
import com.kidoz.sdk.api.dialogs.AboutKidozDialog;
import com.kidoz.sdk.api.dialogs.ParentalLockDialog;
import com.kidoz.sdk.api.general.ContentExecutionHandler;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.WidgetEventMessage;
import com.kidoz.sdk.api.general.enums.ContentType;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.CookieManager;
import com.kidoz.sdk.api.general.utils.OnSwipeTouchListener;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.general.utils.StorageLife;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.loading_progress_view.LoadingProgressView;
import com.kidoz.sdk.api.ui_views.one_item_view.ItemViewPagerAdapter;

import org.greenrobot.eventbus.EventBus;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.LinkedHashMap;

public class HtmlViewWrapper extends RelativeLayout
{
    private static final float LOADING_PROGRESS_DEFAULT_RATIO = 0.35f;

    private static final String JS_FUNC_ON_FOCUS_ON_KEY = "kidozOnFocusOn";
    private static final String JS_FUNC_ON_PREPARE_FOCUS_ON_KEY = "kidozOnPrepapreFocusOn";
    private static final String JS_FUNC_ON_FOCUS_OFF_KEY = "kidozOnFocusOff";
    private static final String JS_FUNC_ON_MAXIMIZE_KEY = "kidozOnMaximize";
    private static final String JS_FUNC_ON_MINIMIZE_KEY = "kidozOnMinimize";

    private static final int ON_LOAD_FINISHED = 0;
    private static final int ON_LOAD_STARTED = 1;
    private static final int ON_VIEW_READY = 2;
    private static final int ON_CLOSE = 3;
    private static final int ON_RECEIVED_ERROR = 4;
    private static final int ON_NOTIFY_VAST_AD_READY = 5;

    private static final String TAG = HtmlViewWrapper.class.getSimpleName();

    private HtmlFiveWebView mWebView;

    private HtmlAndroidJavaScriptInterface mHtmlAndroidWebInterface;
    private String mStyleId;
    private String mWidgetType = "";

    private boolean mAllowClickHandling = true;

    private ContentItem mContentItem;
    private String mHtmlPageUrl;

    private IOnHtmlWebViewInterface mHtmlWeViewListener;

    private JSONObject mCustomFunctionsMap = new JSONObject();
    private boolean mShowMaximize = false;
    private boolean mIsShowClose = true;
    private boolean mIsAvoidAutoImpression = false;

    protected LoadingProgressView mLoadingProgressView;
    private Utils.StaticHandler mEventHandler;
    private ItemViewPagerAdapter.ViewPagerItemClickListener mViewPagerItemClickListener;
    private Utils.StaticHandler mExecutionHandler = new Utils.StaticHandler(Looper.getMainLooper());

    public Context mInFocusActivityContext;

    public HtmlViewWrapper(Context context, boolean clearCache)
    {
        super(context);
        initView(clearCache);
    }

    public void setWidgetType(String widgetType)
    {
        mWidgetType = widgetType;
    }

    public void setStyleID(String styleID)
    {
        mStyleId = styleID;
    }

    public void setData(ContentItem contentItem)
    {
        mContentItem = contentItem;
        mHtmlPageUrl = contentItem.getData();

        setAndApplyExternalProperties(contentItem.getExtraParameters());
    }

    /**
     * Set current in focus activity context
     *
     * @param context
     */
    public void setInFocusActivityContext(Context context)
    {
        mInFocusActivityContext = context;
    }

    /**
     * Set webview properties
     */
    private void setAndApplyExternalProperties(JSONObject jsonObject)
    {
        if (jsonObject != null)
        {
            if (jsonObject.has("webview_properties") && jsonObject.isNull("webview_properties") == false)
            {
                JSONObject mwbViewPropertiesMap = jsonObject.optJSONObject("webview_properties");
                if (mwbViewPropertiesMap != null)
                {
                    mWebView.applyProperties(mwbViewPropertiesMap);
                }
            }

            if (jsonObject.has("customFunctions") && jsonObject.isNull("customFunctions") == false)
            {
                mCustomFunctionsMap = jsonObject.optJSONObject("customFunctions");
            }

            mShowMaximize = jsonObject.optBoolean("showMaximize", false);
            mIsShowClose = jsonObject.optBoolean("showClose", false);
            mIsAvoidAutoImpression = jsonObject.optBoolean("avoidAutoImpression", false);
        }
    }

    public void loadHtml()
    {
        loadHtml(mHtmlPageUrl);
    }

    /**
     * Start HTML Widget/Page loading
     */
    public void loadHtml(String link)
    {
        mHtmlPageUrl = link;
        if (mHtmlPageUrl != null)
        {
            StringBuilder stringBuilder = new StringBuilder(mHtmlPageUrl);

            if (!mHtmlPageUrl.contains("?"))
            {
                stringBuilder.append("?");
            } else
            {
                stringBuilder.append("&");
            }

            stringBuilder.append("publisher_id=" + KidozSDK.getPublisherID());
            stringBuilder.append("&package_id=" + getContext().getPackageName());
            stringBuilder.append("&sdk_version=" + ConstantDef.SDK_VERSION);
            stringBuilder.append("&os_version=" + Build.VERSION.SDK_INT);
            stringBuilder.append("&os_type=" + "android");

            String webviewVersion = "w";
            String useragent = Utils.getDefaultUserAgent();
            if (useragent != null && useragent.contains("Chrome"))
            {
                int index = useragent.indexOf("Chrome");
                String sub = useragent.substring(index, useragent.length());
                String[] res = sub.split(" ");
                if (res != null && res.length > 0)
                {
                    String[] tempVal = res[0].split("/");
                    if (tempVal != null && tempVal.length > 1)
                    {
                        webviewVersion = "c_" + tempVal[1];
                    }
                }
            }

            stringBuilder.append("&webview_version=" + webviewVersion);
            stringBuilder.append("&device_hash=" + Utils.generateUniqeDeviceID(getContext().getPackageName(), KidozSDK.getPublisherID()));
            stringBuilder.append("&widget_type=" + mWidgetType);
            stringBuilder.append("&auth_token=" + KidozSDK.getAuthToken());
            stringBuilder.append("&gid=" + DeviceUtils.getGoogleAdvertisingID(getContext()));

            mWebView.loadUrl(stringBuilder.toString());
        }
    }

    private void initView(boolean clearCache)
    {
        initHandler();
        initWebView(clearCache);

        initProgressLoadingView();
    }

    private void initHandler()
    {
        mEventHandler = new Utils.StaticHandler(Looper.getMainLooper())
        {
            @Override
            public void handleMessage(Message msg)
            {
                super.handleMessage(msg);

                switch (msg.what)
                {
                    case ON_LOAD_STARTED:
                    {
                        if (mHtmlWeViewListener != null)
                        {
                            mHtmlWeViewListener.onHtmlStartLoading();
                        }
                        break;
                    }

                    case ON_LOAD_FINISHED:
                    {
                        if (mHtmlWeViewListener != null)
                        {
                            mHtmlWeViewListener.onHtmlFinishedLoading();
                        }
                        break;
                    }

                    case ON_VIEW_READY:
                    {
                        if (mHtmlWeViewListener != null)
                        {
                            mHtmlWeViewListener.onViewReady();
                        }
                        break;
                    }

                    case ON_CLOSE:
                    {
                        if (mHtmlWeViewListener != null)
                        {
                            mHtmlWeViewListener.onClose();
                        }

                        WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.INTERSTITIAL_AD_CLOSE, WidgetType.WIDGET_TYPE_INTERSTITIAL);
                        EventBus.getDefault().post(eventMessage);
                        break;
                    }

                    case ON_RECEIVED_ERROR:
                    {
                        if (mHtmlWeViewListener != null)
                        {
                            mHtmlWeViewListener.onErrorReceived();
                        }
                        break;
                    }

                    case ON_NOTIFY_VAST_AD_READY:
                    {
                        if (mHtmlWeViewListener != null)
                        {
                            if (msg.arg1 == 1)
                            {
                                mHtmlWeViewListener.onNotifyVastReady(true);
                            } else
                            {
                                mHtmlWeViewListener.onNotifyVastReady(false);
                            }
                        }
                        break;
                    }
                }
            }
        };
    }

    private void initWebView(boolean clearCache)
    {
        mWebView = new HtmlFiveWebView(getContext());
        if (clearCache)
        {
            clearCache();
        }
        mHtmlAndroidWebInterface = initJavaScriptInterface();
        mWebView.addJavascriptInterface(mHtmlAndroidWebInterface, "KidozAndroid");
        mWebView.setWebViewClient(new WebViewClient()
        {
            @Override
            public void onPageStarted(WebView view, String url, Bitmap favicon)
            {
                super.onPageStarted(view, url, favicon);
                mEventHandler.sendEmptyMessage(ON_LOAD_STARTED);
            }

            @Override
            public void onPageFinished(WebView view, String url)
            {
                setVisibility(VISIBLE);
                super.onPageFinished(view, url);
                mEventHandler.sendEmptyMessage(ON_LOAD_FINISHED);
            }

            @Override
            public boolean shouldOverrideUrlLoading(WebView view, final String url)
            {
                boolean isForwardingToGoogle = false;
                if (url.contains("market://") || url.contains("play.google.com"))
                {
                    isForwardingToGoogle = true;
                } else if (url.startsWith("https://m.youtube") || url.startsWith("http://m.youtube") || url.startsWith("https://www.youtube") || url.startsWith("http://www.youtube"))
                {
                } else
                {
                    isForwardingToGoogle = true;
                }

                if (isForwardingToGoogle == true)
                {
                    Context context = mInFocusActivityContext;
                    if (context == null)
                    {
                        context = getContext();
                    }
                    ContentExecutionHandler.checkForParentalLock(context, new ContentExecutionHandler.IOnParentalLockStatusListener()
                    {
                        @Override
                        public void onLockActive(boolean isPasswordCorrect)
                        {
                            if (isPasswordCorrect == true)
                            {
                                openGooglePlayStore(url);
                            } else
                            {
                                EventManager.getInstance(getContext()).logEvent(getContext(), mWidgetType, mStyleId, EventManager.LOG_NORMAL_LEVEL, null, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_PARENTAL_INCORRECT_PASSWORD, mContentItem.getId());
                            }
                        }

                        @Override
                        public void onLockNotActive()
                        {
                            openGooglePlayStore(url);
                        }
                    });
                }
                return true;
            }
        });

        mWebView.setWebChromeClient(new WebChromeClient()
        {
            @Override
            public void onProgressChanged(WebView view, int newProgress)
            {
                super.onProgressChanged(view, newProgress);
            }
        });
        addView(mWebView, new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT));
    }

    private void openGooglePlayStore(String url)
    {
        try
        {
            if (mContentItem != null && mContentItem.getContentType() != null)
            {
                switch (mContentItem.getContentType())
                {
                    case PROMOTED_PLAY_APPLICATION:
                    {
                        openGooglePlayInBackground(url);
                        break;
                    }
                    case HTML:
                    {
                        if (mContentItem.getJSONitem() != null)
                        {
                            try
                            {
                                JSONArray jsonArray = mContentItem.getJSONitem();
                                JSONObject extraJsonObject = jsonArray.getJSONObject(11);
                                if (extraJsonObject.optBoolean("cpi_play", false) == true)
                                {
                                    openGooglePlayInBackground(url);
                                } else
                                {
                                    openGooglePlayInForground(url);
                                }
                            } catch (Exception ex)
                            {
                                openGooglePlayInForground(url);
                            }
                        } else
                        {
                            openGooglePlayInForground(url);
                        }
                        break;
                    }
                    default:
                    {
                        openGooglePlayInForground(url);
                    }
                }
            } else
            {
                openGooglePlayInForground(url);
            }
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to open google play for promoted app: \n" + ex.getMessage());
        }
    }

    private void openGooglePlayInForground(String url)
    {
        Intent intent = new Intent(Intent.ACTION_VIEW);
        intent.setData(Uri.parse(url));
        if (mInFocusActivityContext != null)
        {
            mHtmlWeViewListener.onClose();
            mInFocusActivityContext.startActivity(intent);
        }
    }

    private void openGooglePlayInBackground(String url)
    {
        Context context = mInFocusActivityContext;
        if (context == null)
        {
            context = getContext();
        }

        mHtmlWeViewListener.onClose();

        ContentItem contentItem = new ContentItem();
        contentItem.setContentType(ContentType.PROMOTED_PLAY_APPLICATION);
        contentItem.setData(url);
        contentItem.setId(mContentItem.getId());
        ContentExecutionHandler.handleContentItemClick(context, contentItem, mWidgetType, mStyleId, 0, false, null);
    }

    private void initProgressLoadingView()
    {
        mLoadingProgressView = new LoadingProgressView(getContext());

        Point point = Utils.getScreenSize(getContext());
        int size = (int) (Math.min(point.x, point.y) * LOADING_PROGRESS_DEFAULT_RATIO);
        RelativeLayout.LayoutParams prParams = new RelativeLayout.LayoutParams(size, size);
        prParams.addRule(RelativeLayout.CENTER_IN_PARENT);

        addView(mLoadingProgressView, prParams);
    }

    /**
     * Initiate java script interface that allows communication with the html
     */
    private HtmlAndroidJavaScriptInterface initJavaScriptInterface()
    {
        return new HtmlAndroidJavaScriptInterface(new AndroidJSLocalInterface()
        {
            @Override
            public void onViewReady()
            {
                mEventHandler.sendEmptyMessage(ON_VIEW_READY);
            }

            @Override
            public void onImpressionServed(final String item_id, final String item_name, final String ad_id, final String view_index)
            {
                //SDKLogger.printToast(getContext(), "JS : onImpressionServed");
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        int positionIndexAsInteger = 0;

                        try
                        {
                            positionIndexAsInteger = Integer.parseInt(view_index);
                        } catch (Exception ex)
                        {
                            SDKLogger.printErrorLog(TAG, "Error when trying to parse positionIndex: " + ex.getMessage());
                        }
                        EventManager.getInstance(getContext()).logSponsoredContentImpressionEvent(getContext(), mWidgetType, mStyleId, EventParameters.ACTION_IMPRESSION, item_name, ad_id, item_id, positionIndexAsInteger);
                    }
                });
            }

            @Override
            public void simulateClick(final String json, final int positionIndex)
            {
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        ContentItem item = null;
                        try
                        {
                            JSONArray jsonArray = new JSONArray(json);
                            if (jsonArray != null && jsonArray.length() > 1)
                            {
                                JSONArray keysArray = jsonArray.getJSONArray(0);
                                HashMap<String, Integer> keysMap = new LinkedHashMap<String, Integer>();
                                if (keysArray != null && keysArray.length() > 0)
                                {
                                    for (int i = 0; i < keysArray.length(); i++)
                                    {
                                        keysMap.put(keysArray.getString(i), i);
                                    }
                                }

                                if (keysMap.isEmpty() == false && jsonArray.getJSONArray(1) != null)
                                {
                                    item = new ContentItem(jsonArray.getJSONArray(1), keysMap);
                                }
                            }
                        } catch (JSONException e)
                        {
                        }

                        if (item != null)
                        {
                            if (mAllowClickHandling)
                            {
                                mAllowClickHandling = false;
                                if (mInFocusActivityContext != null)
                                {
                                    ContentExecutionHandler.handleContentItemClick(mInFocusActivityContext, item, mWidgetType, mStyleId, positionIndex, true, new ContentExecutionHandler.IOnHandleClickListener()
                                    {
                                        @Override
                                        public void onRestoreClick()
                                        {
                                            mAllowClickHandling = true;
                                        }
                                    });
                                }
                            }
                        }
                    }
                });
            }

            @Override
            public void onAboutClick()
            {
                //SDKLogger.printToast(getContext(), "JS : onAboutClick");
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        startAboutKidozDialog();
                    }
                });
            }

            @Override
            public void onParentalClick()
            {
                //SDKLogger.printToast(getContext(), "JS : onParentalClick");
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        startParentalLockDialog();
                    }
                });
            }

            @Override
            public void onCloseClick()
            {
                //SDKLogger.printToast(getContext(), "JS : onCloseClick");
                mEventHandler.sendEmptyMessage(ON_CLOSE);
            }

            @Override
            public void onMaximize()
            {
                //SDKLogger.printToast(getContext(), "JS : onMaximize");
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        if (mHtmlWeViewListener != null)
                        {
                            mHtmlWeViewListener.onOpenMaximized();
                        }
                    }
                });
            }

            @Override
            public void onInitWebViewWithProperties(final String properties)
            {
                //SDKLogger.printToast(getContext(), "JS : onUpdateWebViewProperties " + properties);
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        try
                        {
                            mWebView.applyProperties(new JSONObject(properties));
                        } catch (JSONException e)
                        {

                        }
                    }
                });
            }

            @Override
            public void onConversionServed(final String item_id, final String item_name, final String ad_id, final String view_index, final String rewardKey, final String rewardValue)
            {
                //SDKLogger.printToast(getContext(), "JS : onConversionServed");
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        int positionIndexAsInteger = 0;
                        int rewardKeyAsInteger = 0;
                        int rewardValueAsInteger = 0;

                        try
                        {
                            positionIndexAsInteger = Integer.parseInt(view_index);
                        } catch (Exception ex)
                        {
                            SDKLogger.printErrorLog(TAG, "Error when trying to parse positionIndex: " + ex.getMessage());
                        }
                        try
                        {
                            rewardKeyAsInteger = Integer.parseInt(rewardKey);
                        } catch (Exception ex)
                        {
                            SDKLogger.printErrorLog(TAG, "Error when trying to parse rewardKey: " + ex.getMessage());
                        }
                        try
                        {
                            rewardValueAsInteger = Integer.parseInt(rewardValue);
                        } catch (Exception ex)
                        {
                            SDKLogger.printErrorLog(TAG, "Error when trying to parse rewardValue: " + ex.getMessage());
                        }
                        Event event = new Event();
                        EventManager.getInstance(getContext()).logSponsoredConversionEvent(getContext(), mWidgetType, mStyleId, event, item_name, ad_id, item_id, positionIndexAsInteger, rewardKeyAsInteger, rewardValueAsInteger);
                    }
                });
            }

            @Override
            public void onStoreLocalParam(final String key, final String value, final String scope_type)
            {
                //SDKLogger.printToast(getContext(), "JS : onStoreLocalParam");
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        CookieManager.saveData(getContext(), mWidgetType, mHtmlPageUrl, key, value, StorageLife.valueOf(scope_type));
                    }
                });
            }

            @Override
            public void getLocalParam(final String function, final String key)
            {
                //SDKLogger.printToast(getContext(), "JS : getLocalParam : FucntionName: " + function + "  key: " + key);
                final String value = CookieManager.loadData(getContext(), mWidgetType, mHtmlPageUrl, key);

                String defaultName = "kidozReturnedValue";
                if (function != null)
                {
                    defaultName = function;
                }
                final String finalDefaultName = defaultName;
                mEventHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        invokeJSfunction("javascript:" + finalDefaultName + "('" + key + "','" + value + "'  );");
                    }
                });
            }

            @Override
            public void getDonePlayback()
            {
                //SDKLogger.printToast(getContext(), "JS : getDonePlayback");
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        if (mHtmlWeViewListener != null)
                        {
                            mHtmlWeViewListener.onDonePlayBack();
                        }
                    }
                });
            }

            @Override
            public void getParentalLockStatus(final String function)
            {
                //SDKLogger.printToast(getContext(), "JS : getParentalLockStatus : FucntionName: " + function);
                final boolean isLockActive = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), ParentalLockDialog.PARENTAL_LOCK_ACTIVE_KEY) == null ? false : true;

                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        invokeJSfunction("javascript:" + function + "('" + isLockActive + "'  );");
                    }
                });
            }

            @Override
            public void forwardToGooglePlay(final String advertiserID, final String appID, final String googlePlayLink, final String widgetType, final String styleID, final String itemIndex)
            {
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        int itemIndexAsInteger = 0;
                        try
                        {
                            itemIndexAsInteger = Integer.parseInt(itemIndex);
                        } catch (Exception ex)
                        {
                            SDKLogger.printErrorLog(TAG, "Error when trying to parse item index: " + ex.getMessage());
                        }
                        ContentItem contentItem = new ContentItem();
                        contentItem.setId(appID);
                        contentItem.setName("");
                        contentItem.setData(googlePlayLink);
                        contentItem.setContentType(ContentType.PROMOTED_PLAY_APPLICATION);
                        contentItem.setAdvertiserID(advertiserID);
                        contentItem.setIsPromoted(true);
                        if (mViewPagerItemClickListener != null)
                        {
                            mViewPagerItemClickListener.onClickEnd(contentItem, itemIndexAsInteger);
                        } else
                        {
                            if (mInFocusActivityContext != null)
                            {
                                ContentExecutionHandler.handleContentItemClick(mInFocusActivityContext, contentItem, widgetType, styleID, itemIndexAsInteger, false, null);
                            }
                        }
                    }
                });
            }

            @Override
            public void sendEvent(final String type, final String category, final String action, final String label, final String jsonData)
            {
                mExecutionHandler.post(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        SDKLogger.printDebbugLog("RONY", "type = " + type);
                        SDKLogger.printDebbugLog("RONY", "category = " + category);
                        SDKLogger.printDebbugLog("RONY", "action = " + action);
                        SDKLogger.printDebbugLog("RONY", "label = " + label);
                        SDKLogger.printDebbugLog("RONY", "jsonData = " + jsonData);
                        SDKLogger.printDebbugLog("RONY", "mStyleId = " + mStyleId);
                        SDKLogger.printDebbugLog("RONY", "mWidgetType = " + mWidgetType);

                        Event event = new Event();
                        event.setLogLevel(EventManager.LOG_NORMAL_LEVEL);
                        event.addParameterToJsonObject(jsonData);
                        EventManager.getInstance(getContext()).logEvent(getContext(), mWidgetType, mStyleId, EventManager.LOG_NORMAL_LEVEL, event, category, action, label);
                    }
                });
            }

            @Override
            public void toggleLoadingState(final boolean isLoading)
            {
                if (mLoadingProgressView != null)
                {
                    mExecutionHandler.post(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            if (isLoading == true)
                            {
                                mLoadingProgressView.startLoadingAnimation();
                            } else
                            {
                                mLoadingProgressView.stopLoadingAnimation();
                            }
                        }
                    });
                }
            }

            @Override
            public void notifyIsVastAdReady(final boolean isVastReady)
            {
                Message message = Message.obtain();
                message.what = ON_NOTIFY_VAST_AD_READY;
                if (isVastReady)
                {
                    message.arg1 = 1;
                } else
                {
                    message.arg1 = 0;
                }
                mEventHandler.sendMessage(message);
            }
        });
    }

    /**
     * Set parental lock state, update html icon view
     */
    private void setParentalLockState(boolean locked)
    {
        invokeJSfunction("javascript:kidozOnParentalLockStateChanged('" + locked + "');");
    }

    /**
     * Start parental lock dialog
     */
    private void startParentalLockDialog()
    {
        int[] popLocation = new int[2];
        popLocation[0] = (int) (ScreenUtils.getScreenSize(getContext(), true) * 0.5f);
        popLocation[1] = (int) (ScreenUtils.getScreenSize(getContext(), false) * 0.5f);
        if (mInFocusActivityContext != null)
        {
            ParentalLockDialog parentalLockDialog = new ParentalLockDialog(mInFocusActivityContext, false, popLocation);
            parentalLockDialog.setOnDismissListener(new DialogInterface.OnDismissListener()
            {
                @Override
                public void onDismiss(DialogInterface dialog)
                {
                    //Refresh the parental lock icon
                    updateLockIcon();
                }
            });
            parentalLockDialog.setParentLockDialogListener(new ParentalLockDialog.ParentLockDialogListener()
            {
                @Override
                public void onPasswordEntered(boolean isPasswordCorrect)
                {
                    updateLockIcon();
                }
            });
            parentalLockDialog.show();
            updateLockIcon();
        }
    }

    /**
     * Start about kidoz dialog
     */
    private void startAboutKidozDialog()
    {
        int[] popLocation = new int[2];
        popLocation[0] = (int) (ScreenUtils.getScreenSize(getContext(), true) * 0.5f);
        popLocation[1] = (int) (ScreenUtils.getScreenSize(getContext(), false) * 0.5f);
        if (mInFocusActivityContext != null)
        {
            AboutKidozDialog aboutKidozDialog = new AboutKidozDialog(mInFocusActivityContext, popLocation);
            aboutKidozDialog.openDialog();
        }
    }

    /**
     * Update lock icon at HTML view depending on current lock state
     */
    private void updateLockIcon()
    {
        try
        {
            boolean isLockActive = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), ParentalLockDialog.PARENTAL_LOCK_ACTIVE_KEY) == null ? false : true;
            if (isLockActive == true)
            {
                setParentalLockState(true);
            } else
            {
                setParentalLockState(false);
            }
        } catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to load parental lock image: " + ex.getMessage());
        }
    }

    /**
     * Request focus ON item
     */
    public void requestFocusOn(boolean isFullScreen)
    {
        if (mContentItem != null)
        {
            String json = mContentItem.getJSONitem().toString();
            String func = "javascript:kidozOnFocusOn('" + mContentItem.getId() + "','" + json + "','" + isFullScreen + "','" + mWidgetType + "','" + mContentItem.getRealViewIndex() + "');";
            if (mCustomFunctionsMap != null && mCustomFunctionsMap.has(JS_FUNC_ON_FOCUS_ON_KEY))
            {
                func = mCustomFunctionsMap.optString(JS_FUNC_ON_FOCUS_ON_KEY, func);
            }

            invokeJSfunction(func);
        }
    }

    /**
     * Request for prepare content item to get ready for focus on
     */
    public void requestPrepareFocusOn(boolean isFullScreen)
    {
        if (mContentItem != null)
        {
            String json = mContentItem.getJSONitem().toString();
            String func = "javascript:kidozOnPrepareFocusOn('" + mContentItem.getId() + "','" + json + "','" + isFullScreen + "','" + mWidgetType + "','" + mContentItem.getRealViewIndex() + "');";
            if (mCustomFunctionsMap != null && mCustomFunctionsMap.has(JS_FUNC_ON_PREPARE_FOCUS_ON_KEY))
            {
                func = mCustomFunctionsMap.optString(JS_FUNC_ON_PREPARE_FOCUS_ON_KEY, func);
            }

            invokeJSfunction(func);
        }
    }

    /**
     * Request focus ON item
     */
    public void requestFocusOff()
    {
        if (mContentItem != null)
        {
            String func = "javascript:kidozOnFocusOff('" + mContentItem.getId() + "');";
            if (mCustomFunctionsMap != null && mCustomFunctionsMap.has(JS_FUNC_ON_FOCUS_OFF_KEY))
            {
                func = mCustomFunctionsMap.optString(JS_FUNC_ON_FOCUS_OFF_KEY, func);
            }
            invokeJSfunction(func);
        }
    }

    /**
     * Make <b>Maximize</b> request to an html view
     */
    public void requestMaximize()
    {
        String func = "javascript:kidozOnMaximize();";

        if (mCustomFunctionsMap.has(JS_FUNC_ON_MAXIMIZE_KEY))
        {
            func = mCustomFunctionsMap.optString(JS_FUNC_ON_MAXIMIZE_KEY, func);
        }

        invokeJSfunction(func);
    }

    /**
     * Make <b>Minimize</b> request to an html view
     */
    public void requestMinimize()
    {
        String func = "javascript:kidozOnMinimize();";

        if (mCustomFunctionsMap.has(JS_FUNC_ON_MINIMIZE_KEY))
        {
            func = mCustomFunctionsMap.optString(JS_FUNC_ON_MINIMIZE_KEY, func);
        }

        invokeJSfunction(func);
    }

    /**
     * Request widget open
     */
    public void requestWidgetOpen(String widgetType)
    {
        if (widgetType != null)
        {
            invokeJSfunction("javascript:kidozOnWidgetOpen('" + widgetType + "');");
        }
    }

    /**
     * Request widget close
     */
    public void requestWidgetClose(String widgetType)
    {
        if (widgetType != null)
        {
            invokeJSfunction("javascript:kidozOnWidgetClose('" + widgetType + "');");
        }
    }

    /**
     * Request vast ads
     */
    public void requestVastAds()
    {

        invokeJSfunction("javascript:requestAds();");
    }

    /**
     * Start vast ad playback
     */
    public void startVastAd()
    {

        invokeJSfunction("javascript:startVastAd();");
    }

    /**
     * Stop vast ad playback
     */
    public void stopVastAd()
    {

        invokeJSfunction("javascript:stopVastAd();");
    }

    /**
     * Pause vast ad playback
     */
    public void pauseVastAd()
    {

        invokeJSfunction("javascript:pauseVastAd();");
    }

    /**
     * Pause vast ad playback
     */
    public void resumeVastAd()
    {

        invokeJSfunction("javascript:resumeVastAd();");
    }

    /**
     * Clear the content of HTML view
     */
    public void clearHtmlView()
    {
        mWebView.loadUrl("");
        mWebView.stopLoading();
    }

    // Android javascript interface connector
    class HtmlAndroidJavaScriptInterface
    {
        private AndroidJSLocalInterface mScriptInterface;

        /**
         * Instantiate the interface and set the context
         */
        HtmlAndroidJavaScriptInterface(AndroidJSLocalInterface scriptInterface)
        {
            mScriptInterface = scriptInterface;
        }

        @android.webkit.JavascriptInterface
        public void onViewReady()
        {
            mScriptInterface.onViewReady();
        }

        @android.webkit.JavascriptInterface
        public void simulateClick(String contentItemJson, int positionIndex)
        {
            try
            {
                mScriptInterface.simulateClick(contentItemJson, positionIndex);
            } catch (Exception e)
            {
            }
        }

        @android.webkit.JavascriptInterface
        public void onSendImpressionEvent(String item_id, String item_name, String ad_id, String view_index)
        {
            try
            {
                mScriptInterface.onImpressionServed(item_id, item_name, ad_id, view_index);
            } catch (Exception e)
            {
            }
        }

        @android.webkit.JavascriptInterface
        public void onInvokeAboutClick()
        {
            mScriptInterface.onAboutClick();
        }

        @android.webkit.JavascriptInterface
        public void onInvokeParentalClick()
        {
            mScriptInterface.onParentalClick();
        }

        @android.webkit.JavascriptInterface
        public void onInvokeCloseClick()
        {
            mScriptInterface.onCloseClick();
        }

        @android.webkit.JavascriptInterface
        public void onInvokeMaximize()
        {
            mScriptInterface.onMaximize();
        }

        @android.webkit.JavascriptInterface
        public void onInitWebViewWithProperties(String properties)
        {
            try
            {
                mScriptInterface.onInitWebViewWithProperties(properties);
            } catch (Exception e)
            {
            }
        }

        @android.webkit.JavascriptInterface
        public void onSendConversionEvent(String item_id, String item_name, String ad_id, String view_index, String rewardKey, String rewardValue)
        {
            try
            {
                mScriptInterface.onConversionServed(item_id, item_name, ad_id, view_index, rewardKey, rewardValue);
            } catch (Exception e)
            {
            }
        }

        @android.webkit.JavascriptInterface
        public void onStoreLocalParameter(String key, String value, String scope_type)
        {
            try
            {
                mScriptInterface.onStoreLocalParam(key, value, scope_type);
            } catch (Exception e)
            {
            }
        }

        @android.webkit.JavascriptInterface
        public void getLocalParameter(final String function, final String key)
        {
            try
            {
                mScriptInterface.getLocalParam(function, key);
            } catch (Exception e)
            {
            }
        }

        @android.webkit.JavascriptInterface
        public void onDone()
        {
            try
            {
                mScriptInterface.getDonePlayback();
            } catch (Exception e)
            {
            }
        }

        @android.webkit.JavascriptInterface
        public void getParentalLockStatus(String function)
        {
            try
            {
                mScriptInterface.getParentalLockStatus(function);
            } catch (Exception e)
            {
            }
        }

        @android.webkit.JavascriptInterface
        public void forwardToGooglePlay(final String advertiserID, final String appID, final String googlePlayLink, final String widjetType, final String styleID, final String itemIndex)
        {
            mScriptInterface.forwardToGooglePlay(advertiserID, appID, googlePlayLink, widjetType, styleID, itemIndex);
        }

        @android.webkit.JavascriptInterface
        public void sendEvent(String type, String category, String action, String label, String jsonData)
        {
            mScriptInterface.sendEvent(type, category, action, label, jsonData);
        }

        @android.webkit.JavascriptInterface
        public void toggleLoadingState(String isLoading)
        {
            try
            {
                boolean isLoadingAsBoolean = Boolean.parseBoolean(isLoading);
                mScriptInterface.toggleLoadingState(isLoadingAsBoolean);
            } catch (Exception ex)
            {
                SDKLogger.printErrorLog(TAG, "Error when trying to parse isLoading parameter: " + ex.getMessage());
            }
        }

        @android.webkit.JavascriptInterface
        public void notifyIsVastAdReady(boolean isVastReady)
        {
            mScriptInterface.notifyIsVastAdReady(isVastReady);
        }
    }

    private interface AndroidJSLocalInterface
    {
        public void onViewReady();

        public void simulateClick(String contentItemJson, int positionIndex);

        public void onImpressionServed(String item_id, String item_name, String ad_id, String view_index);

        public void onAboutClick();

        public void onParentalClick();

        public void onCloseClick();

        public void onMaximize();

        public void onInitWebViewWithProperties(String properties);

        public void onConversionServed(String item_id, String item_name, String ad_id, String view_index, String rewardKey, String rewardValue);

        public void onStoreLocalParam(String key, String value, String scope_type);

        public void getLocalParam(String function, String key);

        public void getDonePlayback();

        public void getParentalLockStatus(String function);

        public void forwardToGooglePlay(String advertiserID, String appID, String googlePlayLink, String widgetType, String styleID, String itemIndex);

        public void sendEvent(String type, String category, String action, String label, String jsonData);

        public void toggleLoadingState(boolean isLoading);

        public void notifyIsVastAdReady(boolean isVastReady);
    }

    public void setHtmlWebViewListener(IOnHtmlWebViewInterface htmlWeViewListener)
    {
        mHtmlWeViewListener = htmlWeViewListener;
    }

    public void invokeJSfunction(final String function)
    {
        try
        {
            mExecutionHandler.post(new Runnable()
            {
                @Override
                public void run()
                {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT)
                    {
                        mWebView.evaluateJavascript(function, null);
                    } else
                    {
                        mWebView.loadUrl(function);
                    }
                }
            });
        } catch (Exception ex)
        {
        }
    }

    public void setOnSwipeGestureListener(OnSwipeTouchListener onSwipeGEstureListener)
    {
        mWebView.setOnTouchListener(onSwipeGEstureListener);
    }

    /**
     * Get inner html view
     */
    public HtmlFiveWebView getInnerWebView()
    {
        return mWebView;
    }

    /**
     * Is avoid to make auto impression
     */
    public boolean isAvoidAutoImpression()
    {
        return mIsAvoidAutoImpression;
    }

    /**
     * Is show close button
     */
    public boolean isShowClose()
    {
        return mIsShowClose;
    }

    /**
     * Show loading circle progress bar view
     */
    public void showLoadingProgressView()
    {
        if (mLoadingProgressView != null)
        {
            mLoadingProgressView.startLoadingAnimation();
        }
    }

    /**
     * Hide loading circle progress bar view
     */
    public void hideLoadingProgressView()
    {
        if (mLoadingProgressView != null)
        {
            mLoadingProgressView.stopLoadingAnimation();
        }
    }

    /**
     * Set webview alpha value
     */
    public void setWebViewAlpha(float alpha)
    {
        mWebView.setAlpha(alpha);
    }

    /**
     * Stop and release webview instance
     */
    public void stopAndReleaseWebView()
    {
        if (mWebView != null)
        {
            mWebView.stopAndReleaseWebView();
        }
    }

    /**
     * Stop loading content
     */
    public void stopLoading()
    {
        if (mWebView != null)
        {
            mWebView.stopLoading();
        }
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh)
    {
        super.onSizeChanged(w, h, oldw, oldh);

        int size = (int) (Math.min(w, h) * LOADING_PROGRESS_DEFAULT_RATIO);
        mLoadingProgressView.getLayoutParams().height = size;
        mLoadingProgressView.getLayoutParams().width = size;
        mLoadingProgressView.setCircleWidthRelativeToSize(w);
    }

    public void setViewPagerItemClickListener(ItemViewPagerAdapter.ViewPagerItemClickListener viewPagerItemClickListener)
    {
        mViewPagerItemClickListener = viewPagerItemClickListener;
    }

    public ContentItem getInnerContentData()
    {
        return mContentItem;
    }

    public void clearCache()
    {
        if (mWebView != null)
        {
            mWebView.clearCache(true);
            mWebView.clearHistory();
        }
    }
}
