package com.kidoz.sdk.api.players.web_player;

import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.graphics.Color;
import android.graphics.Point;
import android.os.Bundle;
import android.os.Looper;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.BaseDialog;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.WidgetEventMessage;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.KidozTopBar;
import com.kidoz.sdk.api.ui_views.html_view.HtmlViewWrapper;
import com.kidoz.sdk.api.ui_views.html_view.IOnHtmlWebViewInterface;

import org.greenrobot.eventbus.EventBus;

public class WebPlayerDialog extends BaseDialog
{
    private static final int DEFAULT_TOP_BAR_COLOR = Color.parseColor("#36a7ec");

    private static final String TAG = WebPlayerDialog.class.getSimpleName();
    private static final int WEB_PLAYER_RESOURCE_ID = 1;

    private static final String CONTENT_ITEM_SAVE_STATE_KEY = "CONTENT_ITEM_SAVE_STATE_KEY_WP";
    private static final String WEB_PLAYER_TYPE_SAVE_STATE_KEY = "WEB_PLAYER_TYPE_SAVE_STATE_KEY_WP";
    private static final String ORIENTATION_STATE_SAVE_STATE_KEY = "ORIENTATION_STATE_SAVE_STATE_KEY_WP";
    private static final String ORIENTATION_MODIFY_SAVE_STATE_KEY = "ORIENTATION_MODIFY_SAVE_STATE_KEY_WP";

    private RelativeLayout mInnerContainer;
    private ContentItem mContentItem;
    private WebPlayerType mWebPlayerType;
    private KidozTopBar mKidozTopBar;

    private WebPlayerView mWebPlayerView;
    private HtmlViewWrapper mHtmlViewWrapper;

    private boolean mAllowOrientationModify = false;

    private String mWidgetType = null;
    private String mStyleId = null;
    private Utils.StaticHandler staticHandler = new Utils.StaticHandler(Looper.getMainLooper());

    public WebPlayerDialog(Context context, ContentItem contentItem, String widgetType, String styleId, WebPlayerType webPlayerType, boolean allowOrientationModification)
    {
        super(context, android.R.style.Theme_Translucent_NoTitleBar_Fullscreen);
        setCancelable(true);

        mContentItem = contentItem;
        mWebPlayerType = webPlayerType;
        mAllowOrientationModify = allowOrientationModification;
        this.mWidgetType = widgetType;
        mStyleId = styleId;
    }

    /**
     * Set timer to auto close this dialog
     */
    public void setAutoCloseTimer(int millis)
    {
        staticHandler.removeCallbacksAndMessages(null);
        staticHandler.postDelayed(new Runnable()
        {
            @Override
            public void run()
            {
                if (this != null)
                {
                    closeDialog();
                }
            }
        }, millis);
    }

    @Override
    protected void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        if (savedInstanceState != null)
        {
            mAllowOrientationModify = savedInstanceState.getBoolean(ORIENTATION_MODIFY_SAVE_STATE_KEY);
        }

        if (mAllowOrientationModify)
        {
            if (savedInstanceState != null && savedInstanceState.containsKey(ORIENTATION_STATE_SAVE_STATE_KEY))
            {
                mCurrentOrientationState = savedInstanceState.getInt(ORIENTATION_STATE_SAVE_STATE_KEY);
            }
            else
            {
                if (mCallingActivity != null && mCallingActivity.isFinishing() == false)
                {
                    mCurrentOrientationState = mCallingActivity.getResources().getConfiguration().orientation;
                    mCallingActivity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR);
                }
            }
        }

        if (savedInstanceState != null)
        {
            if (savedInstanceState.containsKey(CONTENT_ITEM_SAVE_STATE_KEY))
            {
                mContentItem = (ContentItem) savedInstanceState.getSerializable(CONTENT_ITEM_SAVE_STATE_KEY);
            }
            mWebPlayerType = (WebPlayerType) savedInstanceState.getSerializable(WEB_PLAYER_TYPE_SAVE_STATE_KEY);
        }

        setOnDismissListener(new OnDismissListener()
        {
            @Override
            public void onDismiss(DialogInterface dialog)
            {
            }
        });

        initDialogView();
    }

    private void initDialogView()
    {
        mInnerContainer = new RelativeLayout(getContext());
        setContentView(mInnerContainer, new RelativeLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT));

        if (mWebPlayerType == WebPlayerType.HTML_PLAYBACK)
        {
            addHtmlPlaybackView();
        }
        else
        {
            addWebPlayerView();
        }

        initTopBar();

        if (mWebPlayerType != null && mWebPlayerType == WebPlayerType.EXTERNAL_LINK)
        {
            mInnerContainer.setBackgroundColor(Color.parseColor("#77000000"));
            mKidozTopBar.setVisibility(View.INVISIBLE);
        }
        else
        {
            mInnerContainer.setBackgroundColor(Color.WHITE);
        }

        if (mContentItem.getName() != null)
        {
            mKidozTopBar.setTitle(mContentItem.getName());
        }
        else
        {
            mKidozTopBar.setTitle("");
        }
    }

    private void addHtmlPlaybackView()
    {
        mHtmlViewWrapper = new HtmlViewWrapper(mCallingActivity, false);
        mHtmlViewWrapper.setInFocusActivityContext(mCallingActivity);
        mHtmlViewWrapper.setHtmlWebViewListener(new IOnHtmlWebViewInterface()
        {
            @Override
            public void onHtmlStartLoading()
            {
                super.onHtmlStartLoading();
                mHtmlViewWrapper.showLoadingProgressView();
            }

            @Override
            public void onHtmlFinishedLoading()
            {
                super.onHtmlFinishedLoading();
                mHtmlViewWrapper.hideLoadingProgressView();
            }

            @Override
            public void onClose()
            {
                super.onClose();
                closeDialog();
            }
        });

        mHtmlViewWrapper.setData(mContentItem);
        mHtmlViewWrapper.loadHtml();

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT);
        Point point = Utils.getScreenSize(getContext());
        int paddingTop = (int) (Math.max(point.x, point.y) * KidozTopBar.BACKGROUND_RATIO);
        params.topMargin = paddingTop;
        mInnerContainer.addView(mHtmlViewWrapper, params);
    }

    private void addWebPlayerView()
    {
        mWebPlayerView = new WebPlayerView(mCallingActivity, mWebPlayerType, mWidgetType);
        if (mContentItem != null)
        {
            mWebPlayerView.loadWebUrl(mContentItem.getData(), mContentItem.getId());
        }

        mWebPlayerView.setOnRemoveViewRequestListener(new WebPlayerView.IOnRemoveViewRequestListener()
        {
            @Override
            public void onRemoveView()
            {
                closeDialog();
            }
        });

        mInnerContainer.addView(mWebPlayerView);
    }

    private void initTopBar()
    {
        //Create top header view
        mKidozTopBar = new KidozTopBar(getContext(), true, mWidgetType, null);
        mKidozTopBar.setId(Utils.generateViewId());
        mKidozTopBar.setTopBarBackgroundColor(DEFAULT_TOP_BAR_COLOR);
        mKidozTopBar.setKidozTopBarListener(new KidozTopBar.KidozTopBarListener()
        {
            @Override
            public void onExitClicked(View view)
            {
                closeDialog();
            }
        });
        mKidozTopBar.showParentalLockIcon(false);

        // Add header lay mo view
        mInnerContainer.addView(mKidozTopBar, new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.WRAP_CONTENT));
    }

    @Override
    protected void onStart()
    {
        super.onStart();
    }

    @Override
    protected void onStop()
    {
        super.onStop();
    }

    @Override
    public Bundle onSaveInstanceState()
    {
        Bundle bundle = super.onSaveInstanceState();
        if (bundle != null)
        {
            bundle.putSerializable(WEB_PLAYER_TYPE_SAVE_STATE_KEY, mWebPlayerType);
            bundle.putSerializable(ORIENTATION_STATE_SAVE_STATE_KEY, mCurrentOrientationState);
            bundle.putBoolean(ORIENTATION_MODIFY_SAVE_STATE_KEY, mAllowOrientationModify);
        }
        return bundle;
    }

    @Override
    public void closeDialog()
    {
        super.closeDialog();
    }

    @Override
    public void dismiss()
    {
        super.dismiss();
        if (mWebPlayerType != null)
        {
            if (mWebPlayerType == WebPlayerType.ONLINE_GAME || mWebPlayerType == WebPlayerType.WEB_BROWSER || mWebPlayerType == WebPlayerType.HTML_PLAYBACK)
            {
                WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.PLAYER_CLOSE, WidgetType.createFromString(mWidgetType));
                EventBus.getDefault().post(eventMessage);
            }
        }
    }

    @Override
    public void openDialog()
    {
        super.openDialog();
        String action = "";
        switch (mWebPlayerType)
        {
            case EXTERNAL_LINK:
                action = EventParameters.ACTION_EXTERNAL_LINK_VIEW;
                break;
            case ONLINE_GAME:
            {
                WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.PLAYER_OPEN, WidgetType.createFromString(mWidgetType));
                EventBus.getDefault().post(eventMessage);
                action = EventParameters.ACTION_GAME_VIEW;
                break;
            }
            case WEB_BROWSER:
            {
                WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.PLAYER_OPEN, WidgetType.createFromString(mWidgetType));
                EventBus.getDefault().post(eventMessage);
                action = EventParameters.ACTION_WEBSITE_VIEW;
                break;
            }

            case HTML_PLAYBACK:
            {
                WidgetEventMessage eventMessage = new WidgetEventMessage(EventMessage.MessageType.PLAYER_OPEN, WidgetType.createFromString(mWidgetType));
                EventBus.getDefault().post(eventMessage);
                action = EventParameters.ACTION_HTML_VIEW;
                break;
            }
        }

        EventManager.getInstance(getContext()).logEventWithDuration(this.getContext(), mWidgetType, mStyleId, EventManager.LOG_NORMAL_LEVEL, WEB_PLAYER_RESOURCE_ID, EventManager.LOG_CLOSE_RESOURCE_LEVEL, EventParameters.CATEGORY_WEB_PLAYER, action, mContentItem.getName(), mContentItem.getId(), false);
    }

    @Override
    public void onDetachedFromWindow()
    {
        super.onDetachedFromWindow();
        staticHandler.removeCallbacksAndMessages(null);
        if (mWebPlayerView != null)
        {
            try
            {
                mWebPlayerView.releaseAndcloseView();
            }
            catch (Exception ex)
            {
                SDKLogger.printDebbugLog(TAG, "Error when trying to onDetach: " + ex.getMessage());
            }
        }
        if (mHtmlViewWrapper != null)
        {
            try
            {
                mHtmlViewWrapper.stopAndReleaseWebView();
            }
            catch (Exception ex)
            {
                SDKLogger.printDebbugLog(TAG, "Error when trying to mHtmlViewWrapper.stopAndReleaseWebView(): " + ex.getMessage());
            }
        }


        // Restore previous orientation
        if (mAllowOrientationModify)
        {
            restoreUserOriginalOrientation();
        }

        EventManager.getInstance(getContext()).logEventWithDurationEnd(getContext(), mWidgetType, mStyleId, WEB_PLAYER_RESOURCE_ID, false);
    }

    @Override
    protected void onDialogVisibilityChange(int isVisible)
    {
        super.onDialogVisibilityChange(isVisible);

        if (isVisible != View.VISIBLE)
        {
            staticHandler.removeCallbacksAndMessages(null);
        }
    }

    @Override
    protected void onDialogWindowVisibilityChange(boolean visible)
    {
        super.onDialogWindowVisibilityChange(visible);

        if (visible == false)
        {
            staticHandler.removeCallbacksAndMessages(null);
        }
    }
}
