package com.kidoz.sdk.api.ui_views;

import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Point;
import android.support.v7.widget.GridLayoutManager;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.support.v7.widget.StaggeredGridLayoutManager;
import android.util.AttributeSet;
import android.view.View;
import android.widget.RelativeLayout;

import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentData;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.loading_progress_view.LoadingProgressView;

import java.util.ArrayList;


/**
 * Created by Eugine on 15/07/2015.
 * A Feed panel implementation view
 */

public class ContentPanelView extends AbstractContentPanel {

    private static final float LOADING_PROGRESS_DEFAULT_RATIO = 0.25f;

    /**
     * Feed panel layout view types
     */
    public enum LayoutViewType {
        STAGGERED, GRID, LINEAR_VERTICAL, LINEAR_HORIZONTAL
    }

    private RecyclerView mCustomRecyclerView;
    private RecycleEndlessScrollListener endlesScollPaginghandler;
    private ItemVerticalRecycleViewAdapter mAdapter;

    private StaggeredGridLayoutManager mStaggaredLayoutManager;
    private GridLayoutManager mCustomGridLayoutManager;

    private LoadingProgressView mLoadingProgressView;

    //private SwipeRefreshLayout swipeRefreshLayout;
    private LayoutViewType mLayoutViewType = LayoutViewType.STAGGERED;
    private ArrayList<ContentItem> mItemsList = new ArrayList<ContentItem>();

    private int[] fPos = new int[3];
    private int[] lPos = new int[3];
    private int mCurrentVisiblePosition = 0;

    public ContentPanelView(Context context, LayoutViewType layoutViewType) {
        super(context);
        mLayoutViewType = layoutViewType;
        initPanelView(context);
    }

    public ContentPanelView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initPanelView(context);
    }

    public ContentPanelView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initPanelView(context);
    }

    public void initPanelView(Context context) {
        // initSwipeRefreshLayout(context);
        initRecyclerView(context);
        initAdapter();
        initLayoutManager();
        initProgressBar();
    }

    private void initProgressBar() {

        mLoadingProgressView = new LoadingProgressView(getContext());
        mLoadingProgressView.setVisibility(View.GONE);

        Point point = Utils.getScreenSize(getContext());
        int size = (int) (Math.min(point.x, point.y) * LOADING_PROGRESS_DEFAULT_RATIO);
        RelativeLayout.LayoutParams prParams = new RelativeLayout.LayoutParams(size, size);
        prParams.addRule(RelativeLayout.CENTER_IN_PARENT);

        addView(mLoadingProgressView, prParams);
    }

   /* private void initSwipeRefreshLayout(Context context)
    {
        swipeRefreshLayout = new SwipeRefreshLayout(context);
        swipeRefreshLayout.setOnRefreshListener(null);
        swipeRefreshLayout.setEnabled(false);
        swipeRefreshLayout.setColorSchemeColors(Color.RED, Color.GRAY, Color.CYAN);
    }*/

    private void initRecyclerView(Context context) {
        mCustomRecyclerView = new RecyclerView(context);
        mCustomRecyclerView.setClipToPadding(false);
        addView(mCustomRecyclerView, new RelativeLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
    }

    public RecyclerView getInnerRecyclerView() {
        return mCustomRecyclerView;
    }

    private void initLayoutManager() {
        setLayoutManagerView(mLayoutViewType);
    }

    private void initAdapter() {
        mAdapter = new ItemVerticalRecycleViewAdapter(getContext(), mItemsList, getCurrentSpanCount());

        endlesScollPaginghandler = new RecycleEndlessScrollListener(4) {
            @Override
            public void onLoadMore(int page, int totalItemsCount) {
                if (totalItemsCount > 0) {
                   /* FeedItem feedItem = mAdapter.getLastItemInList();
                    if (feedItem != null)
                    {
                        if (mFeedWallFragmentHelper != null)
                        {
                            mFeedWallFragmentHelper.loadSomeMoreKidFeedsFromServer(feedItem.getFeedItemCreateDateTimeStamp());
                        }
                    }*/
                }
            }
        };

        mCustomRecyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {

            @Override
            public void onScrolled(RecyclerView recyclerView, int dx, int dy) {
                super.onScrolled(recyclerView, dx, dy);
                mAdapter.setIsRegularAnimation(true);
                try {
                    if (mLayoutViewType == LayoutViewType.GRID || mLayoutViewType == LayoutViewType.LINEAR_HORIZONTAL || mLayoutViewType == LayoutViewType.LINEAR_VERTICAL) {
                        fPos[0] = mCustomGridLayoutManager.findFirstVisibleItemPosition();
                        lPos[0] = mCustomGridLayoutManager.findLastVisibleItemPosition();

                        endlesScollPaginghandler.onScroll(fPos[0], lPos[0] - fPos[0], mItemsList.size());

                    } else if (mLayoutViewType == LayoutViewType.STAGGERED) {
                        mStaggaredLayoutManager.findFirstVisibleItemPositions(fPos);
                        mStaggaredLayoutManager.findLastVisibleItemPositions(lPos);

                        endlesScollPaginghandler.onScroll(fPos[0], lPos[0] - fPos[0], mItemsList.size());
                    }
                } catch (Exception e) {
                }

                mCurrentVisiblePosition = fPos[0];


            }

            @Override
            public void onScrollStateChanged(RecyclerView customRecyclerView, int newState) {
                super.onScrollStateChanged(customRecyclerView, newState);
            }
        });

        mCustomRecyclerView.setAdapter(mAdapter);
        if (mCurrentVisiblePosition != 0) {
            mCustomRecyclerView.scrollToPosition(mCurrentVisiblePosition);
        }
    }

    /**
     * Get current spann account calculated by the device type and orientation
     */
    private int getCurrentSpanCount() {
        return getNumberOfColumns(getContext());
    }

    /**
     * Set choosen layout view type
     *
     * @param viewType layout view type
     */
    public void setLayoutManagerView(LayoutViewType viewType) {
        mLayoutViewType = viewType;
        switch (viewType) {
            case STAGGERED: {
                mStaggaredLayoutManager = new StaggeredGridLayoutManager(getCurrentSpanCount(), StaggeredGridLayoutManager.VERTICAL);
                mCustomRecyclerView.setLayoutManager(mStaggaredLayoutManager);
                break;
            }

            case GRID: {
                mCustomGridLayoutManager = new GridLayoutManager(getContext(), getCurrentSpanCount());
                mCustomRecyclerView.setLayoutManager(mCustomGridLayoutManager);
                break;
            }

            case LINEAR_HORIZONTAL:
                LinearLayoutManager mLinearLayoutManager;
            {
                mLinearLayoutManager = new LinearLayoutManager(getContext(), LinearLayoutManager.HORIZONTAL, false);
                mCustomRecyclerView.setLayoutManager(mLinearLayoutManager);
                break;
            }

            case LINEAR_VERTICAL: {
                mLinearLayoutManager = new LinearLayoutManager(getContext(), LinearLayoutManager.VERTICAL, false);
                mCustomRecyclerView.setLayoutManager(mLinearLayoutManager);
                break;
            }
        }
    }

    /**
     * get number of columns depending on device type
     *
     * @return correct number of columns
     */
    private int getNumberOfColumns(Context context) {
        int result = 2;
        if (context.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            if (com.kidoz.sdk.api.general.utils.Utils.getIsDeviceTablet(context) == true) {
                // Landscape and Tablet
                result = 3;
            } else {
                // Landscape and Phone
                result = 3;
            }
        } else {
            if (com.kidoz.sdk.api.general.utils.Utils.getIsDeviceTablet(context) == true) {
                // Portrait and Tablet
                result = 2;
            } else {
                // Portrait and Phone
                result = 2;
            }
        }

        return result;
    }

    /**
     * Set rounding corners size in pixels
     *
     * @param rounding rounding in pixels
     */
    public void setRoundingCorenrsValue(int rounding) {

    }

    /**
     * Set on content item click listener
     *
     * @param listener listener
     */
    public void setOnContentItemClickListener(IOnContentItemClickListener listener) {
        if (mAdapter != null) {
            mAdapter.setOnItemClickListener(listener);
        }
    }


    /**
     * Clear content data from view
     */
    public void clearDataFromView() {
        mItemsList.clear();
        //      mSponsoredContentDisplayed.clear();
        mAdapter.clearContentDataWithNotify();
    }

    /**
     * Add content data to view
     */
    public void addDataAndRefreshView(ContentData contentData) {
        if (contentData != null && contentData.getContentDataItems() != null) {
            mAdapter.setWidgetTypeAndStyleID(contentData.getSourceWidgetType(), contentData.getStyleID());

            mItemsList.clear();
            mItemsList.addAll(contentData.getContentDataItems());
            //add position of promoted items
            mAdapter.notifyDataSetChanged();
        }
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);

        int spanCount = getCurrentSpanCount();
        mAdapter.setSpanCount(spanCount, newConfig.orientation);
        setLayoutManagerView(mLayoutViewType);
        mCustomRecyclerView.scrollToPosition(mCurrentVisiblePosition);
    }

    /**
     * Show or Hide loading progress view
     *
     * @param show
     */
    public void showLoadingProgressView(boolean show) {
        if (show) {
            mLoadingProgressView.setVisibility(View.VISIBLE);
            mLoadingProgressView.startLoadingAnimation();
        } else {
            mLoadingProgressView.stopLoadingAnimation();
            mLoadingProgressView.setVisibility(View.GONE);
        }
    }


    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
    }
}
