package com.kidoz.sdk.api.server_connect;

import android.content.ContentValues;
import android.content.Context;
import android.os.Build;

import com.kidoz.events.DeviceUtils;
import com.kidoz.events.EventBulk;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.EncoderUtils;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentData;

import java.io.BufferedInputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Locale;
import java.util.Set;

import okhttp3.Call;
import okhttp3.Response;


public class SdkAPIManager extends BaseAPIManager {
    private static final String TAG = SdkAPIManager.class.getName();

    private String mPackageName;
    private boolean mDeveloperLogging;

    private static SdkAPIManager mSdkInstnace;

    public static SdkAPIManager getSdkApiInstance(Context context) {
        if (mSdkInstnace == null) {
            mSdkInstnace = new SdkAPIManager(context);
        }
        return mSdkInstnace;
    }

    public static void init(Context context, String pId, String sKey, boolean developerLogging) {
        getSdkApiInstance(context).setProperties(pId, sKey, developerLogging);
    }

    public SdkAPIManager(Context context) {
        initConnectionClient(false);
        mPackageName = context.getPackageName();
    }

    private void setProperties(String pId, String sKey, boolean developerLogging) {
        mPublicherId = pId;
        mSecurityToken = sKey;
        mDeveloperLogging = developerLogging;
    }

    //SDK ApiApi Functions
    private static final String JSON = "JSON";
    private static final String REQUEST_FORMAT = "resFormat";

    private static final String GET_CONTENT = "getContent";
    private static final String INIT_SDK_CONFIGURATION = "initSDK";
    private static final String SEND_KIDOZ_EVENTS_TO_SERVER = "SendKidozEventsToServer";
    private static final String GET_SDK_RESOURCES = "getSharedResources";
    private static final String GET_STYLE = "getStyle";

    public static final String METHOD_NAME = "method_name";

    /**
     * Load content
     *
     * @param styleId      content view style
     * @param resultCallback result callback
     */
    public void loadContent(Context context, String styleId, String widgetType, final ApiResultCallback<ContentData> resultCallback) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(METHOD_NAME, GET_CONTENT);
        contentValues.put("publisher_id", mPublicherId);
        contentValues.put("package_id", mPackageName);
        contentValues.put("device_lang", Locale.getDefault().getLanguage());
        contentValues.put("style_id", styleId);
        contentValues.put("widget_type", widgetType);
        contentValues.put("device_hash", com.kidoz.sdk.api.general.utils.Utils.generateUniqeDeviceID(mPackageName, mPublicherId));
        String addId = DeviceUtils.getGoogleAdvertisingID(context);
        if (addId == null) {
            addId = "";
        }
        contentValues.put("google_id", addId);
        contentValues.put("version", ConstantDef.SDK_VERSION);
        contentValues.put("sdk_version", ConstantDef.SDK_VERSION);
        contentValues.put("actual_sdk_version", ConstantDef.ACTUAL_SDK_VERSION);
        contentValues.put("os_type", "android");
        contentValues.put("os_version", String.valueOf(Build.VERSION.SDK_INT));

        if (Utils.getIsDeviceTablet(context)) {
            contentValues.put("device_type", "2");
        } else {
            contentValues.put("device_type", "1");
        }

        long ts = System.currentTimeMillis() / 1000;
        contentValues.put("ts", ts);
        contentValues.put("hash", generateRequestHash(contentValues, ts, mSecurityToken));

        startServerConnection(context, MAIN_SERVER_URL, CONNECTION_TYPE.GET, SdkRequestType.LOAD_SDK_CONTENT, contentValues, 1, resultCallback, mDeveloperLogging, false);
    }


    public void initSdk(Context context, ApiResultCallback<PropertiesObj> resultCallback) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(METHOD_NAME, INIT_SDK_CONFIGURATION);
        contentValues.put("publisher_id", mPublicherId);
        contentValues.put("package_id", mPackageName);
        contentValues.put("device_lang", Locale.getDefault().getLanguage());
        contentValues.put("device_type", ScreenUtils.getIsTablet(context) == true ? "2" : "1");
        contentValues.put("sdk_version", ConstantDef.SDK_VERSION);
        contentValues.put("actual_sdk_version", ConstantDef.ACTUAL_SDK_VERSION);
        contentValues.put("os_type", "android");
        contentValues.put("device_hash", com.kidoz.sdk.api.general.utils.Utils.generateUniqeDeviceID(mPackageName, mPublicherId));

        try {
            startServerConnection(context, SDK_API_URL + INIT_SDK_CONFIGURATION, CONNECTION_TYPE.GET, SdkRequestType.VALIDATE_SDK, contentValues, 1, resultCallback, mDeveloperLogging, false);
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to validateSDK: " + ex.getMessage());
            resultCallback.onFailed();
        }
    }

    public void getSDKResources(Context context, ApiResultCallback<Boolean> resultCallback) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(METHOD_NAME, GET_SDK_RESOURCES);
        contentValues.put("publisher_id", mPublicherId);
        contentValues.put("device_lang", Locale.getDefault().getLanguage());
        contentValues.put("package_id", mPackageName);
        contentValues.put("device_type", ScreenUtils.getIsTablet(context) == true ? "2" : "1");
        contentValues.put("sdk_version", ConstantDef.SDK_VERSION);
        contentValues.put("actual_sdk_version", ConstantDef.ACTUAL_SDK_VERSION);
        contentValues.put("device_hash", com.kidoz.sdk.api.general.utils.Utils.generateUniqeDeviceID(mPackageName, mPublicherId));
        contentValues.put("os_type", "android");

        try {
            startServerConnection(context, SDK_API_URL + GET_SDK_RESOURCES, CONNECTION_TYPE.GET, SdkRequestType.GET_SDK_RESOURCES, contentValues, 1, resultCallback, mDeveloperLogging, true);
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to validateSDK: " + ex.getMessage());
            resultCallback.onFailed();
        }
    }

    public void getStyle(Context context, int viewType, ApiResultCallback<Boolean> resultCallback) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(METHOD_NAME, "getStyle");
        contentValues.put("publisher_id", mPublicherId);
        contentValues.put("device_lang", Locale.getDefault().getLanguage());
        contentValues.put("widget_type", String.valueOf(viewType));
        contentValues.put("package_id", mPackageName);
        contentValues.put("device_type", ScreenUtils.getIsTablet(context) == true ? "2" : "1");
        contentValues.put("sdk_version", ConstantDef.SDK_VERSION);
        contentValues.put("actual_sdk_version", ConstantDef.ACTUAL_SDK_VERSION);
        contentValues.put("os_type", "android");
        contentValues.put("device_hash", com.kidoz.sdk.api.general.utils.Utils.generateUniqeDeviceID(mPackageName, mPublicherId));

        try {
            startServerConnection(context, SDK_API_URL + GET_STYLE, CONNECTION_TYPE.GET, SdkRequestType.GET_STYLE, contentValues, 1, resultCallback, mDeveloperLogging, false);
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to get style: " + ex.getMessage());
            resultCallback.onFailed();
        }
    }

    /**
     * Send kidoz events to server
     *
     * @param eventsJsonBulk
     */
    public ResultData<Boolean> sendKidozEventsToServerNotAsync(String eventsJsonBulk) {
        ContentValues contentValues = new ContentValues();
        contentValues.put("methodName", SEND_KIDOZ_EVENTS_TO_SERVER);

        String encodedString = eventsJsonBulk;
        try {
            encodedString = URLEncoder.encode(eventsJsonBulk, "UTF-8");
        } catch (UnsupportedEncodingException e) {
        }
        contentValues.put("KidozEventsLogAsJson", encodedString);
        contentValues.put(REQUEST_FORMAT, JSON);

        ResultData<Boolean> resultCallback = null;

        Call call = null;
        try {
            call = BaseConnectionClient.makePostConnection(KIDOZ_ANALITYCS_SERVER_URL, contentValues, "analyticsRequest");
            if (call != null) {

                SDKLogger.printDebbugLog("|KIDOZ SDK| **[REQUEST]** API : [POST],  [SEND_EVENT_ANALYTYCS]");
                Response response = call.execute();
                if (response != null) {
                    if (response.isSuccessful()) {
                        SDKLogger.printDebbugLog("|KIDOZ SDK| **[RESPONSE SUCCESS]** API : [POST],  [SEND_EVENT_ANALYTICS]");
                        try {
                            String responce = StreamToStringConverter.readStream(new BufferedInputStream(response.body().byteStream()), null, false);
                            if (responce != null) {
                                resultCallback = EventBulk.parseResultSuccsesStatus(responce);
                            }

                        } catch (Exception e) {
                            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n Unable to convertByte Stream to String! [ Kidoz Analytics Data Sent ]  : \n\n" + e.getMessage());
                        }
                    } else {
                        com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n Server connectivity Error! [ Kidoz Analytics Data Sent ]  Code : " + response.code());
                    }
                }
            }
        } catch (Exception e) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n IO Exception On [ Kidoz Analytics Data Sent ] \n " + e.getMessage());
        }
        return resultCallback;
    }

    /**
     * Generate security hash for a server request
     */
    private String generateRequestHash(ContentValues contentValues, long ts, String token) {
        Set<String> set = contentValues.keySet();
        ArrayList<String> list = new ArrayList<String>();
        list.addAll(set);
        Collections.sort(list);

        StringBuilder stringBuilder = new StringBuilder();
        for (int i = 0; i < list.size(); i++) {
            stringBuilder.append(contentValues.getAsString(list.get(i)));
        }
        stringBuilder.append(ts);
        stringBuilder.append(token);

        return EncoderUtils.SHA1(stringBuilder.toString());
    }

}

