package com.kidoz.sdk.api;

import android.animation.Animator;
import android.content.Context;
import android.graphics.Point;
import android.util.AttributeSet;
import android.view.View;
import android.view.animation.AccelerateDecelerateInterpolator;
import android.widget.ImageView;
import android.widget.RelativeLayout;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.assets_handling.AssetUtil;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.ui_views.custom_drawables.KidozLogoDrawable;
import com.kidoz.sdk.api.ui_views.flexi_view.MovableView;
import com.kidoz.sdk.api.ui_views.interstitial.LocalIntrst;
import com.kidoz.sdk.api.ui_views.parental_lock.AssetView;

import org.json.JSONObject;

import java.io.File;

import de.greenrobot.event.EventBus;

/**
 * KIDOZ FeedButton widget class
 */
public class FeedButton extends RelativeLayout
{
    private static final String TAG = FeedButton.class.getSimpleName();

    private static final float BUTTON_SIZE_AS_SCREEN_PORTION = 0.20f;
    private AssetView mAssetView;
    private RelativeLayout.LayoutParams mAssetViewParams;
    private RelativeLayout.LayoutParams mContainerParams;

    private ExtendedFeedView mFeedView;

    private View mRootView;

    private int mDeveloperSize = -1;
    private boolean mSizeInitFinished = false;
    private JSONObject mJsonObject;

    private boolean mIsFeedButtonReady = false;
    private boolean mIsAutoShowFeedButton = true;
    protected LocalIntrst mInterstitial;

    public FeedButton(Context context)
    {
        super(context);
        initView(context);
    }

    /**
     * Initiate button view
     *
     * @param context
     * @param size    button size
     */
    public FeedButton(Context context, int size)
    {
        super(context);
        mDeveloperSize = size;
        initView(context);
    }

    /**
     * Initiate button view
     */
    public FeedButton(Context context, AttributeSet attrs)
    {
        super(context, attrs);
        initView(context);
    }

    public FeedButton(Context context, AttributeSet attrs, int defStyleAttr)
    {
        super(context, attrs, defStyleAttr);
        initView(context);
    }

    /**
     * Set kidoz player event listeners.
     * Listen for events generated by any launched player (Unique for single view)
     *
     * @param kidozPlayerListener event listener
     */
   /* public void setKidozPlayerListener(KidozPlayerListener kidozPlayerListener)
    {
        mKidozPlayerEventHelper = new KidozPlayerEventHelper(kidozPlayerListener, WidgetType.WIDGET_TYPE_FEED);
    }*/

    /**
     * Initiate view
     */
    private void initView(Context context)
    {
        setVisibility(View.GONE);
        mIsAutoShowFeedButton = true;

        if (EventBus.getDefault().isRegistered(this) == false) {
            EventBus.getDefault().register(this);
        }

        mRootView = this;

        int size = getDefaultButtonSize(context);

        if (mDeveloperSize != -1)
        {
            size = mDeveloperSize;
        }

        mAssetView = new AssetView(context);
        mAssetView.setScaleType(ImageView.ScaleType.FIT_CENTER);
        mAssetViewParams = new RelativeLayout.LayoutParams(size, size);
        mDeveloperSize = size;
        addView(mAssetView, mAssetViewParams);

        mContainerParams = new RelativeLayout.LayoutParams(size, size);
        setLayoutParams(mContainerParams);

        if (isInEditMode() == false)
        {
            initFeedView(getContext());
        } else
        {
            mAssetView.loadAsset(new File(""), new KidozLogoDrawable(), null);
        }
    }

    // Event bus callback
    public void onEvent(EventMessage event)
    {
        if (event.getMessageType() == EventMessage.MessageType.INIT_FEED_BUTTON)
        {
            loadExistingAssetsIfExists();
            if(EventBus.getDefault().isRegistered(this)) {
                EventBus.getDefault().unregister(this);
            }
        }
    }

    /**
     * Load feed button style view define on server
     */
    private void loadButtonStyle(JSONObject jsonObject)
    {

        mJsonObject = jsonObject;
        int size = getDefaultButtonSize(getContext());
        if (mDeveloperSize != -1)
        {
            size = mDeveloperSize;
        }
        if (mDeveloperSize == -1 || jsonObject.optInt(FeedView.OVERRIDE_USER_PROPERTIES, 0) == 1)
        {
            float scale = (float) jsonObject.optDouble(FeedView.KIDOZ_FEED_BUTTON_SCALE, BUTTON_SIZE_AS_SCREEN_PORTION);
            Point point = Utils.getScreenSize(getContext());
            size = (int) (Math.min(point.x, point.y) * scale);
        }

        mAssetViewParams.width = size;
        mAssetViewParams.height = size;

        mContainerParams.width = size;
        mContainerParams.height = size;

        mSizeInitFinished = true;

        String fileKEy = jsonObject.optString(FeedView.KIDOZ_FEED_BUTTON);
        mAssetView.loadAsset(AssetUtil.getAssetFile(getContext(), fileKEy), new KidozLogoDrawable(), new AssetView.IOnAssetLoadedListener()
        {
            @Override
            public void onAssetLoaded(boolean successfull)
            {
                if (mIsAutoShowFeedButton)
                {
                    showFeedButton();
                }
            }
        });
    }

    /**
     * Show feed button with animation
     */
    private void showFeedButton()
    {
        if (getVisibility() != VISIBLE)
        {
            GenAnimator.playGrowAnimation(FeedButton.this, 300, 0, new AccelerateDecelerateInterpolator(), new Animator.AnimatorListener()
            {
                @Override
                public void onAnimationStart(Animator animation)
                {
                    // If to show or not when the view is ready
                    if (mIsAutoShowFeedButton)
                    {
                        setVisibility(View.VISIBLE);
                    }
                    requestFocus();
                    bringToFront();
                }

                @Override
                public void onAnimationEnd(Animator animation)
                {
                    requestFocus();
                    bringToFront();

                    EventManager.getInstance(getContext()).logEvent(getContext(), WidgetType.WIDGET_TYPE_FEED.getStringValue(), null, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_BUTTON_VIEW, EventParameters.LABEL_FEED_BUTTON_VIEW);
                    // Added to verify when fedd button is ready , witch is used for
                    // sending button view event when hiding and showing it after fedd button view was initiated
                    mIsFeedButtonReady = true;
                }

                @Override
                public void onAnimationCancel(Animator animation)
                {

                }

                @Override
                public void onAnimationRepeat(Animator animation)
                {

                }
            });
        }
    }

    @Override
    public void setVisibility(int visibility)
    {
        super.setVisibility(visibility);

        if (visibility != VISIBLE)
        {
            mIsAutoShowFeedButton = false;
        } else
        {
            if (mIsFeedButtonReady == true)
            {
                EventManager.getInstance(getContext()).logEvent(getContext(), WidgetType.WIDGET_TYPE_FEED.getStringValue(), null, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_BUTTON_VIEW, EventParameters.LABEL_FEED_BUTTON_VIEW);
            }
        }
    }

    private void loadExistingAssetsIfExists()
    {
        if (KidozSDK.isInitialised() == true)
        {
            JSONObject jsonObject = DatabaseManager.getInstance(getContext()).getConfigTable().loadProperties(FeedView.TAG);
            if (jsonObject != null)
            {
                // Init interstitial if needed
                if (jsonObject.optInt(MovableView.LOUNCH_INTERSTITIAL_ON_OPEN, 0) == 1)
                {
                    if (mInterstitial == null)
                    {
                        mInterstitial = new LocalIntrst(getContext());
                    }
                }

                loadButtonStyle(jsonObject);
            }
        }
    }

    /**
     * Initiate button click listener
     */
    private void initClickListener()
    {
        setOnClickListener(new OnClickListener()
        {
            @Override
            public void onClick(View v)
            {

                if (mFeedView != null)
                {
                    GenAnimator.clickItemAnimation(mRootView, 70, new GenAnimator.ViewAnimationListener()
                    {
                        @Override
                        public void onAnimationEnd()
                        {
                            mFeedView.showView();

                            if (mInterstitial != null)
                            {
                                mInterstitial.loadAd();
                            }
                        }

                        @Override
                        public void onAnimationStart()
                        {

                        }
                    });
                }
            }
        });
    }

    /**
     * Initiate interstitial view
     */
    private void initFeedView(Context context)
    {
        mFeedView = new ExtendedFeedView(new FeedView.Builder(context));
        initClickListener();
    }

    /**
     * Get the inner interstitial feed view object
     *
     * @return interstitial view object
     */
    public FeedView getFeedView()
    {
        if (mFeedView == null)
        {
            initFeedView(getContext());
        }

        return mFeedView;
    }

    /**
     * Set button size in pixels
     *
     * @param sizePx size in pixels
     */
    public void setFeedButtonSize(int sizePx)
    {

        boolean allowChange = true;
        if (mSizeInitFinished && mJsonObject != null)
        {
            if (mJsonObject.optInt(FeedView.OVERRIDE_USER_PROPERTIES, 0) == 1)
            {
                allowChange = false;
            }
        }

        if (allowChange)
        {
            mDeveloperSize = sizePx;
            if (mAssetViewParams != null)
            {
                mAssetViewParams.width = sizePx;
                mAssetViewParams.height = sizePx;

                mContainerParams.width = sizePx;
                mContainerParams.height = sizePx;
            }
        }
    }

    /**
     * Set button size in DP
     *
     * @param sizeDp size in pixels
     */
    public void setFeedButtonSizeDp(int sizeDp)
    {

        int sizePx = Utils.dpTOpx(getResources(), sizeDp);

        boolean allowChange = true;
        if (mSizeInitFinished && mJsonObject != null)
        {
            if (mJsonObject.optInt(FeedView.OVERRIDE_USER_PROPERTIES, 0) == 1)
            {
                allowChange = false;
            }
        }

        if (allowChange)
        {
            mDeveloperSize = sizePx;
            if (mAssetViewParams != null)
            {
                mAssetViewParams.width = sizePx;
                mAssetViewParams.height = sizePx;

                mContainerParams.width = sizePx;
                mContainerParams.height = sizePx;
            }
        }
    }

    /**
     * Get button size (Could be 0 if button didn't finished initiating)
     *
     * @return button size in pixels
     */
    public int getButtonSize()
    {
        if (mDeveloperSize == -1)
        {
            return 0;
        } else
        {
            return mDeveloperSize;
        }
    }

    /**
     * Get default button size
     *
     * @return button size in pixels
     */
    private static int getDefaultButtonSize(Context context)
    {
        Point point = Utils.getScreenSize(context);
        return (int) (Math.min(point.x, point.y) * BUTTON_SIZE_AS_SCREEN_PORTION);
    }

    class ExtendedFeedView extends FeedView
    {
        public ExtendedFeedView(Builder builder)
        {
            super(builder);
        }

        @Override
        public void showView()
        {
            //super.showView();
            mIsAutomaticInvocation = false;
            mIsPostLoadData = true;
            mShowWhenInitiated = false;

            if (KidozSDK.isInitialised())
            {
                if (mOnInterstitialViewEventListener != null)
                {
                    mOnInterstitialViewEventListener.onReadyToShow();
                }

                super.openSimpleDialog();
            } else
            {
                mShowWhenInitiated = true;
            }
        }
    }

    @Override
    protected void onWindowVisibilityChanged(int visibility)
    {
        super.onWindowVisibilityChanged(visibility);

        if (KidozSDK.isInitialised())
        {

            if (!isInEditMode() && (visibility == View.GONE || visibility == View.INVISIBLE))
            {
                EventManager.getInstance(getContext()).startEventsSync(getContext());
            }
        }
    }
}
