package com.kidoz.sdk.api.general.utils;

import android.app.Activity;
import android.content.Context;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.graphics.Point;
import android.os.Build;
import android.os.PowerManager;
import android.util.DisplayMetrics;
import android.view.Display;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;

public final class ScreenUtils
{
    private static final String TAG = ScreenUtils.class.getSimpleName();

    public static boolean getIsScreenOff(Context context)
    {
        boolean result = false;
        if (android.os.Build.VERSION.SDK_INT < 20)
        {
            PowerManager powerManager = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
            result = (powerManager.isScreenOn() == false);
        } else
        {
            WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
            Display display = windowManager.getDefaultDisplay();
            result = display.getState() == Display.STATE_OFF ? true : false;
        }
        return result;
    }

    public static double getDeviceScreenSizeInInches(Context context)
    {
        int mWidthPixels;
        int mHeightPixels;

        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = windowManager.getDefaultDisplay();
        DisplayMetrics displayMetrics = new DisplayMetrics();
        display.getMetrics(displayMetrics);

        // since SDK_INT = 1;
        mWidthPixels = displayMetrics.widthPixels;
        mHeightPixels = displayMetrics.heightPixels;

        // includes window decorations (status bar bar/menu bar)
        if (Build.VERSION.SDK_INT >= 14 && Build.VERSION.SDK_INT < 17)
        {
            try
            {
                mWidthPixels = (Integer) Display.class.getMethod("getRawWidth").invoke(display);
                mHeightPixels = (Integer) Display.class.getMethod("getRawHeight").invoke(display);
            } catch (Exception ignored)
            {
            }
        }

        // includes window decorations (status bar bar/menu bar)
        if (Build.VERSION.SDK_INT >= 17)
        {
            try
            {
                Point realSize = new Point();
                Display.class.getMethod("getRealSize", Point.class).invoke(display, realSize);
                mWidthPixels = realSize.x;
                mHeightPixels = realSize.y;
            } catch (Exception ignored)
            {
            }
        }

        DisplayMetrics dm = new DisplayMetrics();
        windowManager.getDefaultDisplay().getMetrics(dm);
        double x = Math.pow(mWidthPixels / dm.xdpi, 2);
        double y = Math.pow(mHeightPixels / dm.ydpi, 2);
        double screenInches = Math.sqrt(x + y);

        return (double) Math.round(screenInches * 10) / 10;
    }

    public static int getScreenSize(Context context, boolean isWidth)
    {
        int result = 0;
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = windowManager.getDefaultDisplay();
        if (isWidth)
        {
            if (android.os.Build.VERSION.SDK_INT >= 19)
            {
                Point point = new Point();
                display.getRealSize(point);
                result = point.x;
            } else
            {
                result = display.getWidth();
            }
        } else
        {
            if (android.os.Build.VERSION.SDK_INT >= 19)
            {
                Point point = new Point();
                display.getRealSize(point);
                result = point.y;
            } else
            {
                result = display.getHeight();
            }
        }
        return result;
    }

    public static void hideNavigationAndStatusBar(final Window window)
    {
        if (window != null)
        {
            if (Build.VERSION.SDK_INT < 16)
            {
                window.setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
            } else if (Build.VERSION.SDK_INT >= 16 && Build.VERSION.SDK_INT < 19)
            {
                window.getDecorView().setSystemUiVisibility(0);
                window.getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_FULLSCREEN);
            }
            if (android.os.Build.VERSION.SDK_INT >= 19)
            {
                final int flags = View.SYSTEM_UI_FLAG_LAYOUT_STABLE | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_FULLSCREEN | View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY;
                window.getDecorView().setSystemUiVisibility(0);
                window.getDecorView().setSystemUiVisibility(flags);
            }
        }
    }

    public static void hideNavigationBar(final Window window)
    {
        if (window != null)
        {
            if (Build.VERSION.SDK_INT < 16)
            {
                window.setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
            } else if (Build.VERSION.SDK_INT >= 16 && Build.VERSION.SDK_INT < 19)
            {
                window.getDecorView().setSystemUiVisibility(0);
                window.getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_FULLSCREEN);
            }
            if (android.os.Build.VERSION.SDK_INT >= 19)
            {
                final int flags = View.SYSTEM_UI_FLAG_LAYOUT_STABLE | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_FULLSCREEN | View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY;
                window.getDecorView().setSystemUiVisibility(0);
                window.getDecorView().setSystemUiVisibility(flags);
            }
        }
    }

    public static void showNavigationAndStatusBar(final Window window)
    {
        if (window != null)
        {
            if (Build.VERSION.SDK_INT < 16)
            {
                window.clearFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN);
            } else if (Build.VERSION.SDK_INT >= 16 && Build.VERSION.SDK_INT < 19)
            {
                window.getDecorView().setSystemUiVisibility(0);
            }
            if (android.os.Build.VERSION.SDK_INT >= 19)
            {
                window.getDecorView().setSystemUiVisibility(0);
            }
        }
    }

    public static boolean getIsTablet(Context context)
    {
        boolean result = true;
        if (context != null)
        {
            if ((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) == Configuration.SCREENLAYOUT_SIZE_NORMAL)
            {
                result = false;
            }
        }
        return result;
    }

    public static void setScreenOrientation(Activity activity, int orientation)
    {
        if (activity != null)
        {
            if (orientation == ActivityInfo.SCREEN_ORIENTATION_PORTRAIT || orientation == ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE || orientation == ActivityInfo.SCREEN_ORIENTATION_SENSOR)
            {
                activity.setRequestedOrientation(orientation);
            }
        }
    }

    /**
     * This method return the factor to multiply by.
     * All graphics should be design for xxhdpi screen so when a xxhdpi device is using this method the graphics won't be scaled.
     * If the device is a large the scale factor is increased by additional 1.5 and if it's extra large the scale factor is increased by additional 2.
     *
     * @param context
     * @return
     */
    public static float getScaleFactor(Context context)
    {
        float result = 1f;
        if (context != null)
        {
            result = context.getResources().getDisplayMetrics().density;
            result /= 3f;
            if ((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) == Configuration.SCREENLAYOUT_SIZE_LARGE)
            {
                result *= 1.25f;
            } else if ((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) == Configuration.SCREENLAYOUT_SIZE_XLARGE)
            {
                result *= 1.5f;
            }
        }
        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>Scale factor = " + String.valueOf(result));
        return result;
    }

    public static float getTextScale(Context context)
    {
        float result = 1f;
        if ((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) == Configuration.SCREENLAYOUT_SIZE_LARGE)
        {
            result = 2f;
        } else if ((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) == Configuration.SCREENLAYOUT_SIZE_XLARGE)
        {
            result = 2.5f;
        }
        return result;
    }

    public static int[] getScaledDimension(int srcWidth, int srcHeight, int dstWidth, int dstHeight)
    {
        int[] result = new int[2];

        float widthScale = dstWidth / (float) srcWidth;
        float heightScale = dstHeight / (float) srcHeight;
        float scale = Math.min(widthScale, heightScale);
        result[0] = Math.round((srcWidth * scale));
        result[1] = Math.round((srcHeight * scale));
        return result;
    }

    public static int getStatusBarHeight(Context context)
    {
        int result = 0;
        int resourceId = context.getResources().getIdentifier("status_bar_height", "dimen", "android");
        if (resourceId > 0)
        {
            result = context.getResources().getDimensionPixelSize(resourceId);
        }
        return result;
    }

    public static int getNavigationBarHeight(Context context)
    {
        int result = 0;
        int resourceId = context.getResources().getIdentifier("navigation_bar_height", "dimen", "android");
        if (resourceId > 0)
        {
            result = context.getResources().getDimensionPixelSize(resourceId);
        }
        return result;
    }
}
