package com.kidoz.sdk.api;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.util.Log;
import android.widget.Toast;

import com.kidoz.events.DeviceUtils;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.assets_handling.ImageAssetsUtils;
import com.kidoz.sdk.api.general.assets_handling.SoundAssetsUtils;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.CookieManager;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.general.utils.StorageLife;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.players.video_player.VideoManager;
import com.kidoz.sdk.api.receivers.SdkReceiver;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.kidoz.sdk.api.ui_views.html_view.HtmlManager;
import com.kidoz.sdk.api.ui_views.html_view.HtmlViewWrapper;
import com.kidoz.sdk.api.ui_views.html_view.IOnHtmlWebViewInterface;
import com.kidoz.sdk.api.ui_views.interstitial.IntrstWrapper;

import org.json.JSONObject;

import java.util.List;

import de.greenrobot.event.EventBus;

/**
 * KIDOZ SDK initiator class
 */
public final class KidozSDK
{
    private static final String TAG = KidozSDK.class.getSimpleName();

    private static boolean isInitialized = false;
    private static String mPublisherId = null;
    private static String mSecurityKey = null;

    private static boolean isDeveloperLoggingON = false;

    /**
     * Initialize KIDOZ Sdk
     *
     * @param context      activity context
     * @param publisher_id valid publisher id
     * @param token        valid security token
     * @throws RuntimeException in case of invalid or missing <b>publisher_id</b> , <b>Security Token </b> or missing <b>Manifest.xml</b> definitions
     */
    public static void initialize(final Context context, String publisher_id, String token)
    {
        // Security check and developer id basic validation
        if (publisher_id == null || token == null)
        {
            throw new RuntimeException("Missing valid Publisher Id or security token!");
        } else
        {
            try
            {
                Long.parseLong(publisher_id);
            } catch (NumberFormatException e)
            {
                throw new RuntimeException("Invalid publisher id! Please check you publisher id..");
            }

            if (token.length() > 48)
            {
                throw new RuntimeException("Invalid Security Token! Please recheck you security token..");
            }
        }

        // Check if any demo publisher id is used and notify the developer
        if (publisher_id != null)
        {
            if(publisher_id.equals("5") || publisher_id.equals("7") || publisher_id.equals("8")) {
                if (context != null)
                {
                    if (context instanceof Activity)
                    {
                        ((Activity) context).runOnUiThread(new Runnable()
                        {
                            @Override
                            public void run()
                            {
                                Toast.makeText(context, "Notice ! You are using Demo PUBLISHER ID !", Toast.LENGTH_SHORT).show();
                            }
                        });
                    } else
                    {
                        Toast.makeText(context, "Notice ! You are using Demo PUBLISHER ID !", Toast.LENGTH_LONG).show();
                    }
                }
            }
        }

        // Check manifest declaration
        if (ConstantDef.SDK_EXTENSION_TYPE == ConstantDef.EXTENSION_TYPE_ANDROID || ConstantDef.SDK_EXTENSION_TYPE == ConstantDef.EXTENSION_TYPE_ECLIPSE || ConstantDef.SDK_EXTENSION_TYPE == ConstantDef.EXTENSION_TYPE_COCOS_DX)
        {
            checkManifestDeclarations(context);
        }

        try
        {
            DeviceUtils.getGoogleAdvertisingID(context);
            Utils.preInitiate(context);

            //SDKLogger.printWarningLog("SDK DEVICE HASH : " + Utils.generateUniqeDeviceID(context.getPackageName(), publisher_id));

            SharedPreferences eventsPref = context.getSharedPreferences(EventManager.EVENTS_SHARED_PREFERENCES_KEY, 0);
            long sessionID = eventsPref.getLong(EventManager.EVENTS_LAST_SESSION_ID_KEY, 0);
            sessionID += 1;
            SharedPreferences.Editor editor = eventsPref.edit();
            editor.putLong(EventManager.EVENTS_LAST_SESSION_ID_KEY, sessionID);
            editor.apply();

            JSONObject eventParams = new JSONObject();
            eventParams.put(EventParameters.DEVELOPER_ID, publisher_id);
            eventParams.put(EventManager.LOG_LEVEL_KEY, EventManager.LOG_NORMAL_LEVEL);
            eventParams.put(EventParameters.SESSION_ID, sessionID);

            EventManager.getInstance(context).init(context, eventParams);

            //EventManager.getInstance(context).logEvent(context, null, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_SDK_INIT, publisher_id, false);
            EventManager.getInstance(context).logEvent(context, null, null, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SESSION, EventParameters.ACTION_SESSION_START, publisher_id);
            // Reset The storage of the impressions for the HTML conent
            CookieManager.resetStorage(context, null, StorageLife.SESSION);

            if (isInitialized)
            {
                SDKLogger.printWarningLog("SDK REPEAT INIT");
                EventManager.getInstance(context).logEvent(context, null, null, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_SDK_ALREADY_INIT, publisher_id);
            } else
            {
                SDKLogger.printWarningLog("SDK INIT");
            }

            SharedPreferencesUtils.saveSharedPreferencesData(context, "PUBLISHER_ID", publisher_id);

            mPublisherId = publisher_id;
            mSecurityKey = token;
            SdkAPIManager.init(context, publisher_id, token, isDeveloperLoggingON);

            validateSDKConfiguration(context, publisher_id);

            EventManager.getInstance(context).startEventsSync(context);
        } catch (Exception ex)
        {
            SDKLogger.printErrorLog("Error initiating " + ex.getMessage());
        }
    }

    private static SdkAPIManager getApiManager(Context context)
    {
        SdkAPIManager.init(context, mPublisherId, mSecurityKey, isDeveloperLoggingON);
        return SdkAPIManager.getSdkApiInstance(context);
    }

    /**
     * Check if Kidoz SDK has been initialised
     *
     * @return initialized
     */
    public static boolean isInitialised()
    {
        return isInitialized;
    }

    /**
     * Validate SDK configuration (Configuration retrieved from remote server)
     */
    private static void validateSDKConfiguration(final Context context, final String publicher_id)
    {
        SDKLogger.printDebbugLog(TAG, ">>>> initSdk");
        if (context != null)
        {
            PropertiesObj appConfig = DatabaseManager.getInstance(context).getConfigTable().loadAppProperties();
            if (appConfig != null)
            {
                // Update Current Sdk style version for Event Log
                EventManager.SDK_STYLE_VERSION_NUM = appConfig.getSdkStyleVersion();
            }

            getApiManager(context).initSdk(context, new ApiResultCallback<PropertiesObj>()
            {
                @Override
                public void onServerResult(ResultData<?> result)
                {
                    if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true && result.getData() != null && result.getData() instanceof PropertiesObj)
                    {
                        PropertiesObj serverConfig = (PropertiesObj) result.getData();
                        PropertiesObj appConfig = DatabaseManager.getInstance(context).getConfigTable().loadAppProperties();

                        boolean clearHtmlCache = true;

                        // We save the config before loading missing graphics
                        if (appConfig == null)
                        {
                            PropertiesObj prop = new PropertiesObj(serverConfig.getData());
                            prop.setConfigVersion(-1); // The value set to -1 to avoid updating the state of version before the graphics is updated/loaded
                            DatabaseManager.getInstance(context).getConfigTable().insertAppProperties(prop);
                        } else
                        {
                            if(appConfig.getHtmlLoaderDefaultLink() != null && appConfig.getHtmlLoaderDefaultLink().equals(serverConfig.getHtmlLoaderDefaultLink())) {
                                clearHtmlCache = false;
                            }

                            appConfig.updateConfigFromData(serverConfig);
                            DatabaseManager.getInstance(context).getConfigTable().insertAppProperties(appConfig);
                        }

                        // Update Current Sdk style version for Event Log
                        if (serverConfig != null)
                        {
                            EventManager.SDK_STYLE_VERSION_NUM = serverConfig.getSdkStyleVersion();
                        }

                        if (appConfig == null || appConfig.getConfigVersion() < serverConfig.getConfigVersion())
                        {
                            SDKLogger.printDebbugLog(TAG, ">>>>Going to download resources...");
                            // Need to update configuration (graphics, strings etc...)
                            getApiManager(context).getSDKResources(context, new ApiResultCallback<Boolean>()
                            {
                                @Override
                                public void onServerResult(ResultData<?> result)
                                {
                                    if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true && result.getData() != null && result.getData() instanceof Boolean)
                                    {
                                        if ((Boolean) result.getData() == true)
                                        {
                                            SharedPreferencesUtils.removeSharedPreferencesData(context, ImageAssetsUtils.IMAGE_ASSETS_SYNCED_FLAG);
                                            SharedPreferencesUtils.removeSharedPreferencesData(context, ImageAssetsUtils.PARENTAL_LOCK_ASSETS_SYNCED_FLAG);
                                            SharedPreferencesUtils.removeSharedPreferencesData(context, SoundAssetsUtils.SOUND_ASSETS_SYNCED_FLAG);
                                        }
                                    }
                                }

                                @Override
                                public void onFailed()
                                {
                                }
                            });
                        }

                        // Load Youtube player HTML and webView
                        VideoManager.getInstance(context);
                        // Load Default Html Player url and webView
                        HtmlViewWrapper viewWrapper = HtmlManager.initDefaultHtmlView(context,clearHtmlCache);
                        viewWrapper.setHtmlWebViewListener(new IOnHtmlWebViewInterface()
                        {
                            @Override
                            public void onHtmlFinishedLoading()
                            {
                                super.onHtmlFinishedLoading();

                                isInitialized = true;
                                IntrstWrapper.initInstance(context);

                                EventManager.getInstance(context).logEvent(context, null, null, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_SDK_INIT, publicher_id);

                                EventMessage eventMessage = new EventMessage(EventMessage.MessageType.INIT_SDK);
                                EventBus.getDefault().post(eventMessage);
                            }
                        });
                        viewWrapper.loadHtml(serverConfig.getHtmlLoaderDefaultLink());
//                        viewWrapper.loadHtml("http://s3.amazonaws.com/kidoz.test.bucket/ronybrosh/test_sdk/test_html_loader.html");
                       // viewWrapper.loadHtml(" http://s3.amazonaws.com/kidoz.test.bucket/ronybrosh/test_sdk_2/test_html_loader.html");


                        if (isDeveloperLoggingON)
                        {
                            Log.d(SDKLogger.GENERAL_TAG, "Kidoz SDK has been successfully Initialized !");
                        }
                    }
                }

                @Override
                public void onFailed()
                {

                }
            });
        }
    }

    // Check necessary manifest declarations that mast be present
    private static void checkManifestDeclarations(Context context)
    {
        final PackageManager packageManager = context.getPackageManager();

        // Check Sdk receiver declaration exists in the manifest
        final Intent intent = new Intent(context, SdkReceiver.class);
        List resolveInfo = packageManager.queryBroadcastReceivers(intent, PackageManager.MATCH_DEFAULT_ONLY);
        if (resolveInfo != null && resolveInfo.size() == 0)
        {
            throw new RuntimeException("Missing Android Manifest declaration ! \n" +
                    "Please add the following lines to your Manifest.xml file..\n...\n" +
                    "<receiver android:name=\"com.kidoz.sdk.api.receivers.SdkReceiver\">\n" +
                    "    <intent-filter>\n" +
                    "         <action android:name=\"android.intent.action.PACKAGE_ADDED\" />\n" +
                    "         <data android:scheme=\"package\" />\n" +
                    "    </intent-filter>\n" +
                    "</receiver> \n...\n");
        }
    }

    /**
     * Set developer logging enabled
     * (Must be set before initialization of the SDK)
     *
     * @param enabled
     */
    public static void setLoggingEnabled(boolean enabled)
    {
        isDeveloperLoggingON = enabled;
    }

    public static String getPublisherID()
    {
        return mPublisherId;
    }

    public static String getAuthToken()
    {
        return mSecurityKey;
    }

    public static String getSDKVersion()
    {
        return ConstantDef.ACTUAL_SDK_VERSION;
    }
}
