package com.kidoz.sdk.api;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.view.View;

import com.kidoz.sdk.api.general.ContentLogicLoader;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.CookieManager;
import com.kidoz.sdk.api.general.utils.StorageLife;
import com.kidoz.sdk.api.interfaces.IOnFeedViewEventListener;
import com.kidoz.sdk.api.interfaces.KidozPlayerEventHelper;
import com.kidoz.sdk.api.interfaces.KidozPlayerListener;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.kidoz.sdk.api.structure.ContentData;
import com.kidoz.sdk.api.ui_views.feed_view.FeedFamilyViewDialog;
import com.kidoz.sdk.api.ui_views.feed_view.FeedViewDialog;

import org.json.JSONObject;

import de.greenrobot.event.EventBus;

/**
 * KIDOZ FeedView widget class
 */
public class FeedView
{
    // Properties keys
    public static final String KIDOZ_FEED_BUTTON = "feedButton";
    public static final String KIDOZ_FEED_BUTTON_SCALE = "feedButtonScale";
    public static final String OVERRIDE_USER_PROPERTIES = "overrideUserProperties";
    public static final String LOUNCH_INTERSTITIAL_ON_OPEN = "lounchInterstitialOnOpen";

    public static final String TAG = FeedView.class.getSimpleName();
    public static final String STYLE_ID = "style_id";

    private Builder mBuilder;

    private FeedViewDialog mFeedViewDialog;
    private FeedFamilyViewDialog mFamilyViewDialog;
    protected ContentLogicLoader mContentLogicLoader;

    protected IOnFeedViewEventListener mOnInterstitialViewEventListener;
    protected boolean mIsAutomaticInvocation = false;
    protected boolean mIsShowing = false;
    protected String mFeedViewType = null;

    //Flag used to check ,is Pre load or Post load the content data
    protected boolean mIsPostLoadData = false;
    protected boolean mShowWhenInitiated = false;
    protected JSONObject mJsonObjectProperties;
    protected static boolean mOncePerSessionLoaded = false;

    private KidozPlayerEventHelper mKidozPlayerEventHelper;
    private Dialog.OnDismissListener dismissListener;
    private KidozPlayerListener mKidozPlayerListener;

    /**
     * Set kidoz players event listeners
     * Used for listening to invoked Player state events (Applied for single view)
     *
     * @param kidozPlayerListener event listener
     */
    public void setKidozPlayerListener(KidozPlayerListener kidozPlayerListener)
    {
        mKidozPlayerListener = kidozPlayerListener;
    }

    protected FeedView(Builder builder)
    {
        mBuilder = builder;

        mKidozPlayerEventHelper = new KidozPlayerEventHelper(new KidozPlayerEventHelper.IPlayerListener()
        {
            @Override
            public void onOpen(boolean isMaximizedPlayer)
            {
                if (mKidozPlayerListener != null)
                {
                    mKidozPlayerListener.onPlayerOpen();
                }

                if (mFamilyViewDialog != null && mFamilyViewDialog.isShowing())
                {
                    mFamilyViewDialog.pauseAction(isMaximizedPlayer);
                }
            }

            @Override
            public void onClose(boolean isMaximizedPlayer)
            {
                if (mKidozPlayerListener != null)
                {
                    mKidozPlayerListener.onPlayerClose();
                }

                if (mFamilyViewDialog != null && mFamilyViewDialog.isShowing())
                {
                    mFamilyViewDialog.resumeAction(isMaximizedPlayer);
                }
            }
        }, WidgetType.WIDGET_TYPE_FEED);

        dismissListener = new DialogInterface.OnDismissListener()
        {
            @Override
            public void onDismiss(DialogInterface dialog)
            {

                if (mKidozPlayerEventHelper != null)
                {
                    mKidozPlayerEventHelper.unRegister();
                }

                if (mOnInterstitialViewEventListener != null)
                {
                    mOnInterstitialViewEventListener.onDismissView();
                }

                mIsShowing = false;
            }
        };

        initContentLogicLoader();

        // Has been initialised
        if (KidozSDK.isInitialised() == true)
        {
            checkIfValidationNeededAndInit();
        } else
        {
            if (EventBus.getDefault().isRegistered(this) == false)
            {
                EventBus.getDefault().register(this);
            }
        }
    }

    /**
     * Check if needed a validation
     */
    private void checkIfValidationNeededAndInit()
    {
        if (mOncePerSessionLoaded == true)
        {
            mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
            if (mJsonObjectProperties != null)
            {
                if (mShowWhenInitiated)
                {
                    showView();
                }

                EventBus.getDefault().post(new EventMessage(EventMessage.MessageType.INIT_FEED_BUTTON));

                if (mOnInterstitialViewEventListener != null)
                {
                    mOnInterstitialViewEventListener.onViewReady();
                }
            } else
            {
                validateLayoutAndAssets();
            }
        } else
        {
            validateLayoutAndAssets();
        }
    }

    // Event bus callback
    public void onEvent(EventMessage event)
    {
        if (event.getMessageType() == EventMessage.MessageType.INIT_SDK)
        {
            checkIfValidationNeededAndInit();
            unregisterEventBus();
        }
    }

    private void unregisterEventBus()
    {
        if (EventBus.getDefault().isRegistered(this))
        {
            EventBus.getDefault().unregister(this);
        }
    }

    private void validateLayoutAndAssets()
    {
        SdkAPIManager.getSdkApiInstance(mBuilder.mContext).getStyle(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getValue(), new ApiResultCallback<Boolean>()
        {
            @Override
            public void onServerResult(ResultData<?> result)
            {
                // If result success notify wrapper to build the banner.
                if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true)
                {
                    if (result.getData() != null)
                    {
                        boolean isAssetsReady = (Boolean) result.getData();
                        if (isAssetsReady == true)
                        {
                            mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
                            if (mJsonObjectProperties != null)
                            {
                                EventBus.getDefault().post(new EventMessage(EventMessage.MessageType.INIT_FEED_BUTTON));
                                mOncePerSessionLoaded = true;

                                if (mOnInterstitialViewEventListener != null)
                                {
                                    mOnInterstitialViewEventListener.onViewReady();
                                }

                                if (mShowWhenInitiated)
                                {
                                    showView();
                                }
                            }
                        } else
                        {
                            onFailed();
                        }
                    }
                } else
                {
                    onFailed();
                }
            }

            @Override
            public void onFailed()
            {

            }
        });
    }

    /**
     * Initiate  content loader logic ,that handles all the logic for loading content
     **/
    private void initContentLogicLoader()
    {
        mContentLogicLoader = new ContentLogicLoader(new ContentLogicLoader.IOnContentDataReadyCallback()
        {
            @Override
            public void onDataReady(ContentData contentData)
            {
                if (mFeedViewType.equals(FeedViewDialog.TAG))
                {
                    mFeedViewDialog.setContentData(contentData);
                } else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
                {
                    mFamilyViewDialog.setContentData(contentData);
                }

                if (mIsPostLoadData == false)
                {
                    if (mOnInterstitialViewEventListener != null)
                    {
                        mOnInterstitialViewEventListener.onReadyToShow();
                    }
                }

                // Reset The storage of the impressions for the HTML content
                CookieManager.resetStorage(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getStringValue(), StorageLife.WIDGET);
            }

            @Override
            public void onLoadContentFailed()
            {

            }
        });
    }

    /**
     * Show feed interstitial view (Will be shown <b>Before</b> content data is ready)
     **/
    public void showView()
    {
        showViewInner();
    }

    private void showViewInner()
    {
        if (KidozSDK.isInitialised())
        {
            mIsAutomaticInvocation = true;
            mIsPostLoadData = true;

            if (mJsonObjectProperties == null)
            {
                mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
            }

            if (mJsonObjectProperties != null)
            {
                openSimpleDialog();
            } else
            {
                mShowWhenInitiated = true;
            }
        } else
        {
            mShowWhenInitiated = true;
        }
    }

    protected synchronized void openSimpleDialog()
    {
        if (mKidozPlayerEventHelper != null)
        {
            mKidozPlayerEventHelper.register();
        }
        if (mBuilder.mContext != null && mBuilder.mContext instanceof Activity)
        {
            mFeedViewType = mJsonObjectProperties.optString(FeedView.TAG);
            if (mFeedViewType != null)
            {
                if (mFeedViewType.equals(FeedViewDialog.TAG))
                {
                    if (mFeedViewDialog == null || mFeedViewDialog.isShowing() == false)
                    {
                        mFeedViewDialog = new FeedViewDialog(mBuilder.mContext, mOnInterstitialViewEventListener, mIsAutomaticInvocation, mJsonObjectProperties);
                        mFeedViewDialog.setOnDismissListener(dismissListener);
                        mFeedViewDialog.setOnCloseButtonClickListener(new View.OnClickListener()
                        {
                            @Override
                            public void onClick(View view)
                            {
                                dismissView();
                                unregisterEventBus();
                            }
                        });

                        if (mOnInterstitialViewEventListener != null)
                        {
                            mOnInterstitialViewEventListener.onReadyToShow();
                        }

                        mFeedViewDialog.openDialog();

                        if (mJsonObjectProperties != null)
                        {
                            mContentLogicLoader.loadContent(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getStringValue(), mJsonObjectProperties.optString(STYLE_ID));
                        }
                    }
                } else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
                {
                    if (mFamilyViewDialog == null || (mFamilyViewDialog.isShowing() == false && mFamilyViewDialog.isWaitingToOpen() == false))
                    {
                        mFamilyViewDialog = new FeedFamilyViewDialog(mBuilder.mContext, mOnInterstitialViewEventListener, mIsAutomaticInvocation, mJsonObjectProperties);
                        mFamilyViewDialog.setOnDismissListener(dismissListener);
                        mFamilyViewDialog.setOnCloseButtonClickListener(new View.OnClickListener()
                        {
                            @Override
                            public void onClick(View view)
                            {
                                dismissView();
                                unregisterEventBus();
                            }
                        });

                        if (mOnInterstitialViewEventListener != null)
                        {
                            mOnInterstitialViewEventListener.onReadyToShow();
                        }

                        mFamilyViewDialog.openDialog();

                        if (mJsonObjectProperties != null)
                        {
                            mContentLogicLoader.loadContent(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getStringValue(), mJsonObjectProperties.optString(STYLE_ID));
                        }
                    }
                }
                mShowWhenInitiated = false;
            }
        }
    }

    /**
     * Dismiss feed view
     **/
    public void dismissView()
    {
        if (mFeedViewType != null)
        {
            if (mFeedViewType.equals(FeedViewDialog.TAG))
            {
                if (mFeedViewDialog != null)
                {
                    mFeedViewDialog.closeDialog();
                }
            } else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
            {
                if (mFamilyViewDialog != null)
                {
                    mFamilyViewDialog.closeDialog();
                }
            }
        }
    }

    /**
     * Set on view event listener
     *
     * @param listener
     **/
    public void setOnFeedViewEventListener(IOnFeedViewEventListener listener)
    {
        mOnInterstitialViewEventListener = listener;
        if (mFeedViewType != null)
        {
            if (mFeedViewType.equals(FeedViewDialog.TAG))
            {
                if (mFeedViewDialog != null)
                {
                    mFeedViewDialog.setOnInterstitialViewEventListener(mOnInterstitialViewEventListener);
                }
            } else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
            {
                if (mFamilyViewDialog != null)
                {
                    mFamilyViewDialog.setOnInterstitialViewEventListener(mOnInterstitialViewEventListener);
                }
            }
        }
    }

    /**
     * Builder class that handles view initiation
     **/
    public static class Builder
    {
        private Context mContext;

        public Builder(Context context)
        {
            mContext = context;
        }

        public FeedView build()
        {
            return new FeedView(this);
        }
    }
}
