package com.kidoz.sdk.api.picasso_related;

import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Shader;

import com.squareup.imagelib.Transformation;

// enables hardware accelerated rounded corners
// original idea here : http://www.curious-creature.org/2012/12/11/android-recipe-1-image-with-rounded-corners/
public class RoundedTransformation implements Transformation
{
    private final int radius;
    private final int margin;  // dp
    private boolean mRoundTopLeft;
    private boolean mRoundTopRight;
    private boolean mRoundBottomLeft;
    private boolean mRoundBottomRight;
    private String mTransformationTAG = "NA";


    // radius is corner radii in dp
    // margin is the board in dp
    public RoundedTransformation(final int radius, final int margin)
    {
        this.radius = radius;
        this.margin = margin;
    }

    public RoundedTransformation(final int radius, final int margin, boolean roundTopLeft)
    {
        this.radius = radius;
        this.margin = margin;
        mRoundTopLeft = roundTopLeft;
    }


    @Override public Bitmap transform(final Bitmap source)
    {
        Bitmap result;
        if (mRoundTopLeft == false)
        {
            result = roundAllCorners(source);
            mTransformationTAG = "Transformation";
        }
        else
        {
            result = getRoundedCornerBitmap(source, radius, false, true, true, true);
            mTransformationTAG = "Transformation2";
        }
        return result;
    }

    private Bitmap roundAllCorners(final Bitmap source)
    {
        final Paint paint = new Paint();
        paint.setShader(new BitmapShader(source, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP));
        Bitmap output = Bitmap.createBitmap(source.getWidth(), source.getHeight(), Config.ARGB_8888);
        Canvas canvas = new Canvas(output);
        canvas.drawRoundRect(new RectF(margin, margin, source.getWidth() - margin, source.getHeight() - margin), radius, radius, paint);
        if (source != output)
        {
            source.recycle();
        }
        return output;
    }

    private Bitmap getRoundedCornerBitmap(final Bitmap source, int radius, boolean squareTL, boolean squareTR, boolean squareBL, boolean squareBR)
    {
        int w = source.getWidth();
        int h = source.getHeight();
        Bitmap output = Bitmap.createBitmap(w, h, Config.ARGB_8888);
        Canvas canvas = new Canvas(output);

        final int color = 0xff424242;
        final Paint paint = new Paint();
        final Rect rect = new Rect(0, 0, w, h);
        final RectF rectF = new RectF(rect);

        //make sure that our rounded corner is scaled appropriately
        final float roundPx = radius;

        paint.setAntiAlias(true);
        canvas.drawARGB(0, 0, 0, 0);
        paint.setColor(color);
        canvas.drawRoundRect(rectF, roundPx, roundPx, paint);

        //draw rectangles over the corners we want to be square
        if (squareTL)
        {
            canvas.drawRect(0, 0, w / 2, h / 2, paint);
        }
        if (squareTR)
        {
            canvas.drawRect(w / 2, 0, w, h / 2, paint);
        }
        if (squareBL)
        {
            canvas.drawRect(0, h / 2, w / 2, h, paint);
        }
        if (squareBR)
        {
            canvas.drawRect(w / 2, h / 2, w, h, paint);
        }

        paint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.SRC_IN));
        canvas.drawBitmap(source, 0, 0, paint);
        if (source != output)
        {
            source.recycle();
        }
        return output;
    }

    @Override public String key()
    {
        return mTransformationTAG;
    }
}