package com.kidoz.sdk.api;

import android.animation.Animator;
import android.animation.AnimatorSet;
import android.animation.TimeInterpolator;
import android.animation.ValueAnimator;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Color;
import android.graphics.Point;
import android.graphics.Rect;
import android.os.Looper;
import android.os.Message;
import android.support.v4.view.MotionEventCompat;
import android.support.v4.view.animation.FastOutSlowInInterpolator;
import android.support.v4.view.animation.LinearOutSlowInInterpolator;
import android.util.AttributeSet;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.View;
import android.view.animation.AnticipateInterpolator;
import android.view.animation.BounceInterpolator;
import android.view.animation.OvershootInterpolator;
import android.widget.FrameLayout;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.AssetUtil;
import com.kidoz.sdk.api.general.utils.SdkSoundManager;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.interfaces.FlexiViewListener;
import com.kidoz.sdk.api.interfaces.KidozPlayerEventHelper;
import com.kidoz.sdk.api.interfaces.KidozPlayerListener;
import com.kidoz.sdk.api.picasso_related.PicassoOk;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.kidoz.sdk.api.structure.ContentData;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.flexi_view.FLEXI_POSITION;
import com.kidoz.sdk.api.ui_views.flexi_view.MovableView;
import com.squareup.imagelib.Callback;

import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;

import de.greenrobot.event.EventBus;

/**
 * Created by Eugine on 1/17/2016.
 */
public class FlexiView extends FrameLayout {
    public static final String TAG = FlexiView.class.getSimpleName();

    private static final int INVALID_POINTER_ID = -1;
    private static final long MOVE_TO_EDGE_DURATION = 400L;
    private static final float MOVE_TO_EDGE_OVERSHOOT_TENSION = 1.0f;
    private static final long NOTICE_DEFAULT_ANIMATION_CYCLE_INTERVAL = 2500;
    private static final int CONTENT_DEFAULT_REFRESH_RATE_SECONDS = 25;

    private static final int START_NOTICE_ME_ANIMATION = 0;
    private static final int STOP_NOTICE_ME_ANIMATION = 1;

    private enum NoticeAnimType {
        PULSE(0), SWING(1), BOUNCE(2), FLASH(3), TADA(4);
        private int id;

        NoticeAnimType(int id) {
            this.id = id;
        }
    }

    private JSONObject mProperties;
    private FlexiViewListener mFlexiViewListener;

    private TimeInterpolator mMoveEdgeInterpolator;
    private TimeInterpolator mFlingMoveLinearInterpolator;
    private AnimatorSet mFlingAnimation;

    private boolean mIsSelected = false;
    private boolean mIsDraggable = true;

    //Reflect the by user changed  properties
    private int mUserSetDragable = -1;
    private int mUserSetClosable = -1;

    private float mLastTouchX;
    private float mLastTouchY;
    private int mSingleFingerActivePointerId;

    private Rect mContainerViewRect = new Rect();
    private Rect mFlexiViewPosRect = new Rect();

    // The movable view
    private MovableView mMovableView;

    private ValueAnimator mMoveEdgeAnimatorX;
    private ValueAnimator mMoveEdgeAnimatorY;

    private ContentLogicLoader mContentLogicLoader;
    private String mStyleId;

    private ArrayList<ContentItem> mShowingListContentList = new ArrayList<ContentItem>();


    private int mCurrentShowingItemIndex = 0;
    private boolean mIsCyclingContent = false;

    private GestureDetector mGestureDetector;
    private Utils.StaticHandler mStaticHandler;

    // Used to disable false flings when dragging the movable view
    private boolean mAllowFling = true;
    private boolean mAllowClickHandling = true;
    // Flag to verify that the view is visible and attached
    // To continue view initiation
    private boolean mAllowViewInitiation = false;
    // To show the view on create automatically
    private boolean mIsAutoShowOnCreate = false;

    //Notice animation definitions
    private Utils.StaticHandler mAnimationHandler;
    private AnimatorSet mNoticeAnimationSet;
    private boolean mIsAnimateNotice = false;
    private NoticeAnimType mNoticeAnimType;

    private long mNoticeAnimCycleRate = NOTICE_DEFAULT_ANIMATION_CYCLE_INTERVAL;
    private long mContentRefreshRate = CONTENT_DEFAULT_REFRESH_RATE_SECONDS;

    private FLEXI_POSITION mFlexiPosition = FLEXI_POSITION.TOP_START;
    private Point mStartPointCoord = new Point(0, 0);
    private KidozPlayerEventHelper mKidozPlayerEventHelper;

    public FlexiView(Context context) {
        super(context);
        initView();
    }

    public FlexiView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initView();
    }

    public FlexiView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initView();
    }

    /**
     * Set kidoz players event listeners
     * Used for listening Open\Close and ect.. event of launch Player for single view
     *
     * @param kidozPlayerListener event listener
     */
    public void setKidozPlayerListener(KidozPlayerListener kidozPlayerListener) {
        mKidozPlayerEventHelper = new KidozPlayerEventHelper(kidozPlayerListener, WidgetType.WIDGET_TYPE_FLEXI_VIEW);
    }

    private void initView() {
        // We set a touch of color to force redraw of the whole view in some cases when the tranparent part is not being invalidated
        setBackgroundColor(Color.parseColor("#01ffffff"));

        EventBus.getDefault().register(this);
        // Initiate interpolations
        mMoveEdgeInterpolator = new OvershootInterpolator(MOVE_TO_EDGE_OVERSHOOT_TENSION);
        mFlingMoveLinearInterpolator = new LinearOutSlowInInterpolator();

        initContentLoader();
        initGestureListener();

        // We check if SDK initiation has successfully finished( Sdk Config data accepted)
        if (KidozSDK.isInitialised() == true) {
            EventBus.getDefault().unregister(this);
            validateLayoutAndAssets();
        }
    }

    // Event bus callback
    public void onEvent(EventMessage event) {
        if (event.getMessageType() == EventMessage.MessageType.INIT_SDK) {
            EventBus.getDefault().unregister(this);
            validateLayoutAndAssets();
        }
    }


    /**
     * Validate layout and load assets for the view
     */
    private void validateLayoutAndAssets() {
        mAllowViewInitiation = true;
        SdkAPIManager.getSdkApiInstance(getContext()).getStyle(getContext(), WidgetType.WIDGET_TYPE_FLEXI_VIEW.getValue(), new ApiResultCallback<Boolean>() {
            @Override
            public void onServerResult(ResultData<?> result) {
                if (mAllowViewInitiation) {
                    // If result success notify wrapper to build the banner.
                    if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true) {
                        if (result.getData() != null) {
                            boolean isAssetsReady = (Boolean) result.getData();
                            if (isAssetsReady == true) {

                                mProperties = DatabaseManager.getInstance(getContext()).getConfigTable().loadProperties(FlexiView.TAG);
                                if (mProperties != null) {

                                    mNoticeAnimCycleRate = mProperties.optLong(FlexiView.NOTICE_ME_ANIMATION_RATE_MILLIS, NOTICE_DEFAULT_ANIMATION_CYCLE_INTERVAL);
                                    mContentRefreshRate = mProperties.optInt(FlexiView.CONTENT_REFRESH_RATE_SECONDS, CONTENT_DEFAULT_REFRESH_RATE_SECONDS) * 1000;

                                    File soundFile = AssetUtil.getAssetFile(getContext(), mProperties.optString(FlexiView.FLEXI_POP_VIEW_SOUND, null));
                                    if (soundFile != null) {
                                        SdkSoundManager.preloadSound(soundFile.getPath());
                                    }

                                    soundFile = AssetUtil.getAssetFile(getContext(), mProperties.optString(FlexiView.CONTENT_SWAP_SOUND, null));
                                    if (soundFile != null) {
                                        SdkSoundManager.preloadSound(soundFile.getPath());
                                    }

                                    mIsDraggable = mProperties.optInt(FlexiView.IS_FLEXI_VIEW_DRUGGABLE, 1) == 1 ? true : false;
                                    boolean closable = mProperties.optInt(FlexiView.IS_FLEXI_VIEW_CLOSABLE, 1) == 1 ? true : false;

                                    if (mProperties.optInt(FlexiView.IS_OVERRIDE_USER_SET, 0) == 0) {
                                        if (mUserSetDragable != -1) {
                                            mIsDraggable = mUserSetDragable == 1 ? true : false;
                                        }
                                        if (mUserSetClosable != -1) {
                                            closable = mUserSetClosable == 1 ? true : false;
                                        }
                                    }

                                    initMovableView(mProperties, closable);
                                    initNoticeAnimation(mProperties);
                                }
                            } else {
                                onFailed();
                            }
                        }
                    } else {
                        onFailed();
                    }
                }
            }

            @Override
            public void onFailed() {
            }
        });
    }

    /**
     * Initiate movable flexi view
     */
    private void initMovableView(JSONObject mProperties, boolean closable) {
        mStyleId = mProperties.optString(STYLE_ID);
        // Initiate movable flexi point view
        mMovableView = new MovableView(getContext(), mProperties, new MovableView.IMovableActionListener() {
            @Override
            public void onCloseClick() {

                hideFlexiView();

                invokeNoticeAnimationEvent(STOP_NOTICE_ME_ANIMATION);

                EventManager.getInstance(getContext()).logEvent(getContext(), WidgetType.WIDGET_TYPE_FLEXI_VIEW.getStringValue(), mStyleId, EventManager.LOG_NORMAL_LEVEL, null, EventParameters.CATEGORY_SDK, EventParameters.ACTION_CLICK, EventParameters.LABEL_CLOSE_BUTTON, false);
            }
        });
        mMovableView.showCloseBtn(closable);

        FrameLayout.LayoutParams mViewParams = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.WRAP_CONTENT);
        addView(mMovableView, mViewParams);

        //Using global layout to find the real size of movable flexi view and to be sure that it finished creating the views
        Utils.setOnGlobalLayoutFinishListener(mMovableView, new Utils.OnGlobalLayoutFinishedListener() {
            @Override
            public void onLayoutFinished() {
                mMovableView.bringToFront();

                mFlexiViewPosRect = new Rect(0, 0, mMovableView.getWidth(), mMovableView.getWidth());
                mMovableView.getLayoutParams().width = mMovableView.getWidth();
                mMovableView.getLayoutParams().height = mMovableView.getWidth();

                //Load all secondary used assets fo flexi view (X button, parental lock and etc)
                mMovableView.loadSecondaryAssets(new MovableView.IMovableAssetReadyListener() {
                    @Override
                    public void onAssetsReady() {
                        //Load flexi view content
                        mContentLogicLoader.loadContent(getContext(), WidgetType.WIDGET_TYPE_FLEXI_VIEW.getStringValue(), mStyleId);
                    }
                });
            }
        });
    }

    /**
     * Initiate content loader that responsible for loading content item data
     */
    private void initContentLoader() {
        mContentLogicLoader = new ContentLogicLoader(new ContentLogicLoader.IOnContentDataReadyCallback() {
            @Override
            public void onDataReady(ContentData contentData) {
                if (contentData != null && contentData.isHasContentToShow()) {

                    mShowingListContentList = new ArrayList<ContentItem>();
                    for (int i = 0; i < contentData.getContentDataItems().size(); i++) {
                        if (contentData.getContentDataItems().get(i).isPromoted()) {
                            mShowingListContentList.add(contentData.getContentDataItems().get(i));
                        }
                    }

                    if (mShowingListContentList.isEmpty()) {
                        mShowingListContentList.addAll(contentData.getContentDataItems());
                    }

                    if (mMovableView != null) {
                        //Load received content item thumbnail view
                        mMovableView.loadMainAsset(mShowingListContentList.get(mCurrentShowingItemIndex).getThumb(), new MovableView.IMovableAssetReadyListener() {
                            @Override
                            public void onAssetsReady() {

                                if (mFlexiViewListener != null) {
                                    mFlexiViewListener.onViewReady();
                                }
                                //If all loaded successfully show the flexi view on screen
                                if (mIsAutoShowOnCreate) {
                                    showFlexiView();
                                }
                            }
                        });
                    }
                }
            }

            @Override
            public void onLoadContentFailed() {
            }
        });
    }

    /**
     * Start content changing cycle .that will change content item , for some predefined interval
     */
    private void startContentChangingCycle() {
        getMyHandlerInstance().removeCallbacksAndMessages(null);
        if (!mShowingListContentList.isEmpty() && mIsCyclingContent == true) {
            if (getWindowVisibility() == View.VISIBLE && mMovableView != null && mMovableView.getVisibility() == View.VISIBLE) {
                getMyHandlerInstance().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        if (mIsCyclingContent == true) {
                            int nextItemIndex = mCurrentShowingItemIndex + 1;
                            if (mShowingListContentList.size() - 1 < nextItemIndex) {
                                nextItemIndex = 0;
                            }

                            ContentItem contentItem = mShowingListContentList.get(nextItemIndex);
                            if (contentItem != null) {
                                if (mMovableView != null) {
                                    final int finalNextItemIndex = nextItemIndex;
                                    // Preload image to memory first, if success continue otherwise stay on same content loaded before
                                    PicassoOk.getPicasso(getContext()).load(contentItem.getThumb()).transform(mMovableView.mImageTransformation).fetch(new Callback() {
                                        @Override
                                        public void onSuccess() {
                                            if (getWindowVisibility() == View.VISIBLE) {
                                                mCurrentShowingItemIndex = finalNextItemIndex;
                                                //Load received content item thumbnail view
                                                mMovableView.loadMainAsset(mShowingListContentList.get(mCurrentShowingItemIndex).getThumb(), new MovableView.IMovableAssetReadyListener() {
                                                    @Override
                                                    public void onAssetsReady() {
                                                        // Send impression log
                                                        sendImpressionLog(mShowingListContentList.get(mCurrentShowingItemIndex));
                                                        //If all loaded successfully show the flexi view on screen
                                                        File soundFile = AssetUtil.getAssetFile(getContext(), mProperties.optString(FlexiView.CONTENT_SWAP_SOUND, null));
                                                        if (soundFile != null) {
                                                            SdkSoundManager.playSound(getContext(), soundFile.getPath());
                                                        }

                                                        startContentChangingCycle();
                                                    }
                                                });
                                            }
                                        }

                                        @Override
                                        public void onError() {
                                            if (getWindowVisibility() == View.VISIBLE) {
                                                startContentChangingCycle();
                                            }
                                        }
                                    });

                                }

                            }
                        }
                    }
                }, mContentRefreshRate);
            }
        }
    }


    /**
     * Set automatically show flexi view when ready
     *
     * @param position
     */
    public void setFlexiViewInitialPosition(FLEXI_POSITION position) {
        mFlexiPosition = position;
    }

    /**
     * Set automatically show flexi view when ready
     *
     * @param autoShow
     */
    public void setAutoShow(boolean autoShow) {
        mIsAutoShowOnCreate = autoShow;
    }

    /**
     * Set Flexi View draggable, can be dragged by user all around the screen
     *
     * @param isDraggable
     */
    public void setDraggable(boolean isDraggable) {

        if (isDraggable) {
            mUserSetDragable = 1;
        } else {
            mUserSetDragable = 0;
        }

        if (mProperties != null && mProperties.optInt(IS_OVERRIDE_USER_SET, 0) == 0) {
            mIsDraggable = isDraggable;
        }
    }

    /**
     * Set Flexi View closable , can be closed by user action
     *
     * @param isClosable
     */
    public void setClosable(boolean isClosable) {
        if (isClosable) {
            mUserSetClosable = 1;
        } else {
            mUserSetClosable = 0;
        }

        if (mProperties != null && mProperties.optInt(IS_OVERRIDE_USER_SET, 0) == 0) {

            if (mMovableView != null) {
                mMovableView.showCloseBtn(isClosable);
            }
        }
    }


    /**
     * Show flexi view
     */
    public void showFlexiView() {
        if (mMovableView != null && !mShowingListContentList.isEmpty() && mMovableView.getVisibility() != View.VISIBLE) {
            GenAnimator.playGrowAnimation(mMovableView, 700, 0, new BounceInterpolator(), new Animator.AnimatorListener() {
                @Override
                public void onAnimationStart(Animator animation) {
                    // On showing movable view calculate the correct position by the gravity definition
                    calculateStartPositionCoordinates();
                    mFlexiViewPosRect = new Rect(mStartPointCoord.x, mStartPointCoord.y, mStartPointCoord.x + mMovableView.getWidth(), mStartPointCoord.y + mMovableView.getHeight());
                    mMovableView.setTranslationX(mStartPointCoord.x);
                    mMovableView.setTranslationY(mStartPointCoord.y);

                    if (getWindowVisibility() == View.VISIBLE) {
                        mMovableView.setVisibility(View.VISIBLE);
                        requestFocus();
                        bringToFront();
                        mMovableView.bringToFront();

                        // Play pop sound
                        File soundFile = AssetUtil.getAssetFile(getContext(), mProperties.optString(FlexiView.FLEXI_POP_VIEW_SOUND, null));
                        if (soundFile != null) {
                            SdkSoundManager.playSound(getContext(), soundFile.getAbsolutePath());
                        }

                        setBackgroundColor(Color.TRANSPARENT);

                        //Widget view
                        EventManager.getInstance(getContext()).logEvent(getContext(), WidgetType.WIDGET_TYPE_FLEXI_VIEW.getStringValue(), mStyleId, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_WIDGET_VIEW, EventParameters.LABEL_FLEXI_VIEW, false);
                    }
                }

                @Override
                public void onAnimationEnd(Animator animation) {
                    if (mMovableView.getVisibility() == View.VISIBLE) {
                        // Start content cycling if possible
                        setViewVisibleAndActive();

                        if (mFlexiViewListener != null) {
                            mFlexiViewListener.onViewVisible();
                        }

                        // Send impression log
                        sendImpressionLog(mShowingListContentList.get(mCurrentShowingItemIndex));
                    }
                }

                @Override
                public void onAnimationCancel(Animator animation) {
                }

                @Override
                public void onAnimationRepeat(Animator animation) {
                }
            });
        }
    }


    /**
     * Hide flexi view
     */
    public void hideFlexiView() {
        if (mMovableView != null && !mShowingListContentList.isEmpty() && mMovableView.getVisibility() == View.VISIBLE) {
            GenAnimator.playShrinkAnimation(mMovableView, 470, new AnticipateInterpolator(), new Animator.AnimatorListener() {
                @Override
                public void onAnimationStart(Animator animation) {
                }

                @Override
                public void onAnimationEnd(Animator animation) {
                    mMovableView.setVisibility(View.INVISIBLE);
                    setViewNotVisibleOrNotActive();

                    if (mFlexiViewListener != null) {
                        mFlexiViewListener.onViewHidden();
                    }
                }

                @Override
                public void onAnimationCancel(Animator animation) {
                }

                @Override
                public void onAnimationRepeat(Animator animation) {
                }
            });
        }
    }

    /**
     * Get flexi view visibility
     *
     * @return visibility
     */
    public boolean getIsFlexiViewVisible() {
        if (mMovableView != null && mMovableView.getVisibility() != View.VISIBLE) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * Initiate gesture detector to catch click on content item image.
     * Used gesture detector because of the limitation of the onTouchEvent
     * and the inability to use onClick listener on content item image
     */
    private void initGestureListener() {
        mGestureDetector = new GestureDetector(new GestureListener(new GestureListener.IOnGestureEventsListener() {
            @Override
            public void onClick() {
                // Play click animation and handle click on content item
                GenAnimator.playClickAnimation(mMovableView, new Animator.AnimatorListener() {
                    @Override
                    public void onAnimationStart(Animator animation) {
                    }

                    @Override
                    public void onAnimationEnd(Animator animation) {
                        handleClick(mShowingListContentList.get(mCurrentShowingItemIndex));
                    }

                    @Override
                    public void onAnimationCancel(Animator animation) {
                    }

                    @Override
                    public void onAnimationRepeat(Animator animation) {
                    }
                });
            }

            @Override
            public void onFling(float distX, float distY) {
                if (mIsDraggable) {
                    if (mAllowFling) {
                        animateFling(distX, distY);
                    } else {
                        invokeNoticeAnimationEvent(START_NOTICE_ME_ANIMATION);
                    }
                }
            }
        }));
    }

    /**
     * Handle content item click
     */
    private void handleClick(ContentItem contentItem) {
        if (contentItem != null && mAllowClickHandling) {
            //If content item from type promoted
            if (mAllowClickHandling) {
                mAllowClickHandling = false;
                ContentExecutionHandler.handleContentItemClick(getContext(), contentItem, WidgetType.WIDGET_TYPE_FLEXI_VIEW.getStringValue(), mStyleId, 0, true, false, new ContentExecutionHandler.IOnHandleClickListener() {
                    @Override
                    public void onRestoreClick() {
                        mAllowClickHandling = true;
                    }
                });
            }
        }
    }

    /**
     * On view visible start animations and enable view interactions
     */
    private void setViewVisibleAndActive() {
        if(getVisibility() == View.VISIBLE) {
            if(mMovableView != null && mMovableView.getVisibility() == VISIBLE) {
                mIsCyclingContent = true;
                mAllowViewInitiation = true;
                mAllowClickHandling = true;

                invokeNoticeAnimationEvent(START_NOTICE_ME_ANIMATION);
                startContentChangingCycle();
            }
        }
    }


    /**
     * On view detached or invisible stop all animations and disable view interactions
     */
    private void setViewNotVisibleOrNotActive() {
        mAllowViewInitiation = false;
        mIsCyclingContent = false;
        mAllowClickHandling = false;

        if (mFlingAnimation != null) {
            mFlingAnimation.cancel();
        }
        invokeNoticeAnimationEvent(STOP_NOTICE_ME_ANIMATION);

        getMyHandlerInstance().removeCallbacksAndMessages(null);
    }

    @Override
    protected void onWindowVisibilityChanged(int visibility) {
        super.onWindowVisibilityChanged(visibility);

        if (visibility == View.VISIBLE) {
            setViewVisibleAndActive();
        } else if (visibility == View.GONE || visibility == View.INVISIBLE) {
            setViewNotVisibleOrNotActive();
            // Start sync log events
            EventManager.getInstance(getContext()).startEventsSync(getContext());
        }
    }


    @Override
    public void onWindowFocusChanged(boolean hasWindowFocus) {
        super.onWindowFocusChanged(hasWindowFocus);

        if (hasWindowFocus) {
            setViewVisibleAndActive();

        } else {
            setViewNotVisibleOrNotActive();
        }
    }

    @Override
    protected void onDetachedFromWindow() {
        if (mMoveEdgeAnimatorX != null) {
            mMoveEdgeAnimatorX.removeAllUpdateListeners();
        }

        if (mMoveEdgeAnimatorY != null) {
            mMoveEdgeAnimatorY.removeAllUpdateListeners();
        }

        if (mFlingAnimation != null) {
            mFlingAnimation.cancel();
        }

        setViewNotVisibleOrNotActive();

        getMyHandlerInstance().removeCallbacksAndMessages(null);
        if (mKidozPlayerEventHelper != null) {
            mKidozPlayerEventHelper.unRegister();
        }

        super.onDetachedFromWindow();
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        if (mKidozPlayerEventHelper != null) {
            mKidozPlayerEventHelper.register();
        }

        if (getLayoutParams() != null) {
            boolean relayout = false;
            if (getLayoutParams().height != LayoutParams.MATCH_PARENT) {
                getLayoutParams().height = LayoutParams.MATCH_PARENT;
                relayout = true;
            }
            if (getLayoutParams().width != LayoutParams.MATCH_PARENT) {
                getLayoutParams().width = LayoutParams.MATCH_PARENT;
                relayout = true;
            }

            if (relayout) {
                requestLayout();
                postInvalidate();
                bringToFront();
            }
        }
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        return handleTouchEvent(event);
    }

    private boolean handleTouchEvent(MotionEvent event) {

        int maskAction = MotionEventCompat.getActionMasked(event);
        if(mMovableView != null) {
            if (mIsDraggable) {

                mGestureDetector.onTouchEvent(event);
                if (maskAction == MotionEvent.ACTION_DOWN) {
                    if (mMovableView.getVisibility() == View.VISIBLE) {
                        invokeNoticeAnimationEvent(STOP_NOTICE_ME_ANIMATION);

                        if (mFlingAnimation != null) {
                            mFlingAnimation.cancel();
                        }

                        if (event.getPointerCount() <= 1) {
                            final int pointerIndex = MotionEventCompat.getActionIndex(event);
                            mLastTouchX = MotionEventCompat.getX(event, pointerIndex);
                            mLastTouchY = MotionEventCompat.getY(event, pointerIndex);

                            // Save the ID of this single touch pointer pointer
                            mSingleFingerActivePointerId = event.getPointerId(pointerIndex);

                            //Check if the touch point is inside the flexi movable container
                            if (mFlexiViewPosRect.contains((int) mLastTouchX, (int) mLastTouchY)) {
                                mIsSelected = true;
                                // Wen the interaction with movable view if started we want to pause the gif animation if exits
                                mMovableView.playOrStopTheGifAnimationIfExists(false);
                            }
                        }
                    } else {
                        return false;
                    }
                } else if (maskAction == MotionEvent.ACTION_MOVE) {
                    if (mMovableView.getVisibility() == View.VISIBLE) {
                        if (mIsDraggable && mIsSelected) {
                            if (event.getPointerCount() <= 1) {
                                int pointerIndex = MotionEventCompat.findPointerIndex(event, mSingleFingerActivePointerId);

                                if (pointerIndex != -1) {
                                    final float x = MotionEventCompat.getX(event, pointerIndex);
                                    final float y = MotionEventCompat.getY(event, pointerIndex);

                                    final int dx = (int) (x - mLastTouchX);
                                    final int dy = (int) (y - mLastTouchY);

                                    mLastTouchX = x;
                                    mLastTouchY = y;

                                    //Update flexi movable view position rectangle
                                    mFlexiViewPosRect.top = mFlexiViewPosRect.top + dy;
                                    mFlexiViewPosRect.left = mFlexiViewPosRect.left + dx;
                                    mFlexiViewPosRect.bottom = mFlexiViewPosRect.bottom + dy;
                                    mFlexiViewPosRect.right = mFlexiViewPosRect.right + dx;

                                    //Update and invalidate flexi movable view position on screen
                                    mMovableView.setTranslationX(mFlexiViewPosRect.left);
                                    mMovableView.setTranslationY(mFlexiViewPosRect.top);

                                    // Need to check the minimum moved distance to not make false fling animation
                                    if (Math.max(Math.abs(dx), Math.abs(dy)) > 10) {
                                        mAllowFling = true;
                                    } else {
                                        mAllowFling = false;
                                    }
                                }
                            }
                        }
                    } else {
                        return false;
                    }
                } else if (maskAction == MotionEvent.ACTION_UP || maskAction == MotionEvent.ACTION_CANCEL) {
                    mSingleFingerActivePointerId = INVALID_POINTER_ID;
                    mIsSelected = false;

                    //Check if need fixing movable view edge bounds
                    if (mContainerViewRect.contains(mFlexiViewPosRect) == false) {
                        fixLeftRightEdgeAlignmentIfNeeded();
                        fixTopBottomEdgeAlignmentIfNeeded();
                    }

                    invokeNoticeAnimationEvent(START_NOTICE_ME_ANIMATION);

                    if (mMovableView.getVisibility() == View.VISIBLE) {
                        // Wen the interaction with movable view if finished we want to resume the gif animation if exits
                        if (mAllowFling == false) {
                            mMovableView.playOrStopTheGifAnimationIfExists(true);
                        }
                    }
                }
            } else {// The view is not set as druggable but we still want to allow click on the content item
                if (mMovableView.getVisibility() == View.VISIBLE) {
                    if (event.getPointerCount() <= 1) {
                        final int pointerIndex = MotionEventCompat.getActionIndex(event);
                        mLastTouchX = MotionEventCompat.getX(event, pointerIndex);
                        mLastTouchY = MotionEventCompat.getY(event, pointerIndex);

                        //Check if the touch point is inside the flexi movable container
                        if (mFlexiViewPosRect.contains((int) mLastTouchX, (int) mLastTouchY)) {
                            mGestureDetector.onTouchEvent(event);
                            return true;
                        }
                    }
                } else {
                    return false;
                }
            }
        }
        return mIsSelected;
    }

    /**
     * Fix left or right exit from view boundaries
     */
    private void fixLeftRightEdgeAlignmentIfNeeded() {
        int currentX = 0;
        int goalPositionX = 0;

        boolean animateX = false;

        if (mFlexiViewPosRect.left < 0) {
            animateX = true;
            currentX = mFlexiViewPosRect.left;
            goalPositionX = 0;
        } else if (mFlexiViewPosRect.right > mContainerViewRect.right) {
            animateX = true;
            currentX = mFlexiViewPosRect.left;
            goalPositionX = mContainerViewRect.right - mMovableView.getWidth();
        }

        if (animateX) {
            mMoveEdgeAnimatorX = ValueAnimator.ofInt(currentX, goalPositionX);
            mMoveEdgeAnimatorX.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
                @Override
                public void onAnimationUpdate(ValueAnimator animation) {

                    int translate = (Integer) animation.getAnimatedValue();
                    mFlexiViewPosRect.left = translate;
                    mFlexiViewPosRect.right = mMovableView.getWidth() + translate;
                    mMovableView.setTranslationX(mFlexiViewPosRect.left);
                }
            });
            mMoveEdgeAnimatorX.setInterpolator(mMoveEdgeInterpolator);
            mMoveEdgeAnimatorX.setDuration(MOVE_TO_EDGE_DURATION);
            mMoveEdgeAnimatorX.start();
        }
    }

    /**
     * Fix top or bottom exit from view boundaries
     */
    private void fixTopBottomEdgeAlignmentIfNeeded() {

        int currentY = 0;
        int goalPositionY = 0;

        boolean animateY = false;
        if (mFlexiViewPosRect.top < 0) {
            animateY = true;
            currentY = mFlexiViewPosRect.top;
            goalPositionY = 0;
        } else if (mFlexiViewPosRect.bottom > mContainerViewRect.bottom) {
            animateY = true;
            currentY = mFlexiViewPosRect.top;
            goalPositionY = mContainerViewRect.bottom - mMovableView.getHeight();
        }

        if (animateY) {
            mMoveEdgeAnimatorY = ValueAnimator.ofInt(currentY, goalPositionY);
            mMoveEdgeAnimatorY.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
                @Override
                public void onAnimationUpdate(ValueAnimator animation) {

                    int translate = (Integer) animation.getAnimatedValue();
                    mFlexiViewPosRect.top = translate;
                    mFlexiViewPosRect.bottom = mMovableView.getHeight() + translate;
                    mMovableView.setTranslationY(mFlexiViewPosRect.top);
                }
            });
            mMoveEdgeAnimatorY.setDuration(MOVE_TO_EDGE_DURATION);
            mMoveEdgeAnimatorY.setInterpolator(mMoveEdgeInterpolator);
            mMoveEdgeAnimatorY.start();
        }
    }


    /**
     * Gesture listener to catch the click on the content item
     */
    private static final class GestureListener extends GestureDetector.SimpleOnGestureListener {

        private static final int SWIPE_THRESHOLD = 100;
        private static final int SWIPE_VELOCITY_THRESHOLD = 130;
        IOnGestureEventsListener mEventListener;

        public GestureListener(IOnGestureEventsListener eventListener) {
            mEventListener = eventListener;
        }

        @Override
        public boolean onDown(MotionEvent e) {
            return true;
        }

        @Override
        public boolean onSingleTapUp(MotionEvent e) {
            mEventListener.onClick();
            return true;
        }

        @Override
        public boolean onDoubleTap(MotionEvent e) {
            return true;
        }

        @Override
        public boolean onFling(MotionEvent e1, MotionEvent e2, float velocityX, float velocityY) {

            if (Math.max(Math.abs(velocityX), Math.abs(velocityY)) > SWIPE_VELOCITY_THRESHOLD) {

                float distX = e2.getX() - e1.getX();
                float distY = e2.getY() - e1.getY();

                if (Math.max(Math.abs(distX), Math.abs(distY)) > SWIPE_THRESHOLD) {
                    mEventListener.onFling(distX, distY);
                }
            }
            return false;
        }

        public interface IOnGestureEventsListener {
            public void onClick();

            public void onFling(float distX, float distY);
        }
    }

    /**
     * Animation fling motion effect
     */
    private void animateFling(double distX, double distY) {
        if (mFlingAnimation != null) {
            mFlingAnimation.cancel();
        }

        if (mMovableView != null) {
            mFlingAnimation = new AnimatorSet();

            ValueAnimator xAnimator = ValueAnimator.ofInt(mFlexiViewPosRect.left, (int) (mFlexiViewPosRect.left + distX));
            xAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
                @Override
                public void onAnimationUpdate(ValueAnimator animation) {

                    int translate = (Integer) animation.getAnimatedValue();
                    mFlexiViewPosRect.left = translate;
                    mFlexiViewPosRect.right = mMovableView.getWidth() + translate;
                    mMovableView.setTranslationX(mFlexiViewPosRect.left);
                }
            });
            xAnimator.setDuration(300);
            xAnimator.setInterpolator(mFlingMoveLinearInterpolator);

            ValueAnimator yAnimator = ValueAnimator.ofInt(mFlexiViewPosRect.top, (int) (mFlexiViewPosRect.top + distY));
            yAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
                @Override
                public void onAnimationUpdate(ValueAnimator animation) {

                    int translate = (Integer) animation.getAnimatedValue();
                    mFlexiViewPosRect.top = translate;
                    mFlexiViewPosRect.bottom = mMovableView.getHeight() + translate;
                    mMovableView.setTranslationY(mFlexiViewPosRect.top);
                }
            });
            yAnimator.setDuration(300);
            yAnimator.setInterpolator(mFlingMoveLinearInterpolator);

            mFlingAnimation.playTogether(xAnimator, yAnimator);
            mFlingAnimation.addListener(new Animator.AnimatorListener() {
                @Override
                public void onAnimationStart(Animator animation) {
                }

                @Override
                public void onAnimationEnd(Animator animation) {
                    //Check if need fixing movable view edge bounds
                    if (mContainerViewRect.contains(mFlexiViewPosRect) == false) {
                        fixLeftRightEdgeAlignmentIfNeeded();
                        fixTopBottomEdgeAlignmentIfNeeded();
                    }

                    mMovableView.playOrStopTheGifAnimationIfExists(true);
                }

                @Override
                public void onAnimationCancel(Animator animation) {
                    //Check if need fixing movable view edge bounds
                    if (mContainerViewRect.contains(mFlexiViewPosRect) == false) {
                        fixLeftRightEdgeAlignmentIfNeeded();
                        fixTopBottomEdgeAlignmentIfNeeded();
                    }
                }

                @Override
                public void onAnimationRepeat(Animator animation) {
                }
            });
            mFlingAnimation.start();
        }
    }


    /**
     * Initiate animation handler
     */
    private void initNoticeAnimation(final JSONObject mProperties) {

        mNoticeAnimType = NoticeAnimType.values()[mProperties.optInt(FlexiView.NOTICE_ME_ANIMATION_TYPE, 0)];

        if (mAnimationHandler == null) {
            mAnimationHandler = new Utils.StaticHandler(Looper.getMainLooper()) {
                @Override
                public void handleMessage(Message message) {
                    if (message != null) {
                        switch (message.arg1) {
                            case START_NOTICE_ME_ANIMATION: {
                                if (mIsAnimateNotice) {
                                    if (getWindowVisibility() == View.VISIBLE) {
                                        postDelayed(new Runnable() {
                                            @Override
                                            public void run() {
                                                if (mIsAnimateNotice) {
                                                    if (mNoticeAnimationSet == null || mNoticeAnimationSet.isStarted() == false) {
                                                        createAnimationInstance();
                                                        mNoticeAnimationSet.start();
                                                    }
                                                }
                                            }
                                        }, mNoticeAnimCycleRate);
                                    }
                                }
                                break;
                            }
                            case STOP_NOTICE_ME_ANIMATION: {
                                mIsAnimateNotice = false;
                                if (mNoticeAnimationSet != null) {
                                    mNoticeAnimationSet.end();
                                }
                                break;
                            }
                        }
                    }
                }
            };
        }
    }

    /**
     * Create animation instance if needed.
     * (Some animation need to ne recreated due to position changes of the movable view)
     */
    private void createAnimationInstance() {
        boolean isResetVals = true;
        switch (mNoticeAnimType) {
            case PULSE: {
                if (mNoticeAnimationSet == null) {
                    mNoticeAnimationSet = GenAnimator.flexiPulseAnimation(mMovableView, 1000, new FastOutSlowInInterpolator());
                }else {
                    isResetVals = false;
                }
                break;
            }
            case SWING: {
                if (mNoticeAnimationSet == null) {
                    mNoticeAnimationSet = GenAnimator.flexiSwingAnimation(mMovableView, 800, null);
                }else {
                    isResetVals = false;
                }
                break;
            }
            case BOUNCE: {
                if(mNoticeAnimationSet != null) {
                    mNoticeAnimationSet.cancel();
                }
                mNoticeAnimationSet = null;
                mNoticeAnimationSet = GenAnimator.flexiBounceAnimation(mMovableView, 800, null);
                break;
            }
            case FLASH: {
                if (mNoticeAnimationSet == null) {
                    mNoticeAnimationSet = GenAnimator.flexiFlashAnimation(mMovableView, 800, null);
                }else {
                    isResetVals = false;
                }
                break;
            }
            case TADA: {
                if(mNoticeAnimationSet != null) {
                    mNoticeAnimationSet.cancel();
                }
                mNoticeAnimationSet = null;
                mNoticeAnimationSet = GenAnimator.flexiTadaAnimation(mMovableView, 900, null);
                break;
            }
        }


        if(isResetVals) {
            mNoticeAnimationSet.setStartDelay(mNoticeAnimCycleRate);
            mNoticeAnimationSet.removeAllListeners();
            mNoticeAnimationSet.addListener(new Animator.AnimatorListener() {
                @Override
                public void onAnimationStart(Animator animation) {
                }

                @Override
                public void onAnimationEnd(Animator animation) {
                    if (mIsAnimateNotice) {
                        invokeNoticeAnimationEvent(START_NOTICE_ME_ANIMATION);
                    }
                }

                @Override
                public void onAnimationCancel(Animator animation) {
                    mAnimationHandler.removeCallbacksAndMessages(null);
                }

                @Override
                public void onAnimationRepeat(Animator animation) {
                }
            });
        }
    }

    /**
     * Invoke notice animation event
     */
    private void invokeNoticeAnimationEvent(int action) {
        if (mAnimationHandler != null) {
            if (action == START_NOTICE_ME_ANIMATION) {
                mIsAnimateNotice = true;
            } else if (action == STOP_NOTICE_ME_ANIMATION) {
                mIsAnimateNotice = false;
            }
            Message message = Message.obtain();
            message.arg1 = action;
            mAnimationHandler.sendMessage(message);
        }
    }

    private void sendImpressionLog(ContentItem contentItem) {

        if (contentItem != null && contentItem.isPromoted() == true) {
            EventManager.getInstance(getContext()).logSponsoredContentImpressionEvent(getContext(), WidgetType.WIDGET_TYPE_FLEXI_VIEW.getStringValue(), mStyleId, EventParameters.ACTION_IMPRESSION, contentItem.getName(), contentItem.getAdvertiserID(), contentItem.getId(), contentItem.getRealViewIndex(), false);
        }
    }


    // Sequential handler
    private Utils.StaticHandler getMyHandlerInstance() {
        if (mStaticHandler == null) {
            mStaticHandler = new Utils.StaticHandler(Looper.getMainLooper()) {
                @Override
                public void handleMessage(Message message) {
                    // This is where you do your work in the UI thread.
                    // Your worker tells you in the message what to do.
                }
            };
        }
        return mStaticHandler;
    }


    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        // Get flexi view operating area container
        mContainerViewRect = new Rect(0, 0, w, h);

        if (mMovableView != null) {
            getMyHandlerInstance().postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (mMovableView != null) {
                        fixLeftRightEdgeAlignmentIfNeeded();
                        fixTopBottomEdgeAlignmentIfNeeded();
                    }
                }
            }, 300);
        }
    }


    public static final String STYLE_ID = "style_id";
    public static final String ANIMATION_OVERLAY = "flexiAnimOverlay";
    public static final String LOCK_CLOSE_BUTTON = "flexiLockCloseBtn";
    public static final String LOCK_OPEN_BUTTON = "flexiLockOpenBtn";
    public static final String CLOSE_BTN = "flexiCloseBtn";
    public static final String CONTENT_THUMB_RATIO = "flexiThumbRatio";
    public static final String ANIM_OVERLAY_RATIO = "flexiAnimOverlayRatio";
    public static final String BORDER_COLOR = "flexiBorderClr";
    public static final String FILL_COLOR = "flexiFillClr";
    public static final String BORDER_WIDTH = "flexiBorderWidth";
    public static final String CORNER_ROUND_RADIUS = "flexiCornerRadius";
    public static final String IS_CIRCLE = "flexiIsCircle";
    public static final String NOTICE_ME_ANIMATION_TYPE = "flexiNoticeMeAnimType";
    public static final String NOTICE_ME_ANIMATION_RATE_MILLIS = "flexiNoticeMeAnimRateMillis";
    public static final String FLEXI_POP_VIEW_SOUND = "flexiPopSound";
    public static final String CONTENT_SWAP_SOUND = "flexiContentSwapSound";
    public static final String CONTENT_REFRESH_RATE_SECONDS = "flexiContentRefreshRateSec";
    public static final String IS_FLEXI_VIEW_DRUGGABLE = "flexiIsDraggable";
    public static final String IS_FLEXI_VIEW_CLOSABLE = "flexiIsClosable";
    public static final String IS_OVERRIDE_USER_SET = "flexiIsOverrideUserSet";


    /**
     * Compare new JSON to old JSON, save the updated JSON to database, delete old assets if needed and download new assets.
     *
     * @param context
     * @param jsonObject
     */
    public static boolean parseFlexiViewData(Context context, JSONObject jsonObject) {

        boolean result = false;
        if (jsonObject != null) {
            try {
                JSONObject oldData = DatabaseManager.getInstance(context).getConfigTable().loadProperties(FlexiView.TAG);

                result = AssetUtil.loadAsset(context, oldData, jsonObject, ANIMATION_OVERLAY);
                if (result) {
                    result = AssetUtil.loadAsset(context, oldData, jsonObject, LOCK_CLOSE_BUTTON);
                }
                if (result) {
                    result = AssetUtil.loadAsset(context, oldData, jsonObject, LOCK_OPEN_BUTTON);
                }
                if (result) {
                    result = AssetUtil.loadAsset(context, oldData, jsonObject, CLOSE_BTN);
                }

                AssetUtil.loadAsset(context, oldData, jsonObject, FLEXI_POP_VIEW_SOUND);
                AssetUtil.loadAsset(context, oldData, jsonObject, CONTENT_SWAP_SOUND);

                // 4. Update database
                if (result == true) {
                    DatabaseManager.getInstance(context).getConfigTable().insertProperties(FlexiView.TAG, jsonObject);
                }
            } catch (Exception ex) {
                result = false;
                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(FlexiView.TAG, "Error when trying to parse kidoz flexi view properties" + ex.getMessage());
            }
        }

        return result;
    }


    /**
     * Calculate the correct first position of the movable view depending on the chosen FLEXI_POSITION type.
     */
    private void calculateStartPositionCoordinates() {
        if (mMovableView != null && mContainerViewRect != null) {
            switch (mFlexiPosition) {
                case TOP_START: {
                    mStartPointCoord = new Point(0, 0);
                    break;
                }

                case TOP_CENTER: {
                    mStartPointCoord = new Point(mContainerViewRect.width() / 2 - mMovableView.getWidth() / 2, 0);
                    break;
                }

                case TOP_END: {
                    mStartPointCoord = new Point(mContainerViewRect.width() - mMovableView.getWidth(), 0);
                    break;
                }
                case MIDDLE_START: {
                    mStartPointCoord = new Point(0, mContainerViewRect.height() / 2 - mMovableView.getWidth() / 2);
                    break;
                }

                case MIDDLE_CENTER: {
                    mStartPointCoord = new Point(mContainerViewRect.width() / 2 - mMovableView.getWidth() / 2, mContainerViewRect.height() / 2 - mMovableView.getWidth() / 2);
                    break;
                }
                case MIDDLE_END: {
                    mStartPointCoord = new Point(mContainerViewRect.width() - mMovableView.getWidth(), mContainerViewRect.height() / 2 - mMovableView.getWidth() / 2);
                    break;
                }
                case BOTTOM_START: {
                    mStartPointCoord = new Point(0, mContainerViewRect.height() - mMovableView.getWidth());
                    break;
                }
                case BOTTOM_CENTER: {
                    mStartPointCoord = new Point(mContainerViewRect.width() / 2 - mMovableView.getWidth() / 2, mContainerViewRect.height() - mMovableView.getWidth());
                    break;
                }

                case BOTTOM_END: {
                    mStartPointCoord = new Point(mContainerViewRect.width() - mMovableView.getWidth(), mContainerViewRect.height() - mMovableView.getWidth());
                    break;
                }

            }
        }
    }


    /**
     * Set flexi view event listener
     *
     * @param eventListener
     */
    public void setOnFlexiViewEventListener(FlexiViewListener eventListener) {
        mFlexiViewListener = eventListener;
    }


}
