package com.kidoz.sdk.api;

import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.os.Bundle;
import android.view.KeyEvent;
import android.view.View;

import com.kidoz.events.Event;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.BaseDialog;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.ContentType;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.AssetUtil;
import com.kidoz.sdk.api.interfaces.IOnFeedViewEventListener;
import com.kidoz.sdk.api.structure.ContentData;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.ContentPanelView;
import com.kidoz.sdk.api.ui_views.FeedFullView;

import org.json.JSONObject;

/**
 * Created by Eugine on 10/1/2015.
 */
public class FeedViewDialog extends BaseDialog
{

    public static final String TAG = FeedViewDialog.class.getSimpleName();

    private static final String SAVE_DATA_INSTANCE_KEY = "SAVE_DATA_INSTANCE_KEY_FV";
    private static final String SAVE_ORIGINAL_ORIENTATION_KEY = "SAVE_ORIGINAL_ORIENTATION_KEY_FV";

    private FeedFullView mFeedFullVeiw;
    private ContentData mContentData = null;
    // Close button click listener
    private View.OnClickListener mCloseBtnClickListener = null;

    private IOnFeedViewEventListener mOnInterstitialViewEventListener;
    private boolean mAllowClickHandling = true;
    private boolean mIsAutomaticInvocation = false;

    private JSONObject mPropertiesObject;

    public FeedViewDialog(Context context, IOnFeedViewEventListener listener, boolean automatic, JSONObject propertiesObject)
    {
        super(context, android.R.style.Theme_Black_NoTitleBar_Fullscreen);
        mPropertiesObject = propertiesObject;
        mIsAutomaticInvocation = automatic;
        mOnInterstitialViewEventListener = listener;
        setCancelable(false);
    }

    @Override
    protected void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        // Init back button listener
        setOnKeyListener(new DialogInterface.OnKeyListener()
        {
            @Override
            public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event)
            {
                boolean result = true;
                if (event.getAction() == KeyEvent.ACTION_DOWN)
                {
                    if (keyCode == KeyEvent.KEYCODE_BACK)
                    {
                        if (mFeedFullVeiw != null && mFeedFullVeiw.getViewBottomBar().isAboutViewVisible())
                        {
                            mFeedFullVeiw.getViewBottomBar().hideAboutView();
                        }
                        else
                        {
                            if (mOnInterstitialViewEventListener != null)
                            {
                                mOnInterstitialViewEventListener.onDismissView();
                            }
                            closeDialog();
                        }
                    }
                }
                return result;
            }
        });

        if (savedInstanceState != null && savedInstanceState.containsKey(SAVE_ORIGINAL_ORIENTATION_KEY))
        {
            mCurrentOrientationState = savedInstanceState.getInt(SAVE_ORIGINAL_ORIENTATION_KEY);
        }
        else
        {
            mCurrentOrientationState = mCallingActivity.getResources().getConfiguration().orientation;
            mCallingActivity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR);
        }

        // Restore current view state data in case the fragment was destroyed due to going offscreen
        if (mContentData == null && savedInstanceState != null && savedInstanceState.containsKey(SAVE_DATA_INSTANCE_KEY))
        {
            mContentData = (ContentData) savedInstanceState.get(SAVE_DATA_INSTANCE_KEY);
        }

        mFeedFullVeiw = new FeedFullView(mCallingActivity, mPropertiesObject);
        mFeedFullVeiw.setOnContentItemClickListener(new ContentPanelView.IOnContentItemClickListener()
        {
            @Override
            public void onItemClick(ContentItem contentItem, int position)
            {
                if (contentItem != null && contentItem.getContentType() != null && contentItem.getContentType() != ContentType.NONE)
                {
                    // Handle content item click
                    JSONObject jsonObject = DatabaseManager.getInstance(getContext()).getConfigTable().loadProperties(FeedView.TAG);
                    if (jsonObject != null)
                    {
                        if (mAllowClickHandling)
                        {
                            mAllowClickHandling = false;
                            ContentExecutionHandler.handleContentItemClick(mCallingActivity, contentItem, WidgetType.WIDGET_TYPE_FEED.getStringValue(), jsonObject.optString(FeedView.STYLE_ID), position, false, false, new ContentExecutionHandler.IOnHandleClickListener()
                            {
                                @Override
                                public void onRestoreClick()
                                {
                                    mAllowClickHandling = true;
                                }
                            });
                        }
                    }
                }
            }
        });

        mFeedFullVeiw.setOnCloseButtonClickListenr(mCloseBtnClickListener);
        mFeedFullVeiw.setContentDataAndRefresh(mContentData);

        setContentView(mFeedFullVeiw);
        mFeedFullVeiw.showLoadingProgressView(true);
    }

    protected void setContentData(ContentData contentData)
    {
        mContentData = contentData;
        if (mFeedFullVeiw != null)
        {
            mFeedFullVeiw.showLoadingProgressView(false);
            mFeedFullVeiw.setContentDataAndRefresh(mContentData);
        }
    }

    /**
     * Set on close button click listener
     *
     * @param clickListenr click listener
     */
    protected void setOnCloseButtonClickListener(View.OnClickListener clickListenr)
    {
        mCloseBtnClickListener = clickListenr;
        if (mFeedFullVeiw != null)
        {
            mFeedFullVeiw.setOnCloseButtonClickListenr(mCloseBtnClickListener);
        }
    }


    @Override
    public Bundle onSaveInstanceState()
    {
        Bundle bundle = super.onSaveInstanceState();
        if (mContentData != null && bundle != null)
        {
            bundle.putSerializable(SAVE_DATA_INSTANCE_KEY, mContentData);
            bundle.putInt(SAVE_ORIGINAL_ORIENTATION_KEY, mCurrentOrientationState);
        }
        return bundle;
    }


    @Override
    public void closeDialog()
    {

        mContentData = null;
        super.closeDialog();
    }

    @Override
    public void openDialog()
    {
        if (KidozSDK.isInitialised())
        {
            super.openDialog();

            Event event = new Event();
            if (mIsAutomaticInvocation)
            {
                event.addParameterToJsonObject(EventParameters.LABEL_VIEW_OPEN_TYPE, EventParameters.AUTOMATIC_OPEN);
            }
            else
            {
                event.addParameterToJsonObject(EventParameters.LABEL_VIEW_OPEN_TYPE, EventParameters.MANUAL_OPEN);
            }
            JSONObject jsonObject = DatabaseManager.getInstance(getContext()).getConfigTable().loadProperties(FeedView.TAG);
            if (jsonObject != null)
            {
                EventManager.getInstance(mCallingActivity).logEvent(mCallingActivity, WidgetType.WIDGET_TYPE_FEED.getStringValue(), jsonObject.optString(FeedView.STYLE_ID), EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_WIDGET_VIEW, EventParameters.FEED_VIEW, false);
            }
        }
    }

    public void setOnInterstitialViewEventListener(IOnFeedViewEventListener onInterstitialViewEventListener)
    {
        if (onInterstitialViewEventListener != null)
        {
            mOnInterstitialViewEventListener = onInterstitialViewEventListener;
        }
    }

    @Override
    public void onDetachedFromWindow()
    {
        super.onDetachedFromWindow();

        // Restore previous orientation
        restoreUserOriginalOrientation();
    }

    @Override
    protected void onDialogWindowVisibilityChange(boolean isVisible)
    {
        super.onDialogWindowVisibilityChange(isVisible);
        if (isVisible == false)
        {
            EventManager.getInstance(getContext()).startEventsSync(getContext());
        }
        mAllowClickHandling = true;
    }

    @Override
    protected void onDialogFocusChange(boolean isVisible)
    {
        super.onDialogFocusChange(isVisible);

        if (isVisible)
        {
            mAllowClickHandling = true;
        }
        else
        {
            mAllowClickHandling = false;
        }
    }

    public static final String EXIT_BUTTON = "exitButton";
    public static final String LOCK_ACTIVE_BUTTON = "lockActiveButton";
    public static final String LOCK_NOT_ACTIVE_BUTTON = "lockNotActiveButton";
    public static final String BOTTOM_BAR_BACKGROUND = "bottomBarBackground";
    public static final String BOTTOM_BAR_TITLE = "bottomBarTitle";
    public static final String TOP_BAR_BACKGROUND = "topBarBackground";
    public static final String TOP_BAR_TITLE = "topBarTitle";
    public static final String TOP_BAR_TITLE_COLOR = "topBarTitleColor";
    public static final String TOP_BAR_BACKGROUND_COLOR = "topBarBgColor";
    public static final String CONTENT_MAIN_BACKGROUND_COLOR_START = "mainBgColorStart";
    public static final String CONTENT_MAIN_BACKGROUND_COLOR_CENTER = "mainBgColorCenter";
    public static final String CONTENT_MAIN_BACKGROUND_COLOR_END = "mainBgColorEnd";


    /**
     * Compare new JSON to old JSON, save the updated JSON to database, delete old assets if needed and download new assets.
     *
     * @param context
     * @param jsonObject
     */
    public static boolean parseFeedData(Context context, JSONObject jsonObject)
    {

        boolean result = false;
        if (jsonObject != null)
        {
            try
            {

                jsonObject.put(FeedView.TAG, FeedViewDialog.TAG);

                JSONObject oldData = DatabaseManager.getInstance(context).getConfigTable().loadProperties(FeedView.TAG);

                result = AssetUtil.loadAsset(context, oldData, jsonObject, FeedView.KIDOZ_FEED_BUTTON);
                if (result)
                {
                    result = AssetUtil.loadAsset(context, oldData, jsonObject, EXIT_BUTTON);
                }
                if (result)
                {
                    result = AssetUtil.loadAsset(context, oldData, jsonObject, LOCK_ACTIVE_BUTTON);
                }
                if (result)
                {
                    result = AssetUtil.loadAsset(context, oldData, jsonObject, LOCK_NOT_ACTIVE_BUTTON);
                }
                if (result)
                {
                    result = AssetUtil.loadAsset(context, oldData, jsonObject, BOTTOM_BAR_BACKGROUND);
                }
                if (result)
                {
                    result = AssetUtil.loadAsset(context, oldData, jsonObject, TOP_BAR_BACKGROUND);
                }

                if (result == true)
                {
                    DatabaseManager.getInstance(context).getConfigTable().insertProperties(FeedView.TAG, jsonObject);
                }
            }
            catch (Exception ex)
            {
                result = false;
                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(FeedView.TAG, "Error when trying to parse kidoz PanelView properties" + ex.getMessage());
            }
        }
        return result;
    }
}
