package com.kidoz.sdk.api.general.utils;

import android.annotation.SuppressLint;
import android.content.Context;
import android.media.AudioAttributes;
import android.media.AudioManager;
import android.media.SoundPool;
import android.os.Build;

import java.util.HashMap;

@SuppressLint("NewApi")
public class SdkSoundManager
{
    private static String TAG = SdkSoundManager.class.getSimpleName();
    private static SoundPool sSoundPool;
    private static HashMap<String, Integer> sLoadedSounds = new HashMap<String, Integer>();

    // Private methods
    @SuppressLint("NewApi")
    private static void initSoundPool()
    {
        if (sSoundPool == null)
        {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP)
            {
                SoundPool.Builder builder = new SoundPool.Builder();
                builder.setMaxStreams(3);
                AudioAttributes.Builder atrBuilder = new AudioAttributes.Builder();
                atrBuilder.setContentType(AudioAttributes.CONTENT_TYPE_SONIFICATION);
                atrBuilder.setLegacyStreamType(AudioAttributes.CONTENT_TYPE_SONIFICATION);
                atrBuilder.setUsage(AudioAttributes.USAGE_MEDIA);
                builder.setAudioAttributes(atrBuilder.build());
                sSoundPool = builder.build();
            }
            else
            {
                sSoundPool = new SoundPool(3, AudioManager.STREAM_MUSIC, 0);
            }
        }
    }

    private static void playLoadedSound(Context context, int sound)
    {
        initSoundPool();
        if (context != null)
        {
            AudioManager audioManager = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
            float actualVolume = (float) audioManager.getStreamVolume(AudioManager.STREAM_MUSIC);
            float maxVolume = (float) audioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC);
            float volume = actualVolume / maxVolume;
            sSoundPool.play(sound, volume, volume, 1, 0, 1f);
        }
    }


    /**
     * Preload sound to memory
     *
     * @param soundResource resource of sound
     */
    public static void preloadSound(Context context, int soundResource)
    {
        initSoundPool();
        if (context != null && sLoadedSounds != null && sLoadedSounds.get(soundResource + "") == null)
        {
            final int soundID = sSoundPool.load(context, soundResource, 1);
            sLoadedSounds.put(soundResource + "", soundID);
        }
    }

    /**
     * Preload sound to memory
     *
     * @param path path sound file
     */
    public static void preloadSound(String path)
    {
        SDKLogger.printDebbugLog(TAG, "*** preloadSound");
        initSoundPool();
        if (path != null && sLoadedSounds != null && sLoadedSounds.get(path) == null)
        {
            int soundID = sSoundPool.load(path, 1);
            sLoadedSounds.put(path, soundID);
            SDKLogger.printDebbugLog(TAG, "preloadSound: path = " + path + ", soundID = " + String.valueOf(soundID));
        }
    }

    /**
     * Play sound from resource
     *
     * @param soundResource sound resource id
     */
    public static void playSound(Context context, int soundResource)
    {
        initSoundPool();
        if (context != null && sLoadedSounds != null && sLoadedSounds.get(soundResource + "") == null)
        {
            int soundID = sSoundPool.load(context, soundResource, 1);
            sLoadedSounds.put(soundResource + "", soundID);
            playLoadedSound(context, soundID);
        }
        else if (sLoadedSounds != null)
        {
            playLoadedSound(context, sLoadedSounds.get("" + soundResource));
        }
    }

    /**
     * Play sound from path
     *
     * @param path sound path
     */
    public static void playSound(Context context, String path)
    {
        SDKLogger.printDebbugLog(TAG, "*** playSound");
        initSoundPool();
        if (context != null && path != null && sSoundPool != null)
        {
            if (sLoadedSounds != null && sLoadedSounds.get(path) == null)
            {
                int soundID = sSoundPool.load(path, 1);
                sLoadedSounds.put(path, soundID);
                playLoadedSound(context, soundID);
                SDKLogger.printDebbugLog(TAG, "playSound: path = " + path + ", soundID = " + String.valueOf(soundID));
            }
            else if (sLoadedSounds != null)
            {
                playLoadedSound(context, sLoadedSounds.get(path));
                SDKLogger.printDebbugLog(TAG, "playSound: path = " + path + ", soundID = " + String.valueOf(sLoadedSounds.get(path)));
            }
        }
    }

    /**
     * Release sound manager
     */
    public static void releaseManager()
    {
        if (sSoundPool != null)
        {
            sSoundPool.release();
        }
        sSoundPool = null;
        if (sLoadedSounds != null)
        {
            sLoadedSounds.clear();
        }
    }
    // End of public methods
}
