package com.kidoz.sdk.api;

import android.content.Context;
import android.graphics.Color;
import android.util.AttributeSet;
import android.widget.RelativeLayout;

import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.interfaces.IOnPanelViewEventListener;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.kidoz.sdk.api.ui_views.panel_view.AbstractPanelView;
import com.kidoz.sdk.api.ui_views.panel_view.FamilyPanelView;
import com.kidoz.sdk.api.ui_views.panel_view.HANDLE_POSITION;
import com.kidoz.sdk.api.ui_views.panel_view.PANEL_TYPE;
import com.kidoz.sdk.api.ui_views.panel_view.StandardPanelView;

import org.json.JSONObject;

import de.greenrobot.event.EventBus;

/**
 * Created by Eugine on 12/8/2015.
 */
public class PanelView extends RelativeLayout {

    public static final String TAG = PanelView.class.getSimpleName();

    private AbstractPanelView mAbstractPanelView;
    private PANEL_TYPE mPanelType = null;
    private HANDLE_POSITION mHandlePosition = null;
    private IOnPanelViewEventListener mIOnPanelViewEventListener = null;
    private int mColor = Color.TRANSPARENT;

    public PanelView(Context context) {
        super(context);
        initView();
    }

    public PanelView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initView();
    }

    public PanelView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initView();
    }

    private void initView() {

        EventBus.getDefault().register(this);

        setBackgroundColor(Color.TRANSPARENT);
        setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));

        // We check if SDK initiation has successfully finished( Sdk Config data accepted)
        if (KidozSDK.isInitialised() == true) {
            EventBus.getDefault().unregister(this);
            validateLayoutAndAssets();
        }
    }

    private void validateLayoutAndAssets() {
        SdkAPIManager.getSdkApiInstance(getContext()).getStyle(getContext(), WidgetType.WIDGET_TYPE_PANEL.getValue(), new ApiResultCallback<Boolean>() {
            @Override
            public void onServerResult(ResultData<?> result) {
                // If result success notify wrapper to build the banner.
                if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true) {
                    if (result.getData() != null) {
                        boolean isAssetsReady = (Boolean) result.getData();
                        if (isAssetsReady == true) {
                            buildPanelView();
                        } else {
                            onFailed();
                        }
                    }
                } else {
                    onFailed();
                }
            }

            @Override
            public void onFailed() {
            }
        });
    }

    /**
     * Build panel view according to accepted parameters from server
     */
    private void buildPanelView() {
        JSONObject jsonObject = DatabaseManager.getInstance(getContext()).getConfigTable().loadProperties(PanelView.TAG);
        if (jsonObject != null) {
            String panelType = jsonObject.optString(PanelView.TAG);
            if (panelType.equals(StandardPanelView.TAG)) {
                mAbstractPanelView = new StandardPanelView(getContext(), jsonObject);

            } else if (panelType.equals(FamilyPanelView.TAG)) {
                mAbstractPanelView = new FamilyPanelView(getContext(), jsonObject);
            }

            if (mPanelType != null && mHandlePosition != null) {
                mAbstractPanelView.setPanelConfiguration(mPanelType, mHandlePosition);
            }

            if (mIOnPanelViewEventListener != null) {
                mAbstractPanelView.setOnPanelViewEventListener(mIOnPanelViewEventListener);
            }

            int color = Color.WHITE;
            try {
                color = Color.parseColor(jsonObject.optString(AbstractPanelView.PANEL_BG_COLOR, "#ffffff"));
            } catch (Exception e) {
            }

            if (jsonObject.optInt(AbstractPanelView.OVERRIDE_USER_PROPERTIES, 0) == 0 && mColor != Color.TRANSPARENT) {
                color = mColor;
            }
            mAbstractPanelView.setPanelColor(color);
        }
        addView(mAbstractPanelView);
    }

    // Event bus callback
    public void onEvent(EventMessage event) {
        if(event.getMessageType() == EventMessage.MessageType.INIT_SDK) {
            EventBus.getDefault().unregister(this);
            validateLayoutAndAssets();
        }
    }

    /**
     * Set panel color
     *
     * @param color
     */
    public void setPanelColor(int color) {
        mColor = color;

        if(mAbstractPanelView != null) {
            JSONObject jsonObject = DatabaseManager.getInstance(getContext()).getConfigTable().loadProperties(PanelView.TAG);
            if(jsonObject != null) {
                if (jsonObject.optInt(AbstractPanelView.OVERRIDE_USER_PROPERTIES, 0) == 0) {
                    mAbstractPanelView.setPanelColor(color);
                }
            }
        }
    }


    /**
     * Collapse panel view
     */
    public void collapsePanelView() {
        if (mAbstractPanelView != null) {
            mAbstractPanelView.collapsePanelView();
        }
    }

    /**
     * Expand (Open) Panel view
     */
    public void expandPanelView() {
        if (mAbstractPanelView != null) {
            mAbstractPanelView.expandPanelView();
        }
    }

    /**
     * Set panel view configuration
     *
     * @param panelType      panel view type (Possible values <code> PANEL_TYPE.TOP, PANEL_TYPE.BOTTOM, PANEL_TYPE.LEFT ,PANEL_TYPE.RIGHT </code>
     * @param handlePosition handle position  (Possible values <code> HANDLE_POSITION.START, HANDLE_POSITION.CENTER, HANDLE_POSITION.END </code>
     */
    public void setPanelConfiguration(PANEL_TYPE panelType, HANDLE_POSITION handlePosition) {
        mPanelType = panelType;
        mHandlePosition = handlePosition;

        if (mAbstractPanelView != null) {
            mAbstractPanelView.setPanelConfiguration(panelType, handlePosition);
        }
    }


    /**
     * Set panel events listener
     *
     * @param iOnPanelViewEventListener listener
     */
    public void setOnPanelViewEventListener(IOnPanelViewEventListener iOnPanelViewEventListener) {
        mIOnPanelViewEventListener = iOnPanelViewEventListener;
        if (mAbstractPanelView != null) {
            mAbstractPanelView.setOnPanelViewEventListener(iOnPanelViewEventListener);
        }
    }

    /**
     * Get panel view state
     *
     * @return is panel expanded pr closed
     */
    public boolean getIsPanelViewExpanded() {
        if (mAbstractPanelView != null) {
            return mAbstractPanelView.getIsPanelViewExpanded();
        } else {
            return false;
        }
    }

    @Override
    protected void onDetachedFromWindow() {
        try {
            EventBus.getDefault().unregister(this);
        } catch (Exception e) {

        }
        super.onDetachedFromWindow();

    }
}
