package com.kidoz.sdk.api.ui_views.parental_lock;

import android.content.Context;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;
import android.widget.ImageView;

import com.kidoz.sdk.api.picasso_related.PicassoOk;
import com.squareup.imagelib.Callback;

import java.io.File;
import java.io.IOException;

import pl.droidsonroids.gif.GifDrawable;


public class AssetView extends ImageView
{
    private boolean mIsAssetLoaded;

    private GifDrawable mAnimatedGifDrawable = null;

    public AssetView(Context context)
    {
        super(context);

        initView();
    }

    public AssetView(Context context, AttributeSet attrs)
    {
        super(context, attrs);

        initView();
    }

    public AssetView(Context context, AttributeSet attrs, int defStyleAttr)
    {
        super(context, attrs, defStyleAttr);

        initView();
    }

    private void initView()
    {
        setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        setScaleType(ImageView.ScaleType.FIT_XY);
    }

    public void loadAsset(File file, IOnAssetLoadedListener loadedListener)
    {
        loadAsset(file, null, loadedListener);
    }

    public void loadAsset(String path, IOnAssetLoadedListener loadedListener)
    {
        loadAsset(new File(path), null, loadedListener);
    }

    public void loadAsset(String path, Drawable defaultImage, IOnAssetLoadedListener loadedListener)
    {
        loadAsset(new File(path), defaultImage, loadedListener);
    }

    public void loadAsset(File file, Drawable defaultImage, final IOnAssetLoadedListener loadedListener)
    {
        mIsAssetLoaded = false;
        if (defaultImage != null)
        {
            setAssetBackgroundDrawable(defaultImage);
        }

        if (file != null)
        {
            if (file.exists())
            {
                String path = file.getPath();
                if (path.contains(".png") || path.contains(".PNG") || path.contains(".jpg") || path.contains(".JPG"))
                {

                    PicassoOk.getPicasso(getContext()).load(file).into(this, new Callback()
                    {
                        @Override
                        public void onSuccess()
                        {
                            setAssetBackgroundDrawable(null);
                            mIsAssetLoaded = true;
                            if (loadedListener != null)
                            {
                                loadedListener.onAssetLoaded();
                            }
                        }

                        @Override
                        public void onError()
                        {
                            if (loadedListener != null)
                            {
                                loadedListener.onAssetLoaded();
                            }
                        }
                    });

                }
                else if (path.contains(".gif") || path.contains(".GIF"))
                {
                    try
                    {
                        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
                        {
                            mAnimatedGifDrawable.stop();
                            mAnimatedGifDrawable.recycle();
                            mAnimatedGifDrawable = null;
                        }

                        mAnimatedGifDrawable = new GifDrawable(file.getAbsolutePath());
                        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
                        {
                            setImageDrawable(mAnimatedGifDrawable);
                            setAssetBackgroundDrawable(null);
                            mAnimatedGifDrawable.start();
                        }
                        mIsAssetLoaded = true;
                        if (loadedListener != null)
                        {
                            loadedListener.onAssetLoaded();
                        }

                    }
                    catch (IOException e)
                    {
                        // If something went wrong delete the file
                        if (defaultImage != null)
                        {
                            setAssetBackgroundDrawable(defaultImage);

                            if (loadedListener != null)
                            {
                                loadedListener.onAssetLoaded();
                            }
                        }
                    }

                }
                else
                {
                    setAssetBackgroundDrawable(defaultImage);

                    if (loadedListener != null)
                    {
                        loadedListener.onAssetLoaded();
                    }
                }
            }
            else
            {
                setAssetBackgroundDrawable(defaultImage);

                if (loadedListener != null)
                {
                    loadedListener.onAssetLoaded();
                }
            }
        }
    }

    public void setAssetBackgroundDrawable(Drawable defaultImage)
    {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
        {
            setBackground(defaultImage);
        }
        else
        {
            setBackgroundDrawable(defaultImage);
        }
        postInvalidate();
    }

    public boolean getIsAssetLoaded()
    {
        return mIsAssetLoaded;
    }

    @Override
    protected void onFocusChanged(boolean gainFocus, int direction, Rect previouslyFocusedRect)
    {
        super.onFocusChanged(gainFocus, direction, previouslyFocusedRect);

        if (gainFocus)
        {
            if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled() && mAnimatedGifDrawable.isVisible())
            {
                if (!mAnimatedGifDrawable.isPlaying())
                {
                    mAnimatedGifDrawable.start();
                }
            }
        }
    }

    @Override
    protected void onDetachedFromWindow()
    {
        super.onDetachedFromWindow();
        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
        {
            mAnimatedGifDrawable.stop();
            mAnimatedGifDrawable.recycle();
            mAnimatedGifDrawable = null;
        }
    }

    @Override
    protected void onVisibilityChanged(View changedView, int visibility)
    {
        super.onVisibilityChanged(changedView, visibility);

        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled())
        {
            if (visibility == View.VISIBLE)
            {
                if (!mAnimatedGifDrawable.isPlaying())
                {
                    mAnimatedGifDrawable.start();
                }
            }
            else
            {
                mAnimatedGifDrawable.stop();
            }
        }
    }


    public interface IOnAssetLoadedListener
    {
        void onAssetLoaded();
    }
}
