package com.kidoz.sdk.api.ui_views.panel_view;

import android.animation.Animator;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Color;
import android.graphics.Point;
import android.os.Looper;
import android.os.Message;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.kidoz.events.Event;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.ContentLogicLoader;
import com.kidoz.sdk.api.KidozSDK;
import com.kidoz.sdk.api.dialogs.AboutKidozDialog;
import com.kidoz.sdk.api.dialogs.ParentalLockDialog;
import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.AssetUtil;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.interfaces.IOnPanelViewEventListener;
import com.kidoz.sdk.api.ui_views.custom_drawables.ShadowView;
import com.kidoz.sdk.api.ui_views.one_item_view.OneItemView;

import org.json.JSONObject;


public abstract class AbstractPanelView extends RelativeLayout {

    protected static final int DEFAULT_SHADOW_LINE_SIZE = 3;

    protected PANEL_TYPE mPanelType = PANEL_TYPE.BOTTOM;
    protected HANDLE_POSITION mHandlePosition = HANDLE_POSITION.START;

    protected RelativeLayout mContainer;
    protected RelativeLayout mInnerContainer;
    protected ContentLogicLoader mContentLogicLoader;

    protected PanelButton mParentalLockButton;
    protected PanelButton mAboutKidozButton;

    protected PanelHandle mPanelHandle;
    protected TextView mPoweredByTextView;
    protected RelativeLayout mBottomBarContainer;
    protected OneItemView mOneItemView;
    protected IOnPanelViewEventListener mIOnPanelViewEventListener;

    protected boolean mIsOpen;
    protected boolean mIsAnimationRunning;
    protected boolean mIsClicksBlocked;
    protected boolean mAllowClickHandle = true;

    protected Utils.StaticHandler mStaticHandler;
    protected Utils.StaticHandler mUiLoaderStaticHandler;

    protected ShadowView mShadowView;
    protected LinearLayout mShadowLine;

    protected JSONObject mWidgetProperties;

    protected int mAnimType = 1;
    protected String mBtnClosedUrl = null;
    protected String mBtnOpenUrl = null;
    protected float mBtnScaleSize = 0.15f;
    protected int mPanelBgColor = Color.WHITE;
    protected String mLabelText = "";
    protected int mLabelColor = Color.BLACK;

    // The style id of the widget
    protected String mStyleId = null;

    public AbstractPanelView(Context context, JSONObject jsonObject) {
        super(context);
        mWidgetProperties = jsonObject;
        prepareProperties();

        initView();
    }

    protected void prepareProperties() {
        mStyleId = mWidgetProperties.optString(STYLE_ID, null);

        mAnimType = mWidgetProperties.optInt(BTN_ANIMATION_TYPE, 1);
        mBtnClosedUrl = mWidgetProperties.optString(BTN_CLOSED_URL, null);
        mBtnOpenUrl = mWidgetProperties.optString(BTN_OPENED_URL, null);
        mBtnScaleSize = (float) mWidgetProperties.optDouble(BTN_SCALE_SIZE, 0.15f);
        mLabelText = mWidgetProperties.optString(LABEL_TEXT, "");

        try {
            mPanelBgColor = Color.parseColor(mWidgetProperties.optString(PANEL_BG_COLOR, "#FFFFFF"));
        } catch (Exception e) {
            mPanelBgColor = Color.WHITE;
        }

        try {
            mLabelColor = Color.parseColor(mWidgetProperties.optString(LABEL_TEXT_COLOR, "#000000"));
        } catch (Exception e) {
            mLabelColor = Color.BLACK;
        }
    }

    protected void initView() {
        initPanelTypeAndHandlePosition();
        initPanelContainer();
        initHandleButton();
        initInnerPanelContentView();
        initParentalLockButton();
        initAboutKidozButton();
        initBottomTitleBar();
        initPanelViewsRules();
        initContentLogicLoader();
        preparePanel();
    }

    protected abstract void initInnerPanelContentView();

    protected abstract void initPanelViewsRules();

    protected abstract void initContentLogicLoader();

    protected abstract void initShadowView();

    protected void initPanelTypeAndHandlePosition() {
        mPanelType = com.kidoz.sdk.api.ui_views.panel_view.PANEL_TYPE.BOTTOM;
    }

    private void initPanelContainer() {
        mContainer = new RelativeLayout(getContext());
        mContainer.setBackgroundColor(Color.TRANSPARENT);
        mContainer.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        setLayoutParams(new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT));

        initShadowView();

        LayoutParams params = new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        addView(mContainer, params);

        mInnerContainer = new RelativeLayout(getContext());
        mInnerContainer.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        mInnerContainer.setBackgroundColor(Color.WHITE);

        params = new RelativeLayout.LayoutParams(LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        mContainer.addView(mInnerContainer, params);
    }


    private void initHandleButton() {

        Point point = Utils.getScreenSize(getContext());
        int size = (int) (Math.min(point.x, point.y) * mBtnScaleSize);

        mPanelHandle = new PanelHandle(getContext(), mPanelType, mAnimType, size, new PanelHandle.IOnPanelHandleClickListener() {
            @Override
            public void onHandleClick() {
                toggleAnimation();
            }
        });

        // Default handle button
        mPanelHandle.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        mContainer.addView(mPanelHandle, params);

    }


    protected void initBottomTitleBar() {

        mBottomBarContainer = new RelativeLayout(getContext());
        mBottomBarContainer.setBackgroundColor(mPanelBgColor);
        mBottomBarContainer.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT);
        mInnerContainer.addView(mBottomBarContainer, params);

        mPoweredByTextView = new TextView(getContext());
        mPoweredByTextView.setText(mLabelText);
        mPoweredByTextView.setTextColor(mLabelColor);
        mPoweredByTextView.setTextSize(TypedValue.COMPLEX_UNIT_SP, 12);
        mPoweredByTextView.setLines(1);
        mPoweredByTextView.setEllipsize(TextUtils.TruncateAt.END);

        params = new RelativeLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT);
        params.addRule(RelativeLayout.CENTER_VERTICAL);
        params.addRule(RelativeLayout.LEFT_OF, mParentalLockButton.getId());
        params.addRule(RelativeLayout.RIGHT_OF, mAboutKidozButton.getId());
        params.setMargins(Utils.dpTOpx(getContext(), 4), Utils.dpTOpx(getContext(), 5), Utils.dpTOpx(getContext(), 4), Utils.dpTOpx(getContext(), 5));
        mBottomBarContainer.addView(mPoweredByTextView, params);
    }


    private void initParentalLockButton() {
        mParentalLockButton = new PanelButton(getContext(), PANEL_TYPE.BOTTOM, com.kidoz.sdk.api.ui_views.panel_view.PANEL_BUTTON_TYPE.PARENTAL_LOCK, new PanelButton.PanelButtonListener() {
            @Override
            public void onClickStarted() {

            }

            @Override
            public void onClickEnd() {
                if (mIsClicksBlocked == false) {
                    mIsClicksBlocked = true;
                    if (ParentalLockDialog.getIsCanShowDialog(getContext()) == true) {
                        int[] popLocation = new int[2];
                        popLocation[0] = (int) (ScreenUtils.getScreenSize(getContext(), true) * 0.5f);
                        popLocation[1] = (int) (ScreenUtils.getScreenSize(getContext(), false) * 0.5f);
                        ParentalLockDialog parentalLockDialog = new ParentalLockDialog(getContext(), false, popLocation);
                        parentalLockDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
                            @Override
                            public void onDismiss(DialogInterface dialog) {
                                //Refresh the parental lock icon
                                boolean isLockActive = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), ParentalLockDialog.PARENTAL_LOCK_ACTIVE_KEY) == null ? false : true;
                                mParentalLockButton.setIsLockActive(isLockActive);
                                mIsClicksBlocked = false;
                            }
                        });
                        parentalLockDialog.setParentLockDialogListener(new ParentalLockDialog.ParentLockDialogListener() {
                            @Override
                            public void onPasswordEntered(boolean isPasswordCorrect) {
                                boolean isLockActive = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), ParentalLockDialog.PARENTAL_LOCK_ACTIVE_KEY) == null ? false : true;
                                mParentalLockButton.setIsLockActive(isLockActive);
                            }
                        });
                        parentalLockDialog.show();
                    } else {
                        mIsClicksBlocked = false;
                    }
                }
            }
        });
        mParentalLockButton.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        mParentalLockButton.setRoundedCornerDirection(PanelButton.RoundedCornerDirection.LEFT);
        mInnerContainer.addView(mParentalLockButton, params);
        boolean isLockActive = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), ParentalLockDialog.PARENTAL_LOCK_ACTIVE_KEY) == null ? false : true;
        mParentalLockButton.setIsLockActive(isLockActive);
    }


    private void initAboutKidozButton() {
        mAboutKidozButton = new PanelButton(getContext(), PANEL_TYPE.BOTTOM, com.kidoz.sdk.api.ui_views.panel_view.PANEL_BUTTON_TYPE.ABOUT_KIDOZ, new PanelButton.PanelButtonListener() {
            @Override
            public void onClickStarted() {

            }

            @Override
            public void onClickEnd() {
                int[] popLocation = new int[2];
                popLocation[0] = (int) (ScreenUtils.getScreenSize(getContext(), true) * 0.5f);
                popLocation[1] = (int) (ScreenUtils.getScreenSize(getContext(), false) * 0.5f);
                AboutKidozDialog aboutKidozDialog = new AboutKidozDialog(getContext(), popLocation);
                aboutKidozDialog.openDialog();
            }
        });
        mAboutKidozButton.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        mAboutKidozButton.setRoundedCornerDirection(PanelButton.RoundedCornerDirection.RIGHT);
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        mInnerContainer.addView(mAboutKidozButton, params);

    }

    /**
     * Set panel color
     *
     * @param color
     */
    public void setPanelColor(int color) {
        mPanelBgColor = color;

        if (mParentalLockButton != null) {
            mParentalLockButton.setBaseColor(mPanelBgColor);
        }

        if (mAboutKidozButton != null) {
            mAboutKidozButton.setBaseColor(mPanelBgColor);
        }

        if (mBottomBarContainer != null) {
            mBottomBarContainer.setBackgroundColor(mPanelBgColor);
        }

        if (mInnerContainer != null) {
            mInnerContainer.setBackgroundColor(mPanelBgColor);
        }

        if (mPanelHandle != null) {
            mPanelHandle.setBaseColor(mPanelBgColor);
        }
    }

    private void toggleAnimation() {
        if (mIsOpen == true) {
            collapsePanelView();
        } else {
            expandPanelViewInternal(false);
        }
    }

    /**
     * Collapse panel view
     */
    public void collapsePanelView() {
        if (KidozSDK.isInitialised()) {
            if (mIsAnimationRunning == false && mIsOpen == true) {
                GenAnimator.closePanelView(mContainer, 200, mInnerContainer, mPanelType, new Animator.AnimatorListener() {
                    @Override
                    public void onAnimationStart(Animator animation) {
                        GenAnimator.animateHandleClose(mPanelHandle, null);
                    }

                    @Override
                    public void onAnimationEnd(Animator animation) {
                        onCollapsePanelAnimationEnded();
                        EventManager.getInstance(getContext()).startEventsSync(getContext());
                    }

                    @Override
                    public void onAnimationCancel(Animator animation) {

                    }

                    @Override
                    public void onAnimationRepeat(Animator animation) {

                    }
                });
            }
        }
    }

    protected void onCollapsePanelAnimationEnded() {

        mPanelHandle.setPanelButtonState(PanelHandle.PANEL_HANDLE_VIEW_STATE.CLOSED);

        mIsOpen = false;
        mIsAnimationRunning = false;
        if (mIOnPanelViewEventListener != null) {
            mIOnPanelViewEventListener.onPanelViewCollapsed();
        }
    }


    /**
     * Expand (Open) Panel view
     */
    public void expandPanelView() {
        if (KidozSDK.isInitialised()) {
            expandPanelViewInternal(true);
        }
    }

    /**
     * Expand (Open) Panel view
     *
     * @param byUserAction if invoked by user action or automatically
     */
    private void expandPanelViewInternal(boolean byUserAction) {
        if (mIsAnimationRunning == false && mIsOpen == false) {
            // Send log
            Event event = new Event();
            if (byUserAction) {
                event.addParameterToJsonObject(EventParameters.LABEL_VIEW_OPEN_TYPE, EventParameters.MANUAL_OPEN);
            } else {
                event.addParameterToJsonObject(EventParameters.LABEL_VIEW_OPEN_TYPE, EventParameters.AUTOMATIC_OPEN);
            }

            EventManager.getInstance(getContext()).logEvent(getContext(), WidgetType.WIDGET_TYPE_PANEL.getStringValue(), mStyleId, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_WIDGET_VIEW, EventParameters.PANEL_VIEW, false);

            //Refresh the parental lock icon
            boolean isLockActive = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), ParentalLockDialog.PARENTAL_LOCK_ACTIVE_KEY) == null ? false : true;
            mParentalLockButton.setIsLockActive(isLockActive);

            GenAnimator.openPanelView(mContainer, mInnerContainer, mPanelType, new Animator.AnimatorListener() {
                @Override
                public void onAnimationStart(Animator animation) {
                    onExpandPanelAnimationStarted();
                }

                @Override
                public void onAnimationEnd(Animator animation) {
                    mPanelHandle.setPanelButtonState(PanelHandle.PANEL_HANDLE_VIEW_STATE.OPENED);

                    mIsOpen = true;
                    mIsAnimationRunning = false;

                    if (mIOnPanelViewEventListener != null) {
                        mIOnPanelViewEventListener.onPanelViewExpanded();
                    }
                }

                @Override
                public void onAnimationCancel(Animator animation) {

                }

                @Override
                public void onAnimationRepeat(Animator animation) {

                }
            });
        }
    }


    protected void onExpandPanelAnimationStarted() {
        GenAnimator.animateHandleOpen(mPanelHandle, null);
    }


    /**
     * Set panel view configuration
     *
     * @param panelType      panel view type (Possible values <code> PANEL_TYPE.TOP, PANEL_TYPE.BOTTOM, PANEL_TYPE.LEFT ,PANEL_TYPE.RIGHT </code>
     * @param handlePosition handle position  (Possible values <code> HANDLE_POSITION.START, HANDLE_POSITION.CENTER, HANDLE_POSITION.END </code>
     */
    public void setPanelConfiguration(PANEL_TYPE panelType, HANDLE_POSITION handlePosition) {
        if (KidozSDK.isInitialised()) {
            setPanelConfiguration(panelType, handlePosition, null);
        }
    }

    /**
     * Set panel view configuration
     *
     * @param panelType      panel view type (Possible values <code> PANEL_TYPE.TOP, PANEL_TYPE.BOTTOM, PANEL_TYPE.LEFT ,PANEL_TYPE.RIGHT </code>
     * @param handlePosition handle position  (Possible values <code> HANDLE_POSITION.START, HANDLE_POSITION.CENTER, HANDLE_POSITION.END </code>
     * @param customHandle   user defined custom handle iew
     */
    public void setPanelConfiguration(PANEL_TYPE panelType, HANDLE_POSITION handlePosition, View customHandle) {

        if (KidozSDK.isInitialised()) {
            mPanelType = panelType;
            mHandlePosition = handlePosition;
            mPanelHandle.setPanelType(mPanelType);
            mParentalLockButton.setPanelType(PANEL_TYPE.BOTTOM);
            mAboutKidozButton.setPanelType(PANEL_TYPE.BOTTOM);

            mPanelHandle.setVisibility(View.VISIBLE);

            initPanelViewsRules();
        }
    }

    /**
     * Set panel events listener
     *
     * @param iOnPanelViewEventListener listener
     */
    public void setOnPanelViewEventListener(IOnPanelViewEventListener iOnPanelViewEventListener) {
        mIOnPanelViewEventListener = iOnPanelViewEventListener;
    }

    /**
     * Get panel view state
     *
     * @return is panel expanded pr closed
     */
    public boolean getIsPanelViewExpanded() {
        return mIsOpen;
    }


    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        getUiHandlerInstance().removeCallbacksAndMessages(null);
    }


    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();

        if (!isInEditMode() && KidozSDK.isInitialised()) {
            EventManager.getInstance(getContext()).logEvent(getContext(), WidgetType.WIDGET_TYPE_PANEL.getStringValue(), mStyleId, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_BUTTON_VIEW, EventParameters.LABEL_PANEL_BUTTON_VIEW, false);
        }
    }

    @Override
    protected void onWindowVisibilityChanged(int visibility) {
        super.onWindowVisibilityChanged(visibility);

        getMyHandlerInstance().removeCallbacksAndMessages(null);

        if (KidozSDK.isInitialised()) {
            if (visibility == View.VISIBLE) {
                mAllowClickHandle = true;
            } else if (!isInEditMode() && (visibility == View.GONE || visibility == View.INVISIBLE)) {
                EventManager.getInstance(getContext()).logAllTempStoredEvents(getContext());
                EventManager.getInstance(getContext()).startEventsSync(getContext());

                mAllowClickHandle = true;
            }
        }
    }

    /**
     * Preaper pane handle for view
     */
    private void prepareHandle() {

        // Load proper handle button image for close and open state
        mPanelHandle.loadAssets(AssetUtil.getAssetFile(getContext(), mBtnOpenUrl), AssetUtil.getAssetFile(getContext(), mBtnClosedUrl), new PanelHandle.IonPanelHandlePreparedListener() {
            @Override
            public void onPanelPrepared() {
                GenAnimator.playGrowAnimation(mPanelHandle, 350, 0, new Animator.AnimatorListener() {
                    @Override
                    public void onAnimationStart(Animator animation) {
                        mPanelHandle.setPanelButtonState(PanelHandle.PANEL_HANDLE_VIEW_STATE.CLOSED);
                    }

                    @Override
                    public void onAnimationEnd(Animator animation) {
                        mIsOpen = false;
                        mIsAnimationRunning = false;
                        if (mIOnPanelViewEventListener != null) {
                            mIOnPanelViewEventListener.onPanelReady();
                        }
                    }

                    @Override
                    public void onAnimationCancel(Animator animation) {

                    }

                    @Override
                    public void onAnimationRepeat(Animator animation) {

                    }
                });
            }
        });
    }

    private void preparePanel() {

        mIsOpen = true;
        mIsAnimationRunning = true;
        Utils.setOnGlobalLayoutFinishListener(this, new Utils.OnGlobalLayoutFinishedListener() {
            @Override
            public void onLayoutFinished() {

                GenAnimator.closePanelView(mContainer, 0, mInnerContainer, mPanelType, new Animator.AnimatorListener() {
                    @Override
                    public void onAnimationStart(Animator animation) {
                        GenAnimator.animateHandleClose(mPanelHandle, null);
                    }

                    @Override
                    public void onAnimationEnd(Animator animation) {
                        prepareHandle();
                    }

                    @Override
                    public void onAnimationCancel(Animator animation) {

                    }

                    @Override
                    public void onAnimationRepeat(Animator animation) {

                    }
                });
            }
        });
    }

    /**
     * Get handler instance
     */
    protected Utils.StaticHandler getMyHandlerInstance() {
        if (mStaticHandler == null) {
            mStaticHandler = new Utils.StaticHandler(Looper.getMainLooper()) {
                @Override
                public void handleMessage(Message message) {
                    // This is where you do your work in the UI thread.
                    // Your worker tells you in the message what to do.
                }
            };
        }
        return mStaticHandler;
    }

    /**
     * Get handler instance
     */
    protected Utils.StaticHandler getUiHandlerInstance() {
        if (mUiLoaderStaticHandler == null) {
            mUiLoaderStaticHandler = new Utils.StaticHandler(Looper.getMainLooper()) {
                @Override
                public void handleMessage(Message message) {
                    // This is where you do your work in the UI thread.
                    // Your worker tells you in the message what to do.
                }
            };
        }
        return mUiLoaderStaticHandler;
    }


    public static final String BTN_OPENED_URL = "btnOpenedUrl";
    public static final String BTN_CLOSED_URL = "btnClosedUrl";
    public static final String BTN_SCALE_SIZE = "btnScaleSize";
    public static final String BTN_ANIMATION_TYPE = "btnAnimType";
    public static final String PANEL_BG_COLOR = "panelBgClr";
    public static final String LABEL_TEXT = "labelText";
    public static final String LABEL_TEXT_COLOR = "labelColor";
    public static final String STYLE_ID = "style_id";
    public static final String OVERRIDE_USER_PROPERTIES = "overrideUserProperties";

}
