package com.kidoz.sdk.api;


import android.app.Activity;
import android.content.Context;
import android.util.Log;
import android.view.View;

import com.kidoz.events.EventManager;
import com.kidoz.sdk.api.general.EventMessage;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.interfaces.IOnFeedViewEventListener;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.kidoz.sdk.api.structure.ContentData;
import com.kidoz.sdk.api.structure.SDKConfigurationData;

import org.json.JSONObject;

import de.greenrobot.event.EventBus;

public class FeedView
{
    // Properties keys
    public static final String KIDOZ_FEED_BUTTON = "feedButton";
    public static final String KIDOZ_FEED_BUTTON_SCALE = "feedButtonScale";
    public static final String OVERRIDE_USER_PROPERTIES = "overrideUserProperties";

    public static final String TAG = FeedView.class.getSimpleName();
    public static final String STYLE_ID = "style_id";

    private Builder mBuilder;

    private FeedViewDialog mFeedViewDialog;
    private FeedFamilyViewDialog mFamilyViewDialog;
    protected ContentLogicLoader mContentLogicLoader;

    protected IOnFeedViewEventListener mOnInterstitialViewEventListener;
    protected boolean mIsAutomaticInvocation = false;

    protected String mFeedViewType = null;
    /**
     * Flag used to check ,is Pre load or Post load the content data
     */
    protected boolean mIsPostLoadData = false;
    protected boolean mShowWhenInitiated = false;
    protected JSONObject mJsonObjectProperties;
    protected static boolean mOncePerSessionLoaded = false;

    protected FeedView(Builder builder)
    {
        mBuilder = builder;

        initContentLogicLoader();

        // Has been initialised
        if (KidozSDK.isInitialised() == true)
        {
            checkIfValidationNeededAndInit();
        }
        else
        {
            EventBus.getDefault().register(this);
        }
    }

    /**
     * Check if needed a validation
     */
    private void checkIfValidationNeededAndInit()
    {
        if (mOncePerSessionLoaded == true)
        {
            mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
            if (mJsonObjectProperties != null)
            {
                mFeedViewType = mJsonObjectProperties.optString(FeedView.TAG);

                if (mShowWhenInitiated)
                {
                    showView();
                }

                EventBus.getDefault().post(new EventMessage(EventMessage.MessageType.INIT_FEED_BUTTON));

                if (mOnInterstitialViewEventListener != null)
                {
                    mOnInterstitialViewEventListener.onViewReady();
                }

            }
            else
            {
                validateLayoutAndAssets();
            }
        }
        else
        {
            validateLayoutAndAssets();
        }
    }

    // Event bus callback
    public void onEvent(SDKConfigurationData event)
    {
        unregisterEventBus();

        checkIfValidationNeededAndInit();
    }

    private void unregisterEventBus()
    {
        EventBus.getDefault().unregister(this);
    }

    private void validateLayoutAndAssets()
    {
        SdkAPIManager.getSdkApiInstance(mBuilder.mContext).getStyle(mBuilder.mContext, WidgetType.WIDGET_TYPE_FEED.getValue(), new ApiResultCallback<Boolean>()
        {
            @Override
            public void onServerResult(ResultData<?> result)
            {
                // If result success notify wrapper to build the banner.
                if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true)
                {
                    if (result.getData() != null)
                    {
                        boolean isAssetsReady = (Boolean) result.getData();
                        if (isAssetsReady == true)
                        {
                            mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
                            if (mJsonObjectProperties != null)
                            {
                                mFeedViewType = mJsonObjectProperties.optString(FeedView.TAG);

                                EventBus.getDefault().post(new EventMessage(EventMessage.MessageType.INIT_FEED_BUTTON));
                                mOncePerSessionLoaded = true;

                                if (mOnInterstitialViewEventListener != null)
                                {
                                    mOnInterstitialViewEventListener.onViewReady();
                                }

                                if (mShowWhenInitiated)
                                {
                                    showView();
                                }
                            }
                        }
                        else
                        {
                            onFailed();
                        }
                    }
                }
                else
                {
                    onFailed();
                }
            }

            @Override
            public void onFailed()
            {

            }
        });
    }


    /**
     * Initiate  content loader logic ,that handles all the logic for loading content
     **/
    private void initContentLogicLoader()
    {
        mContentLogicLoader = new ContentLogicLoader(new ContentLogicLoader.IOnContentDataReadyCallback()
        {
            @Override
            public void onDataReady(ContentData contentData)
            {
                if (mFeedViewType.equals(FeedViewDialog.TAG))
                {
                    mFeedViewDialog.setContentData(contentData);
                }
                else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
                {
                    mFamilyViewDialog.setContentData(contentData);
                }


                if (mIsPostLoadData == false)
                {
                    if (mOnInterstitialViewEventListener != null)
                    {
                        mOnInterstitialViewEventListener.onReadyToShow();
                    }
                }
            }

            @Override
            public void onLoadContentFailed()
            {

            }
        });
    }


    /**
     * Show feed interstitial view (Will be shown <b>Before</b> content data is ready)
     **/
    public void showView()
    {
        if (KidozSDK.isInitialised())
        {
            mShowWhenInitiated = false;

            mIsAutomaticInvocation = true;
            mIsPostLoadData = true;

            if (mOnInterstitialViewEventListener != null)
            {
                mOnInterstitialViewEventListener.onReadyToShow();
            }

            openSimpleDialog();
        }
        else
        {
            mShowWhenInitiated = true;
        }
    }

    /**
     * Create and open simple (Not fragment based) Dialog view
     */
    protected void openSimpleDialog()
    {
        if (mBuilder.mContext != null && mBuilder.mContext instanceof Activity)
        {
            if (mFeedViewType != null)
            {
                if (mFeedViewType.equals(FeedViewDialog.TAG))
                {
                    if (mFeedViewDialog == null || mFeedViewDialog.isShowing() == false)
                    {
                        mFeedViewDialog = new FeedViewDialog(mBuilder.mContext, mOnInterstitialViewEventListener, mIsAutomaticInvocation, mJsonObjectProperties);
                        assignCloseButtonClickListener();
                        mFeedViewDialog.openDialog();
                    }
                }
                else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
                {
                    if (mFamilyViewDialog == null || mFamilyViewDialog.isShowing() == false)
                    {
                        mFamilyViewDialog = new FeedFamilyViewDialog(mBuilder.mContext, mOnInterstitialViewEventListener, mIsAutomaticInvocation, mJsonObjectProperties);
                        assignCloseButtonClickListener();
                        mFamilyViewDialog.openDialog();
                    }
                }

                if (mJsonObjectProperties == null)
                {
                    mJsonObjectProperties = DatabaseManager.getInstance(mBuilder.mContext).getConfigTable().loadProperties(FeedView.TAG);
                }

                if (mJsonObjectProperties != null)
                {
                    mContentLogicLoader.loadContent(mBuilder.mContext,WidgetType.WIDGET_TYPE_FEED.getStringValue(), mJsonObjectProperties.optString(STYLE_ID));
                }
            }
            else
            {
                Log.e(SDKLogger.GENERAL_TAG, "Feed view object has not been initiated yet !");
            }
        }
    }


    private void assignCloseButtonClickListener()
    {
        if (mFeedViewType != null)
        {
            if (mFeedViewType.equals(FeedViewDialog.TAG))
            {
                mFeedViewDialog.setOnCloseButtonClickListener(new View.OnClickListener()
                {
                    @Override
                    public void onClick(View view)
                    {
                        dismissView();
                        EventManager.getInstance(mBuilder.mContext).clearAllTempStoredEvents();
                        unregisterEventBus();
                    }
                });
            }
            else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
            {
                mFamilyViewDialog.setOnCloseButtonClickListener(new View.OnClickListener()
                {
                    @Override
                    public void onClick(View view)
                    {
                        dismissView();
                        EventManager.getInstance(mBuilder.mContext).clearAllTempStoredEvents();
                        unregisterEventBus();
                    }
                });
            }
        }
    }

    /**
     * Dismiss view
     **/
    public void dismissView()
    {
        if (mFeedViewType != null)
        {
            if (mFeedViewType.equals(FeedViewDialog.TAG))
            {
                if (mFeedViewDialog != null)
                {
                    mFeedViewDialog.closeDialog();
                }
            }
            else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
            {
                if (mFamilyViewDialog != null)
                {
                    mFamilyViewDialog.closeDialog();
                }
            }
        }

        if (mOnInterstitialViewEventListener != null)
        {
            mOnInterstitialViewEventListener.onDismissView();
        }
    }

    /**
     * Set on view event listener
     *
     * @param listener
     **/
    public void setOnFeedViewEventListener(IOnFeedViewEventListener listener)
    {
        mOnInterstitialViewEventListener = listener;
        if (mFeedViewType != null)
        {
            if (mFeedViewType.equals(FeedViewDialog.TAG))
            {
                if (mFeedViewDialog != null)
                {
                    mFeedViewDialog.setOnInterstitialViewEventListener(mOnInterstitialViewEventListener);
                }
            }
            else if (mFeedViewType.equals(FeedFamilyViewDialog.TAG))
            {
                if (mFamilyViewDialog != null)
                {
                    mFamilyViewDialog.setOnInterstitialViewEventListener(mOnInterstitialViewEventListener);
                }
            }
        }
    }

    /**
     * Builder class that handles view initiation
     **/
    public static class Builder
    {
        private Context mContext;


        public Builder(Context context)
        {
            mContext = context;
        }

        public FeedView build()
        {
            return new FeedView(this);
        }
    }

}
