package com.kidoz.sdk.api.ui_views.parental_lock;

import android.content.Context;
import android.graphics.BitmapFactory;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.RelativeLayout;

import com.kidoz.events.DeviceUtils;
import com.kidoz.sdk.api.picasso_related.PicassoOk;
import com.kidoz.sdk.api.server_connect.IOnButtonStyleCallback;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.kidoz.sdk.api.ui_views.custom_drawables.KidozLogoDrawable;
import com.squareup.imagelib.MemoryPolicy;

import java.io.File;
import java.io.IOException;

import pl.droidsonroids.gif.GifDrawable;

/**
 * Created by Eugine on 11/5/2015.
 */
public class StyleableButtonView extends RelativeLayout
{

    private GifDrawable mAnimatedGifDrawable1 = null;
    private GifDrawable mAnimatedGifDrawable2 = null;

    protected ImageView mImageView1;
    protected ImageView mImageView2;

    protected Drawable mDefaultDrawable1;
    protected Drawable mDefaultDrawable2;

    private int mDefaultSize;

    public enum ButtonState
    {
        FIRST, SECOND
    }

    public StyleableButtonView(Context context)
    {
        super(context);
        initView();
    }

    public StyleableButtonView(Context context, AttributeSet attrs)
    {
        super(context, attrs);
        initView();
    }

    public StyleableButtonView(Context context, AttributeSet attrs, int defStyleAttr)
    {
        super(context, attrs, defStyleAttr);
        initView();
    }

    private void initView()
    {
        mImageView1 = new ImageView(getContext());
        mImageView1.setScaleType(ImageView.ScaleType.FIT_XY);
        mImageView2 = new ImageView(getContext());
        mImageView2.setScaleType(ImageView.ScaleType.FIT_XY);

        if (mImageView1.getId() == -1)
        {
            mImageView1.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        }

        if (mImageView2.getId() == -1)
        {
            mImageView2.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        }

        if (mAnimatedGifDrawable1 != null)
        {
            SdkAPIManager.getSdkApiInstance(getContext()).cancelStyleLoadingRequest(mImageView1.getId() + "1");
            mAnimatedGifDrawable1.recycle();
            mAnimatedGifDrawable1 = null;
            mImageView1.setImageDrawable(null);
        }

        if (mAnimatedGifDrawable2 != null)
        {
            SdkAPIManager.getSdkApiInstance(getContext()).cancelStyleLoadingRequest(mImageView1.getId() + "2");
            mAnimatedGifDrawable2.recycle();
            mAnimatedGifDrawable2 = null;
            mImageView2.setImageDrawable(null);
        }

        RelativeLayout.LayoutParams relParams = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        addView(mImageView1, relParams);
        addView(mImageView2, relParams);

        mImageView1.setVisibility(View.VISIBLE);
        mImageView2.setVisibility(View.INVISIBLE);

        // Added to bve able and see the default view of the button in xml editor mode
        if (isInEditMode())
        {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
            {
                mImageView1.setBackground(new KidozLogoDrawable());
            }
            else
            {
                mImageView1.setBackgroundDrawable(new KidozLogoDrawable());
            }
        }
    }

    public void setImageViewScaleType(ImageView.ScaleType scaleType)
    {
        mImageView1.setScaleType(scaleType);
        mImageView2.setScaleType(scaleType);
    }

    /**
     * Load Button style for single state image
     */
    public void loadButtonStyle(int buttonWidthSize, String url, final Drawable defaultImageDrawable, String tag)
    {

        mDefaultSize = buttonWidthSize;
        getLayoutParams().width = buttonWidthSize;
        getLayoutParams().height = buttonWidthSize;

        mImageView1.getLayoutParams().height = buttonWidthSize;
        mImageView1.getLayoutParams().width = buttonWidthSize;
        ((RelativeLayout.LayoutParams) mImageView1.getLayoutParams()).addRule(RelativeLayout.CENTER_IN_PARENT);

        mDefaultDrawable1 = defaultImageDrawable;
        loadStyleableImage(buttonWidthSize, ButtonState.FIRST, url, "temp_Style1" + tag);
    }

    /**
     * Load Button style for dual state image
     */
    public void loadButtonStyle(int buttonWidthSize, String url_main, final Drawable defaultImageDrawable_main, String url_secondary, final Drawable defaultImageDrawable_secondary, String tag)
    {

        mDefaultSize = buttonWidthSize;
        getLayoutParams().width = buttonWidthSize;
        getLayoutParams().height = buttonWidthSize;

        //setBackgroundColor(Color.CYAN);

        mImageView1.getLayoutParams().width = buttonWidthSize;
        mImageView1.getLayoutParams().height = buttonWidthSize;
        mImageView2.getLayoutParams().width = buttonWidthSize;
        mImageView2.getLayoutParams().height = buttonWidthSize;

        mDefaultDrawable1 = defaultImageDrawable_main;
        mDefaultDrawable2 = defaultImageDrawable_secondary;
        loadStyleableImage(buttonWidthSize, ButtonState.FIRST, url_main, "temp_Style1" + tag);
        loadStyleableImage(buttonWidthSize, ButtonState.SECOND, url_secondary, "temp_Style2" + tag);
    }


    /**
     * Load styleable image for a button
     */
    private void loadStyleableImage(final int buttonWidthSize, final ButtonState buttonState, String url, String savedFileName)
    {
        boolean loadDefault = true;
        if (url != null && url.equals("") == false)
        {
            // 1. Check if valid link and of Gif type
            if (isValidStyleFormat(url))
            {
                // Extract file name
                String realFileName = url.substring(url.lastIndexOf("/") + 1, url.length());

                String preFix = savedFileName;
                String postFix = "";
                int index = realFileName.lastIndexOf(".");
                if (index != -1)
                {
                    postFix = realFileName.substring(index, realFileName.length());
                }

                String styleLocalFileName = preFix + postFix;

                File storageLocation = DeviceUtils.getInnerStoragePathIfPosible(getContext());
                //1.1 Check if storage exists
                if (storageLocation != null)
                {
                    File styleFilePath = new File(storageLocation, styleLocalFileName);
                    //1.2 Check image or animation exists
                    if (styleFilePath.exists())
                    {
                        // 1.3 Check if image or animation is a valid file that has size
                        if (styleFilePath.length() > 0)
                        {
                            // 1.4.1 If we have new animation file we want to delete the previous and rename the new to the same name
                            File tempAnimFile = new File(storageLocation, realFileName);
                            // 1.4.2  We check if we have a new image or file downloaded from server and ready to be swaped we the used one
                            if (tempAnimFile != null && tempAnimFile.exists())
                            {
                                styleFilePath.delete();
                                tempAnimFile.renameTo(styleFilePath);
                            }
                        }

                        // 1.5 Don't load default button view
                        loadDefault = false;

                        // 1.8 Get saved image ratio and correct the button view size
                        applyCorrectAndCorrectViewsSizeByImageRatio(buttonWidthSize, buttonState, styleFilePath.getAbsolutePath());

                        // 1.7 Load save image or animation to view
                        loadStyleImageOrAnimationToView(buttonState, styleFilePath);

                        // 1.8 Launch a thread that checks if need to download a custom button style image or animation, Check if the file exists checks if it needed to be updated
                        SdkAPIManager.getSdkApiInstance(getContext()).loadButtonStyle(getContext(), url, realFileName, styleLocalFileName, getId() + "", null, false);

                    }
                    else
                    {
                        // 1.8 Don't load default button view
                        loadDefault = false;
                        // 1.9 Launch a thread that download a custom button style image or animation
                        // In case of failure set the default style view
                        SdkAPIManager.getSdkApiInstance(getContext()).loadButtonStyle(getContext(), url, realFileName, styleLocalFileName, getId() + "", new IOnButtonStyleCallback()
                        {
                            @Override public void onImageLoaded(File file)
                            {

                                if (file != null && file.exists())
                                {

                                    //Get saved image ratio and correct the button view size
                                    applyCorrectAndCorrectViewsSizeByImageRatio(buttonWidthSize, buttonState, file.getAbsolutePath());

                                    loadStyleImageOrAnimationToView(buttonState, file);
                                }
                                else
                                {

                                    setBackgroundDrawableCompat(buttonState);
                                }
                            }
                        }, false);
                    }
                }
            }
        }

        // 2.0 Load the default style button view
        if (loadDefault == true)
        {
            setBackgroundDrawableCompat(buttonState);
        }
    }

    /**
     * Apply correct button view suze calculated by the ratio of the loaded image style
     */
    private void applyCorrectAndCorrectViewsSizeByImageRatio(int buttonWidth, final ButtonState buttonState, String filePath)
    {
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeFile(filePath, options);
        float ratio = options.outHeight / ((float) options.outWidth);

        int viewHeight = (int) (buttonWidth * ratio);

        getLayoutParams().height = viewHeight;
        getLayoutParams().width = buttonWidth;

        switch (buttonState)
        {
            case FIRST:
            {
                mImageView1.getLayoutParams().height = viewHeight;
                mImageView1.getLayoutParams().width = buttonWidth;
                break;
            }

            case SECOND:
            {
                mImageView2.getLayoutParams().height = viewHeight;
                mImageView2.getLayoutParams().width = buttonWidth;
                break;
            }
        }
    }

    /**
     * Load style image or animation to the view
     */
    private void loadStyleImageOrAnimationToView(ButtonState buttonState, File styleFilePath)
    {
        if (styleFilePath.getAbsolutePath().contains(".gif") || styleFilePath.getAbsolutePath().contains(".GIF"))
        {
            if (styleFilePath.length() > 0)
            {
                try
                {
                    switch (buttonState)
                    {
                        case FIRST:
                        {
                            mAnimatedGifDrawable1 = new GifDrawable(styleFilePath);
                            if (mAnimatedGifDrawable1 != null && !mAnimatedGifDrawable1.isRecycled())
                            {
                                mImageView1.setImageDrawable(mAnimatedGifDrawable1);
                                mAnimatedGifDrawable1.start();
                            }
                            break;
                        }

                        case SECOND:
                        {
                            mAnimatedGifDrawable2 = new GifDrawable(styleFilePath);
                            if (mAnimatedGifDrawable2 != null && !mAnimatedGifDrawable2.isRecycled())
                            {
                                mImageView2.setImageDrawable(mAnimatedGifDrawable2);
                                mAnimatedGifDrawable2.start();
                            }
                            break;
                        }
                    }

                    clearBackgroundDrawableCompat(buttonState);
                    // Create animation drawable from newly downloaded file

                }
                catch (IOException e)
                {
                    // If something went wrong delete the file
                    if (styleFilePath != null && styleFilePath.exists() == true)
                    {
                        styleFilePath.delete();
                    }

                    setBackgroundDrawableCompat(buttonState);
                }
            }
            else
            {
                styleFilePath.delete();
                setBackgroundDrawableCompat(buttonState);
            }
        }
        else if (styleFilePath.getAbsolutePath().contains(".png") || styleFilePath.getAbsolutePath().contains(".PNG"))
        {

            switch (buttonState)
            {
                case FIRST:
                {
                    PicassoOk.getPicasso(getContext()).load(styleFilePath).noFade().memoryPolicy(MemoryPolicy.NO_CACHE, MemoryPolicy.NO_STORE).into(mImageView1);
                    break;
                }

                case SECOND:
                {
                    PicassoOk.getPicasso(getContext()).load(styleFilePath).noFade().memoryPolicy(MemoryPolicy.NO_CACHE, MemoryPolicy.NO_STORE).into(mImageView2);
                    break;
                }
            }
            clearBackgroundDrawableCompat(buttonState);
        }
    }


    /**
     * Check if valid animation or image format
     */
    private boolean isValidStyleFormat(String url)
    {
        return url != null && url.contains(".gif") || (url != null ? url.contains(".GIF") : false) || (url != null ? url.contains(".png") : false) || (url != null ? url.contains(".PNG") : false);

    }

    /**
     * Set correct drawable to a proper layer image view
     */
    private void setBackgroundDrawableCompat(ButtonState buttonState)
    {
        switch (buttonState)
        {
            case FIRST:
            {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
                {
                    mImageView1.setBackground(mDefaultDrawable1);
                }
                else
                {
                    mImageView1.setBackgroundDrawable(mDefaultDrawable1);
                }
                mImageView1.postInvalidate();
                break;
            }

            case SECOND:
            {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
                {
                    mImageView2.setBackground(mDefaultDrawable2);
                }
                else
                {
                    mImageView2.setBackgroundDrawable(mDefaultDrawable2);
                }
                mImageView2.postInvalidate();
                break;
            }
        }
    }

    /**
     * Clear drawable from a proper layer image view
     */
    private void clearBackgroundDrawableCompat(ButtonState buttonState)
    {
        switch (buttonState)
        {
            case FIRST:
            {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
                {
                    mImageView1.setBackground(null);
                }
                else
                {
                    mImageView1.setBackgroundDrawable(null);
                }
                break;
            }

            case SECOND:
            {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
                {
                    mImageView2.setBackground(null);
                }
                else
                {
                    mImageView2.setBackgroundDrawable(null);
                }
                break;
            }
        }
    }


    @Override protected void onVisibilityChanged(View changedView, int visibility)
    {
        super.onVisibilityChanged(changedView, visibility);

        if (visibility == View.VISIBLE)
        {
            changeGifAnimationState(true);
        }
        else
        {
            changeGifAnimationState(false);
        }
    }

    @Override protected void onFocusChanged(boolean gainFocus, int direction, Rect previouslyFocusedRect)
    {
        super.onFocusChanged(gainFocus, direction, previouslyFocusedRect);

        if (gainFocus)
        {
            changeGifAnimationState(true);
        }
    }

    private void changeGifAnimationState(boolean isStateState)
    {
        if (mAnimatedGifDrawable1 != null && !mAnimatedGifDrawable1.isRecycled())
        {
            if (isStateState)
            {
                if (!mAnimatedGifDrawable1.isPlaying())
                {
                    mAnimatedGifDrawable1.start();
                }
            }
            else
            {
                mAnimatedGifDrawable1.stop();
            }
        }

        if (mAnimatedGifDrawable2 != null && !mAnimatedGifDrawable2.isRecycled())
        {
            if (isStateState)
            {
                if (!mAnimatedGifDrawable2.isPlaying())
                {
                    mAnimatedGifDrawable2.start();
                }
            }
            else
            {
                mAnimatedGifDrawable2.stop();
            }
        }
    }

    @Override protected void onDetachedFromWindow()
    {
        super.onDetachedFromWindow();

        if (mAnimatedGifDrawable1 != null && !mAnimatedGifDrawable1.isRecycled())
        {
            mAnimatedGifDrawable1.recycle();
            mAnimatedGifDrawable1 = null;
        }

        if (mAnimatedGifDrawable2 != null && !mAnimatedGifDrawable2.isRecycled())
        {
            mAnimatedGifDrawable2.recycle();
            mAnimatedGifDrawable2 = null;
        }
    }

    /**
     * Show the cuurent visible state of the button
     *
     * @param buttonState
     */
    public void showButtonState(ButtonState buttonState)
    {
        if (buttonState == ButtonState.FIRST)
        {
            mImageView1.setVisibility(View.VISIBLE);
            mImageView2.setVisibility(View.INVISIBLE);

            if (mAnimatedGifDrawable1 != null && !mAnimatedGifDrawable1.isRecycled())
            {
                if (!mAnimatedGifDrawable1.isPlaying())
                {
                    mAnimatedGifDrawable1.start();
                }
            }

            if (mAnimatedGifDrawable1 != null && !mAnimatedGifDrawable1.isRecycled())
            {
                mAnimatedGifDrawable1.stop();
            }
        }
        else if (buttonState == ButtonState.SECOND)
        {

            mImageView1.setVisibility(View.INVISIBLE);
            mImageView2.setVisibility(View.VISIBLE);

            if (mAnimatedGifDrawable2 != null && !mAnimatedGifDrawable2.isRecycled())
            {
                if (!mAnimatedGifDrawable2.isPlaying())
                {
                    mAnimatedGifDrawable2.start();
                }
            }

            if (mAnimatedGifDrawable2 != null && !mAnimatedGifDrawable2.isRecycled())
            {
                mAnimatedGifDrawable2.stop();
            }
        }
    }

}
