package com.kidoz.sdk.api.ui_views;

import android.content.Context;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;
import android.widget.ImageView;

import com.kidoz.events.DeviceUtils;
import com.kidoz.sdk.api.picasso_related.PicassoOk;
import com.kidoz.sdk.api.server_connect.IOnButtonStyleCallback;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.squareup.imagelib.MemoryPolicy;

import java.io.File;
import java.io.IOException;

import pl.droidsonroids.gif.GifDrawable;

/**
 * Created by Eugine on 8/17/2015.
 */
public class ButtonStyleableImageView extends ImageView {

    private GifDrawable mAnimatedGifDrawable = null;

    public ButtonStyleableImageView(Context context) {
        super(context);
        initView();
    }

    public ButtonStyleableImageView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initView();
    }

    public ButtonStyleableImageView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initView();
    }

    private void initView() {
        setScaleType(ScaleType.FIT_CENTER);

        if (isInEditMode() == false) {
            if (getId() == -1) {
                setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
            }

            if (mAnimatedGifDrawable != null) {
                SdkAPIManager.getSdkApiInstance(getContext()).cancelStyleLoadingRequest(getId() + "");
                mAnimatedGifDrawable.recycle();
                mAnimatedGifDrawable = null;
                setImageDrawable(null);
            }
        }
    }


    /**
     * Load stylable image for a button
     */
    public void loadStyleableImage(String url, final Drawable defaultImageDrawable) {
        boolean loadDefault = true;
        if (url != null && url.equals("") == false) {
            // Check if valid link and of Gif type
            if (isValidStyleFormat(url)) {
                // Extract file name
                String defaultName = url.substring(url.lastIndexOf("/") + 1, url.length());
                String postFix = "";
                String preFix = "temp_styleImg";

                int index = defaultName.lastIndexOf(".");
                if(index != -1) {
                    postFix = defaultName.substring(index, defaultName.length());
                }

                String styleLocalFileName = preFix + postFix;

                File storageLocation = DeviceUtils.getInnerStoragePathIfPosible(getContext());
                if (storageLocation != null) {
                    File styleFilePath = new File(storageLocation, styleLocalFileName);

                    if (styleFilePath.exists()) {
                        // Replace the newly downloaded file with the previous one that used for showing
                        if (styleFilePath.length() > 0) {
                            // If we have new animation file we want to delete the previous and rename the new to the same name
                            File tempAnimFile = new File(storageLocation, defaultName);
                            if (tempAnimFile != null && tempAnimFile.exists()) {
                                styleFilePath.delete();
                                tempAnimFile.renameTo(styleFilePath);
                            }
                        }

                        loadDefault = false;
                        loadStyleImageOrAnimationToView(styleFilePath);

                        SdkAPIManager.getSdkApiInstance(getContext()).loadButtonStyle(getContext(), url, defaultName, styleLocalFileName, getId() + "", null,false);

                    } else {

                        loadDefault = false;
                        SdkAPIManager.getSdkApiInstance(getContext()).loadButtonStyle(getContext(), url, defaultName, styleLocalFileName, getId() + "", new IOnButtonStyleCallback() {
                            @Override
                            public void onImageLoaded(File file) {

                                if(file != null && file.exists()) {
                                    loadStyleImageOrAnimationToView(file);
                                } else {

                                    setBackgroundDrawableCompat(defaultImageDrawable);
                                }
                            }
                        },false);
                    }
                }
            }
        }

        if(loadDefault == true) {
            setBackgroundDrawableCompat(defaultImageDrawable);
        }
    }


    /**
     * Load style image or animation to the view
     */
    private void loadStyleImageOrAnimationToView(File styleFilePath) {
        if (styleFilePath.getAbsolutePath().contains(".gif") || styleFilePath.getAbsolutePath().contains(".GIF")) {
            if (styleFilePath.length() > 0) {
                try {
                    // Create animation drawable from newly downloaded file
                    mAnimatedGifDrawable = new GifDrawable(styleFilePath);
                    if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled()) {
                        setImageDrawable(mAnimatedGifDrawable);
                        mAnimatedGifDrawable.start();

                        setBackgroundDrawableCompat(null);
                    }

                } catch (IOException e) {
                    // If something went wrong delete the file
                    if (styleFilePath != null && styleFilePath.exists() == true) {
                        styleFilePath.delete();
                    }
                }
            } else {
                styleFilePath.delete();
            }
        } else if (styleFilePath.getAbsolutePath().contains(".png") || styleFilePath.getAbsolutePath().contains(".PNG")) {
            PicassoOk.getPicasso(getContext())
                    .load(styleFilePath)
                    .noFade()
                    .memoryPolicy(MemoryPolicy.NO_CACHE,MemoryPolicy.NO_STORE)
                    .into(ButtonStyleableImageView.this);
            setBackgroundDrawableCompat(null);
        }
    }

    /**
     * Check if valid animation or image format
     */
    private boolean isValidStyleFormat(String url) {
        return url != null && url.contains(".gif")
                || (url != null ? url.contains(".GIF") : false)
                || (url != null ? url.contains(".png") : false)
                || (url != null ? url.contains(".PNG") : false);

    }


    private void setBackgroundDrawableCompat(Drawable drawable) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            setBackground(drawable);
        } else {
            setBackgroundDrawable(drawable);
        }
    }


    public void startGifAnimation() {
        if (mAnimatedGifDrawable != null) {
            mAnimatedGifDrawable.stop();
        }
    }

    public void stopGifAnimation() {
        if (mAnimatedGifDrawable != null) {
            mAnimatedGifDrawable.start();
        }
    }

    public void pauseGifAnimation() {
        if (mAnimatedGifDrawable != null) {
            if (mAnimatedGifDrawable.canPause()) {
                mAnimatedGifDrawable.pause();
            }
        }
    }


    @Override
    protected void onWindowVisibilityChanged(int visibility) {
        super.onWindowVisibilityChanged(visibility);

      /*  if (visibility == View.VISIBLE) {
            changeGifAnimationState(true);
        } else {
            changeGifAnimationState(false);
        }*/
    }

    @Override
    protected void onVisibilityChanged(View changedView, int visibility) {
        super.onVisibilityChanged(changedView, visibility);

        if (visibility == View.VISIBLE) {
            changeGifAnimationState(true);
        } else {
            changeGifAnimationState(false);
        }
    }

    @Override
    protected void onFocusChanged(boolean gainFocus, int direction, Rect previouslyFocusedRect) {
        super.onFocusChanged(gainFocus, direction, previouslyFocusedRect);

        if (gainFocus) {
            changeGifAnimationState(true);
        }
    }

    private void changeGifAnimationState(boolean isStateState) {
        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled()) {
            if (isStateState) {
                if (!mAnimatedGifDrawable.isPlaying()) {
                    mAnimatedGifDrawable.start();
                }
            } else {
                mAnimatedGifDrawable.stop();
            }
        }
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();

        if (mAnimatedGifDrawable != null && !mAnimatedGifDrawable.isRecycled()) {
            mAnimatedGifDrawable.recycle();
            mAnimatedGifDrawable = null;
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
    }

}
