package com.kidoz.sdk.api.general.utils;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.graphics.Point;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.Display;
import android.view.View;
import android.view.ViewTreeObserver;
import android.view.WindowManager;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Created by User on 13/07/2015.
 */
public class Utils
{
    private static final String TAG = Utils.class.getSimpleName();
    private static final AtomicInteger sNextGeneratedId = new AtomicInteger(1);

    /**
     * Generate a value suitable for use in  #setId(int).
     * This value will not collide with ID values generated at build time by aapt for R.id.
     *
     * @return a generated ID value
     */
    public static int generateViewId()
    {
        for (; ; )
        {
            final int result = sNextGeneratedId.get();
            // aapt-generated IDs have the high byte nonzero; clamp to the range under that.
            int newValue = result + 1;
            if (newValue > 0x00FFFFFF)
            {
                newValue = 1; // Roll over to 1, not 0.
            }
            if (sNextGeneratedId.compareAndSet(result, newValue))
            {
                return result;
            }
        }
    }

    public static class StaticHandler extends Handler
    {
         public StaticHandler(Looper looper) {
             super(looper);
         }

        @Override public void handleMessage(Message msg)
        {
            super.handleMessage(msg);
        }
    }


    /**
     * Convert amount to proper pixel amount according to device DP
     *
     * @param dp size in DP
     * @return size in pixels
     */
    public static int dpTOpx(Context context, int dp)
    {
        return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dp, context.getResources().getDisplayMetrics());
    }

    /**
     * Convert amount to proper pixel amount according to device DP
     *
     * @param dp size in DP
     * @return size in pixels
     */
    public static int dpTOpx(Resources resources, int dp)
    {
        return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dp, resources.getDisplayMetrics());
    }

    /**
     * Convert amount to proper pixel amount according to device SP
     *
     * @param sp size in SP
     * @return size in pixels
     */
    public static int spTOpx(Context context, int sp)
    {
        return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP, sp, context.getResources().getDisplayMetrics());
    }


    /**
     * Create correct uri from and object (Object can be and url , file path or local resource and etc)
     *
     * @param res different resource type
     */
    public static Uri prepareCorrectUri(Object res)
    {
        if (res != null)
        {
            if (res instanceof Uri)
            {
                return (Uri) res;
            }
            else if (res instanceof Integer)
            {
                return Uri.parse("res:///" + res);
            }
            else if (res instanceof File)
            {
                return Uri.fromFile((File) res);
            }
            else
            {
                int val = -1;
                try
                {
                    val = Integer.parseInt((String) res);
                }
                catch (NumberFormatException nfe)
                {
                }

                if (val != -1)
                {
                    return Uri.parse("res:///" + val);
                }
                else
                {
                    if (((String) res).startsWith("http://") || ((String) res).startsWith("https://"))
                    {
                        return Uri.parse((String) res);
                    }
                    else
                    {
                        if (((String) res).contains("mnt/") || ((String) res).contains("storage/"))
                        {
                            return Uri.fromFile(new File(((String) res)));
                        }
                        else
                        {
                            return Uri.parse("asset:///" + res);
                        }
                    }
                }
            }
        }
        else
        {
            return Uri.EMPTY;
        }

    }


    /**
     * Get screen size
     *
     * @return Point with device size values
     */
    @SuppressLint("NewApi") public static Point getScreenSize(Context context)
    {
        Point size = new Point();
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = windowManager.getDefaultDisplay();

        if (android.os.Build.VERSION.SDK_INT >= 19)
        {
            display.getRealSize(size);
        }
        else
        {
            size.x = display.getWidth();
            size.y = display.getHeight();
        }

        return size;
    }


    /**
     * Get device size in inches
     *
     * @param context context
     * @return size in icnes
     */
    public static double getDeviceScreenSizeInInches(Context context)
    {
        int mWidthPixels;
        int mHeightPixels;

        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = windowManager.getDefaultDisplay();
        DisplayMetrics displayMetrics = new DisplayMetrics();
        display.getMetrics(displayMetrics);

        // since SDK_INT = 1;
        mWidthPixels = displayMetrics.widthPixels;
        mHeightPixels = displayMetrics.heightPixels;

        // includes window decorations (statusbar bar/menu bar)
        if (Build.VERSION.SDK_INT >= 14 && Build.VERSION.SDK_INT < 17)
        {
            try
            {
                mWidthPixels = (Integer) Display.class.getMethod("getRawWidth").invoke(display);
                mHeightPixels = (Integer) Display.class.getMethod("getRawHeight").invoke(display);
            }
            catch (Exception ignored)
            {
            }
        }

        // includes window decorations (statusbar bar/menu bar)
        if (Build.VERSION.SDK_INT >= 17)
        {
            try
            {
                Point realSize = new Point();
                Display.class.getMethod("getRealSize", Point.class).invoke(display, realSize);
                mWidthPixels = realSize.x;
                mHeightPixels = realSize.y;
            }
            catch (Exception ignored)
            {
            }
        }

        DisplayMetrics dm = new DisplayMetrics();
        windowManager.getDefaultDisplay().getMetrics(dm);
        double x = Math.pow(mWidthPixels / dm.xdpi, 2);
        double y = Math.pow(mHeightPixels / dm.ydpi, 2);
        double screenInches = Math.sqrt(x + y);

        return (double) Math.round(screenInches * 10) / 10;
    }


    /**
     * Fix url link for image in case it arrived with errors
     *
     * @param url link to fix
     * @return fixed link
     */
    public static String fixUrlLink(String url)
    {
        String result = url;
        if (result != null && (result.contains("http") || result.contains("https")))
        {
            result = result.trim();
            if (result.contains(" "))
            {
                result = result.replace(" ", "%20");
            }
            if (result.contains("+"))
            {
                result = result.replace("+", "_");
            }
        }
        return result;
    }

    /**
     * Check if device has google account
     */
   /* public static boolean isDeviceHasGoogleAccount(Context context)
    {
        AccountManager accMan = AccountManager.get(context);
        Account[] accArray = accMan.getAccountsByType("com.google");

        boolean has = accArray.length >= 1 ? true : false;
        return has;
    }*/


    /**
     * Check if certain manifest permission exists
     *
     * @param context
     * @param permissison manifest permission to check (Use Manifest.permission)
     */
    public static boolean isManifetsPermissionExists(Context context, String permissison)
    {
        int result = context.checkCallingOrSelfPermission(permissison);
        return result == PackageManager.PERMISSION_GRANTED;
    }

    /**
     * Convenient method for seating global layout listener and getting notified when layout is finished (To get the size of the view)
     *
     * @param view                   the view to layout
     * @param layoutFinishedListener on finished listener
     */
    @SuppressLint("NewApi") public static void setOnGlobalLayoutFinishListener(final View view, final OnGlobalLayoutFinishedListener layoutFinishedListener)
    {
        view.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener()
        {
            @SuppressWarnings("deprecation") @Override public void onGlobalLayout()
            {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
                {
                    view.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                }
                else
                {
                    view.getViewTreeObserver().removeGlobalOnLayoutListener(this);
                }

                if (layoutFinishedListener != null)
                {
                    layoutFinishedListener.onLayoutFinished();
                }
            }
        });
    }


    public interface OnGlobalLayoutFinishedListener
    {
        void onLayoutFinished();
    }

    /**
     * Get is device is a tablet or phone
     *
     * @param context
     * @return true or false
     */
    public static boolean getIsDeviceTablet(Context context)
    {
        boolean result = true;
        if ((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) == Configuration.SCREENLAYOUT_SIZE_NORMAL)
        {
            result = false;
        }
        return result;
    }


    /*public static int getTextSizeBySceenDensity(Context context) {
        int size = 18;
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        DisplayMetrics metrics = new DisplayMetrics();
        windowManager.getDefaultDisplay().getMetrics(metrics);

        switch (metrics.densityDpi)
        {
            case DisplayMetrics.DENSITY_LOW:
            {
                size = 12;
                break;
            }
            case DisplayMetrics.DENSITY_MEDIUM:
            {
                size = 14;
                break;
            }
            case DisplayMetrics.DENSITY_HIGH:
            {
                size = 15;
                break;
            }
            case DisplayMetrics.DENSITY_XHIGH:
            {
                size = 17;
                break;
            }
            case DisplayMetrics.DENSITY_XXHIGH:
            {
                size = 18;
                break;
            }
            case DisplayMetrics.DENSITY_XXXHIGH:
            {
                size = 20;
                break;
            }
        }
        return  size;
    }*/


    /**
     * Read string from a text file in assets folder
     *
     * @param context
     * @param path    text file path in the assets folder
     */
    public static String readStringFromAssets(Context context, String path)
    {
        String res = null;
        try
        {
            StringBuilder buf = new StringBuilder();

            InputStream json = context.getAssets().open(path);
            BufferedReader in = new BufferedReader(new InputStreamReader(json, "UTF-8"));
            String str;

            while ((str = in.readLine()) != null)
            {
                buf.append(str);
            }

            in.close();
            res = buf.toString();

        }
        catch (IOException e)
        {
            e.printStackTrace();
        }

        return res;
    }


    public static String generateUniqeDeviceID(String packageName,String publisherId)
    {
        String scrumbler = "KidozSDK";

        String m_szDevIDShort = "35" +
                (Build.BOARD.length() % 10) +
                (Build.BRAND.length() % 10) +
                (Build.CPU_ABI.length() % 10) +
                (Build.DEVICE.length() % 10) +
                (Build.MANUFACTURER.length() % 10) +
                (Build.MODEL.length() % 10) +
                (Build.PRODUCT.length() % 10);

        if (Build.SERIAL != null)
        {
            scrumbler = scrumbler + Build.SERIAL;
        }

        String preliminaryId = new UUID(m_szDevIDShort.hashCode(), scrumbler.hashCode()).toString();
        return EncoderUtils.SHA1(preliminaryId + packageName + publisherId);
    }


    /**
     * Get bottom navigation bar height
     *
     * @param context
     *            context
     *
     * @return height
     */
    public static int getNavigationBarHeight(Context context)
    {
        int height = 0;
        try {
            Resources resources = context.getResources();
            int orientation = resources.getConfiguration().orientation;
            int id = resources.getIdentifier(orientation == Configuration.ORIENTATION_PORTRAIT ? "navigation_bar_height" : "navigation_bar_height_landscape", "dimen", "android");
            if (id > 0)
            {
                height = resources.getDimensionPixelSize(id);
            }
        } catch (Resources.NotFoundException e) {}
        return height;
    }


    public static String getDeviceReferral(Context context)
    {
        String publisherID = SharedPreferencesUtils.loadSharedPreferencesData(context, "PUBLISHER_ID");
        String result = "SDK_";
        if (context != null)
        {
            result += context.getPackageName();
            result += "_";
            if (publisherID != null)
            {
                result += publisherID;
            }
        }
        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "Device Referral>> " + result);
        return result;
    }


    /**
     * Get screen size
     *
     * @return file path
     */
    @SuppressLint("NewApi") public static int getScreenSize(Context context, boolean isWidth)
    {
        int result = 0;
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = windowManager.getDefaultDisplay();
        if (isWidth)
        {
            if (Build.VERSION.SDK_INT >= 19)
            {
                Point point = new Point();
                display.getRealSize(point);
                result = point.x;
            }
            else
            {
                result = display.getWidth();
            }
        }
        else
        {
            if (Build.VERSION.SDK_INT >= 19)
            {
                Point point = new Point();
                display.getRealSize(point);
                result = point.y;
            }
            else
            {
                result = display.getHeight();
            }
        }
        return result;
    }

    /**
     * Check if device is a tablet
     *
     * @return context
     */
    public static boolean isTablet(Context context) {
        boolean xlarge = ((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) == Configuration.SCREENLAYOUT_SIZE_XLARGE);
        boolean large = ((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) == Configuration.SCREENLAYOUT_SIZE_LARGE);

        boolean isTable =  (xlarge || large);

        if(isTable ||  getDeviceScreenSizeInInches(context) >= 7.0) {
            return true;
        }else {
            return false;
        }
    }




}
