/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kidoz.sdk.api.general.custom_views.CustomCardView;

import android.content.Context;
import android.view.View;

class KidozCardViewApi21 implements KidozCardViewImpl {

    @Override
    public void initialize(KidozCardViewDelegate cardView, Context context, int backgroundColor,
            float radius, float elevation, float maxElevation) {
        final KidozRoundRectDrawable backgroundDrawable = new KidozRoundRectDrawable(backgroundColor, radius);
        cardView.setBackgroundDrawable(backgroundDrawable);
        View view = (View) cardView;
        view.setClipToOutline(true);
        view.setElevation(elevation);
        setMaxElevation(cardView, maxElevation);
    }

    @Override
    public void setRadius(KidozCardViewDelegate cardView, float radius) {
        ((KidozRoundRectDrawable) (cardView.getBackground())).setRadius(radius);
    }

    @Override
    public void initStatic() {
    }

    @Override
    public void setMaxElevation(KidozCardViewDelegate cardView, float maxElevation) {
        ((KidozRoundRectDrawable) (cardView.getBackground())).setPadding(maxElevation,
                cardView.getUseCompatPadding(), cardView.getPreventCornerOverlap());
        updatePadding(cardView);
    }

    @Override
    public float getMaxElevation(KidozCardViewDelegate cardView) {
        return ((KidozRoundRectDrawable) (cardView.getBackground())).getPadding();
    }

    @Override
    public float getMinWidth(KidozCardViewDelegate cardView) {
        return getRadius(cardView) * 2;
    }

    @Override
    public float getMinHeight(KidozCardViewDelegate cardView) {
        return getRadius(cardView) * 2;
    }

    @Override
    public float getRadius(KidozCardViewDelegate cardView) {
        return ((KidozRoundRectDrawable) (cardView.getBackground())).getRadius();
    }

    @Override
    public void setElevation(KidozCardViewDelegate cardView, float elevation) {
        ((View) cardView).setElevation(elevation);
    }

    @Override
    public float getElevation(KidozCardViewDelegate cardView) {
        return ((View) cardView).getElevation();
    }

    @Override
    public void updatePadding(KidozCardViewDelegate cardView) {
        if (!cardView.getUseCompatPadding()) {
            cardView.setShadowPadding(0, 0, 0, 0);
            return;
        }
        float elevation = getMaxElevation(cardView);
        final float radius = getRadius(cardView);
        int hPadding = (int) Math.ceil(KidozRoundRectDrawableWithShadow
                .calculateHorizontalPadding(elevation, radius, cardView.getPreventCornerOverlap()));
        int vPadding = (int) Math.ceil(KidozRoundRectDrawableWithShadow
                .calculateVerticalPadding(elevation, radius, cardView.getPreventCornerOverlap()));
        cardView.setShadowPadding(hPadding, vPadding, hPadding, vPadding);
    }

    @Override
    public void onCompatPaddingChanged(KidozCardViewDelegate cardView) {
        setMaxElevation(cardView, getMaxElevation(cardView));
    }

    @Override
    public void onPreventCornerOverlapChanged(KidozCardViewDelegate cardView) {
        setMaxElevation(cardView, getMaxElevation(cardView));
    }

    @Override
    public void setBackgroundColor(KidozCardViewDelegate cardView, int color) {
        ((KidozRoundRectDrawable) (cardView.getBackground())).setColor(color);
    }
}