package com.kidoz.sdk.api;


import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;

import com.kidoz.events.DeviceUtils;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.assets_handling.ImageAssetsUtils;
import com.kidoz.sdk.api.general.assets_handling.SoundAssetsUtils;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.SdkSoundManager;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.players.video_player.VideoManager;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;

import org.json.JSONObject;

public final class KidozSDK
{
    private static final String TAG = KidozSDK.class.getSimpleName();

    private static boolean isInitialized = false;
    private static String mPublisherKey = null;
    private static String mSecurityKey = null;

    private static boolean isDeveloperLoggingON = false;

    /**
     * Initialize Kidoz SDK
     *
     * @param context      activity context
     * @param publicher_id valid publisher id
     * @param token        valid security token
     * @throws RuntimeException in case of invalid or missing publisher_id or security token
     */
    public static void initialize(final Context context, String publicher_id, String token)
    {
        try
        {

            if (context instanceof Activity)
            {
                ((Activity) context).runOnUiThread(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        VideoManager.getInstance(context);
                    }
                });
            }
            else
            {
                VideoManager.getInstance(context);
            }

            DeviceUtils.getGoogleAdvertisingID(context);

            SharedPreferences eventsPref = context.getSharedPreferences(EventManager.EVENTS_SHARED_PREFERENCES_KEY, 0);
            long sessionID = eventsPref.getLong(EventManager.EVENTS_LAST_SESSION_ID_KEY, 0);
            sessionID += 1;
            SharedPreferences.Editor editor = eventsPref.edit();
            editor.putLong(EventManager.EVENTS_LAST_SESSION_ID_KEY, sessionID);
            editor.apply();

            SharedPreferencesUtils.saveSharedPreferencesData(context, "PUBLISHER_ID", publicher_id);

            JSONObject eventParams = new JSONObject();
            eventParams.put(EventParameters.DEVELOPER_ID, publicher_id);
            eventParams.put(EventManager.LOG_LEVEL_KEY, EventManager.LOG_NORMAL_LEVEL);
            eventParams.put(EventParameters.SESSION_ID, sessionID);

            EventManager.getInstance(context).init(context, eventParams);

            EventManager.getInstance(context).logEvent(context, WidgetType.WIDGET_TYPE_NONE, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_SDK_INIT, publicher_id, false);
            EventManager.getInstance(context).logEvent(context, WidgetType.WIDGET_TYPE_NONE, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SESSION, EventParameters.ACTION_SESSION_START, publicher_id, false);

            if (isInitialized)
            {
                SDKLogger.printWarningLog("@@@@@@  ALREADY INITIALIZED ");
                EventManager.getInstance(context).logEvent(context, WidgetType.WIDGET_TYPE_NONE, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_SDK_ALREADY_INIT, publicher_id, false);
            }
            else
            {
                SDKLogger.printWarningLog("@@@@@@@ INIT SDK");
            }

            EventManager.getInstance(context).startEventsSync(context, 2000);
        }
        catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error initiating " + ex.getMessage());
        }

        if (publicher_id == null || token == null)
        {
            throw new RuntimeException("Missing valid publisher id or security token!");
        }
        else
        {
            mPublisherKey = publicher_id;
            mSecurityKey = token;
            SdkAPIManager.init(context, publicher_id, token, isDeveloperLoggingON);
            SdkSoundManager.initSoundPool(context);

            isInitialized = true;
            validateSDKConfiguration(context);
        }

        if (isDeveloperLoggingON)
        {
            Log.d(SDKLogger.GENERAL_TAG, "Kidoz SDK has been successfully Initialized !");
        }
    }

    private static SdkAPIManager getApiManager(Context context)
    {
        SdkAPIManager.init(context, mPublisherKey, mSecurityKey, isDeveloperLoggingON);
        return SdkAPIManager.getSdkApiInstance(context);
    }

    /**
     * Check if Kidoz SDK has been initialised
     *
     * @return initialized
     */
    public static boolean isInitialised()
    {
        if (isInitialized == false)
        {
            Log.e("[KIDOZ SDK]", "KIDOZ SDK must to be initialized !");
        }
        return isInitialized;
    }

    /**
     * Validate SDK configuration (Configuration retrieved from remote server)
     */
    private static void validateSDKConfiguration(final Context context)
    {
        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>validateSDKConfiguration");

        if (context != null)
        {
            PropertiesObj appConfig = DatabaseManager.getInstance(context).getConfigTable().loadAppProperties();
            if (appConfig != null)
            {
                // Update Current Sdk style version for Event Log
                EventManager.SDK_STYLE_VERSION_NUM = appConfig.getSdkStyleVersion();
            }

            getApiManager(context).validateSDKConfiguration(context, new ApiResultCallback<PropertiesObj>()
            {
                @Override
                public void onServerResult(ResultData<?> result)
                {
                    if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true && result.getData() != null && result.getData() instanceof PropertiesObj)
                    {
                        final PropertiesObj serverConfig = (PropertiesObj) result.getData();

                        PropertiesObj appConfig = DatabaseManager.getInstance(context).getConfigTable().loadAppProperties();

                        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>Server config version = " + serverConfig.getConfigVersion());
                        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>App config version = " + (appConfig == null ? "NULL" : appConfig.getConfigVersion()));

                        // We save the config before loading missing graphics
                        if (appConfig == null)
                        {
                            PropertiesObj prop = new PropertiesObj(serverConfig.getData());
                            prop.setConfigVersion(-1); // The value set to -1 to avoid updating the state of version before the graphics is updated/loaded
                            DatabaseManager.getInstance(context).getConfigTable().insertAppProperties(prop);
                        }
                        else
                        {
                            appConfig.updateConfigFromData(serverConfig);
                            DatabaseManager.getInstance(context).getConfigTable().insertAppProperties(appConfig);
                        }

                        // Update Current Sdk style version for Event Log
                        EventManager.SDK_STYLE_VERSION_NUM = serverConfig.getSdkStyleVersion();

                        if (appConfig == null || appConfig.getConfigVersion() < serverConfig.getConfigVersion())
                        {
                            com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>Going to download resources...");
                            // Need to update configuration (graphics, strings etc...)
                            getApiManager(context).getSDKResources(context, new ApiResultCallback<Boolean>()
                            {
                                @Override
                                public void onServerResult(ResultData<?> result)
                                {
                                    if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true && result.getData() != null && result.getData() instanceof Boolean)
                                    {
                                        if ((Boolean) result.getData() == true)
                                        {
                                            com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>All resources downloaded");
                                            SharedPreferencesUtils.removeSharedPreferencesData(context, ImageAssetsUtils.IMAGE_ASSETS_SYNCED_FLAG);
                                            SharedPreferencesUtils.removeSharedPreferencesData(context, ImageAssetsUtils.PARENTAL_LOCK_ASSETS_SYNCED_FLAG);
                                            SharedPreferencesUtils.removeSharedPreferencesData(context, SoundAssetsUtils.SOUND_ASSETS_SYNCED_FLAG);

                                            DatabaseManager.getInstance(context).getConfigTable().insertAppProperties(serverConfig);
                                        }
                                    }
                                }

                                @Override
                                public void onFailed()
                                {

                                }
                            });
                        }
                    }
                }

                @Override
                public void onFailed()
                {

                }
            });
        }
    }


    /**
     * Set developer logging enabled
     * (Must be set before initialization of the SDK)
     *
     * @param enabled
     */
    public static void setLoggingEnabled(boolean enabled)
    {
        isDeveloperLoggingON = enabled;
    }
}
