package com.kidoz.sdk.api;

import android.content.Context;
import android.graphics.Point;
import android.os.Looper;
import android.os.Message;
import android.util.AttributeSet;
import android.view.View;
import android.widget.RelativeLayout;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.ui_views.custom_drawables.KidozLogoDrawable;
import com.kidoz.sdk.api.ui_views.parental_lock.StyleableButtonView;

public class FeedButton extends RelativeLayout
{
    private static final String TAG = FeedButton.class.getSimpleName();


    private static final int BUTTON_STYLE_RESOURCE_NOT_READY_REPEAT_COUNT = 7;
    private static final float BUTTON_SIZE_AS_SCREEN_PORTION = 0.25f;
    private StyleableButtonView mStyleableButtonView;
    private ExtendedFeedView mFeedView;

    private View mRootView;
    private KidozLogoDrawable kidozLogoDrawable;
    private int mNotReadyRepeatCount = 0;

    private Utils.StaticHandler mUiLoaderStaticHandler;

    public FeedButton(Context context)
    {
        super(context);
        initView(context, -1);
    }

    /**
     * Initiate button view
     *
     * @param size button size
     */
    public FeedButton(Context context, int size)
    {
        super(context);
        initView(context, size);
    }

    /**
     * Initiate button view
     */
    public FeedButton(Context context, AttributeSet attrs)
    {
        super(context, attrs);
        initView(context, -1);
    }

    public FeedButton(Context context, AttributeSet attrs, int defStyleAttr)
    {
        super(context, attrs, defStyleAttr);
        initView(context, -1);
    }

    /**
     * Initiate view
     */
    private void initView(Context context, int initSize)
    {
        if(KidozSDK.isInitialised()) {
            mRootView = this;

            int size = getDefaultButtonSize(context);

            if (initSize != -1)
            {
                size = initSize;
            }

            mStyleableButtonView = new StyleableButtonView(context);

            RelativeLayout.LayoutParams animatableImageParams = new RelativeLayout.LayoutParams(size, size);
            addView(mStyleableButtonView, animatableImageParams);

            RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(size, size);
            setLayoutParams(params);


            if (isInEditMode() == false)
            {
                initFeedView(context);
            }

            kidozLogoDrawable = new KidozLogoDrawable();
            mNotReadyRepeatCount = 0;
            loadButtonStyleView(size);
        }
    }

    /**
     * Load feed button style view define on server
     */
    private void loadButtonStyleView(final int size)
    {
        PropertiesObj appConfig = DatabaseManager.getInstance(getContext()).getConfigTable().loadAppProperties();
        if (appConfig != null)
        {
            if (mStyleableButtonView != null)
            {
                mStyleableButtonView.loadButtonStyle(size, appConfig.getFeedButtonUrl(), kidozLogoDrawable, "fdBtn");
            }
        }
        else
        {
            if (mNotReadyRepeatCount < BUTTON_STYLE_RESOURCE_NOT_READY_REPEAT_COUNT)
            {
                getUiHandlerInstance().postDelayed(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        mNotReadyRepeatCount++;
                        loadButtonStyleView(size);
                    }
                }, 1000);
            }
        }
    }

    /**
     * Initiate button click listener
     */
    private void initClickListener()
    {
        setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mFeedView != null) {
                    GenAnimator.clickItemAnimation(mRootView, 100, new GenAnimator.ViewAnimationListener() {
                        @Override
                        public void onAnimationEnd() {
                            mFeedView.showView();
                        }

                        @Override
                        public void onAnimationStart() {

                        }
                    });
                }
            }
        });
    }

    /**
     * Initiate interstitial view
     */
    private void initFeedView(Context context)
    {
        mFeedView = new ExtendedFeedView(new FeedView.Builder(context));
        initClickListener();
    }

    /**
     * Get the inner interstitial feed view object
     *
     * @return interstitial view object
     */
    public FeedView getFeedView()
    {
        if (mFeedView == null)
        {
            initFeedView(getContext());
        }

        return mFeedView;
    }

    /**
     * Set button size in pixels
     *
     * @param sizePx size in pixels
     */
    public void setFeedButtonSize(int sizePx)
    {
        if(mStyleableButtonView != null) {
            mStyleableButtonView.getLayoutParams().height = sizePx;
            mStyleableButtonView.getLayoutParams().width = sizePx;
        }

        getLayoutParams().height = sizePx;
        getLayoutParams().width = sizePx;
    }

    /**
     * Set button size in DP
     *
     * @param sizeDp size in pixels
     */
    public void setFeedButtonSizeDp(int sizeDp)
    {

        int h = Utils.dpTOpx(getResources(), sizeDp);
        int w = Utils.dpTOpx(getResources(), sizeDp);

        if(mStyleableButtonView != null) {
            mStyleableButtonView.getLayoutParams().height = h;
            mStyleableButtonView.getLayoutParams().width = w;
        }

        getLayoutParams().height = h;
        getLayoutParams().width = w;
    }

    /**
     * Get default button size
     *
     * @return button size in pixels
     */
    public static int getDefaultButtonSize(Context context)
    {
        Point point = Utils.getScreenSize(context);
        return (int) (Math.min(point.x, point.y) * BUTTON_SIZE_AS_SCREEN_PORTION);
    }


    @Override
    protected void onAttachedToWindow()
    {
        super.onAttachedToWindow();

        if (isInEditMode() == false && KidozSDK.isInitialised())
        {
            EventManager.getInstance(getContext()).logEvent(getContext(), WidgetType.WIDGET_TYPE_NONE, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_BUTTON_VIEW, EventParameters.LABEL_FEED_BUTTON_VIEW, false);
            EventManager.getInstance(getContext()).startEventsSync(getContext(), 1500);
        }
    }


    /**
     * Get handler instance
     */
    private Utils.StaticHandler getUiHandlerInstance()
    {
        if (mUiLoaderStaticHandler == null)
        {
            mUiLoaderStaticHandler = new Utils.StaticHandler(Looper.getMainLooper())
            {
                @Override
                public void handleMessage(Message message)
                {
                    // This is where you do your work in the UI thread.
                    // Your worker tells you in the message what to do.
                }
            };
        }
        return mUiLoaderStaticHandler;
    }


    class ExtendedFeedView extends FeedView {
        public ExtendedFeedView(Builder builder) {
            super(builder);
        }

        @Override
        public void showView() {
            //super.showView();
            mIsAutomaticInvocation = false;
            mIsPostLoadData = true;

            if(KidozSDK.isInitialised()) {
                if (mOnInterstitialViewEventListener != null) {
                    mOnInterstitialViewEventListener.onReadyToShow();
                }

                openSimpleDialog();
            }
        }
    }
}
