package com.kidoz.sdk.api.ui_views.one_item_view;


import android.content.Context;
import android.graphics.Point;
import android.support.v4.view.PagerAdapter;
import android.support.v4.view.ViewPager;
import android.view.View;

import com.kidoz.sdk.api.general.utils.ScreenUtils;


/**
 * A ViewPager that allows pseudo-infinite paging with a wrap-around effect. Should be used with an
 * InfinitePagerAdapter.
 */
public class InfiniteViewPager extends ViewPager
{
    private final String TAG = InfiniteViewPager.class.getSimpleName();
    // Start of multi page view.
    private int mMaxWidth = -1;
    private int mMaxHeight = -1;
    private int mMatchWidthChildResId;
    private boolean mNeedsMeasurePage;
    private final Point size;
    private final Point maxSize;

    private static void constrainTo(Point size, Point maxSize)
    {
        if (maxSize.x >= 0)
        {
            if (size.x > maxSize.x)
            {
                size.x = maxSize.x;
            }
        }
        if (maxSize.y >= 0)
        {
            if (size.y > maxSize.y)
            {
                size.y = maxSize.y;
            }
        }
    }

    @Override protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec)
    {
        size.set(MeasureSpec.getSize(widthMeasureSpec), MeasureSpec.getSize(heightMeasureSpec));
        if (mMaxWidth >= 0 || mMaxHeight >= 0)
        {
            maxSize.set(mMaxWidth, mMaxHeight);
            constrainTo(size, maxSize);
            widthMeasureSpec = MeasureSpec.makeMeasureSpec(size.x, MeasureSpec.EXACTLY);
            heightMeasureSpec = MeasureSpec.makeMeasureSpec(size.y, MeasureSpec.EXACTLY);
        }
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        onMeasurePage(widthMeasureSpec, heightMeasureSpec);
    }

    protected void onMeasurePage(int widthMeasureSpec, int heightMeasureSpec)
    {
        // Only measure if a measurement pass was scheduled
        if (!mNeedsMeasurePage)
        {
            return;
        }
        if (mMatchWidthChildResId == 0)
        {
            mNeedsMeasurePage = false;
        }
        else if (getChildCount() > 0)
        {
            View child = getChildAt(0);
            child.measure(widthMeasureSpec, heightMeasureSpec);
            int pageWidth = child.getMeasuredWidth();
            View match = child.findViewById(mMatchWidthChildResId);
            if (match == null)
            {
                throw new NullPointerException("MatchWithChildResId did not find that ID in the first fragment of the ViewPager; " + "is that view defined in the child view's layout? Note that MultiViewPager " + "only measures the child for index 0.");
            }
            int childWidth = match.getMeasuredWidth();
            // Check that the measurement was successful
            if (childWidth > 0)
            {
                mNeedsMeasurePage = false;
                //                int difference = pageWidth - childWidth;
                //                setPageMargin(-difference);
                int pageMargin = (int) (ScreenUtils.getScreenSize(getContext(), true) * 0.15);
                setPageMargin(-pageMargin);
                int offscreen = (int) Math.ceil((float) pageWidth / (float) childWidth) + 1;
                setOffscreenPageLimit(offscreen);
                requestLayout();
            }
        }
    }

    @Override protected void onSizeChanged(int w, int h, int oldw, int oldh)
    {
        super.onSizeChanged(w, h, oldw, oldh);
        // Schedule a new measurement pass as the dimensions have changed
        mNeedsMeasurePage = true;
    }

    /**
     * Sets the child view inside a page to match the page size against.
     *
     * @param matchChildWidthResId the child id
     */
    public void setMatchChildWidth(int matchChildWidthResId)
    {
        if (mMatchWidthChildResId != matchChildWidthResId)
        {
            mMatchWidthChildResId = matchChildWidthResId;
            mNeedsMeasurePage = true;
        }
    }

    /**
     * Sets the maximum size.
     *
     * @param width in pixels
     */
    public void setMaxWidth(int width)
    {
        mMaxWidth = width;
    }

    /**
     * Sets the maximum size.
     *
     * @param height in pixels
     */
    public void setMaxHeight(int height)
    {
        mMaxHeight = height;
    }
    // End of multi page view.

    public InfiniteViewPager(Context context)
    {
        super(context);
        size = new Point();
        maxSize = new Point();
    }

    @Override public void setAdapter(PagerAdapter adapter)
    {
        super.setAdapter(adapter);
        // offset first element so that we can scroll to the left
        setCurrentItem(0);
    }

    @Override public void setCurrentItem(int item)
    {
        // offset the current item to ensure there is space to scroll
        setCurrentItem(item, false);
    }

    @Override public void setCurrentItem(int item, boolean smoothScroll)
    {
        if (getAdapter().getCount() == 0)
        {
            super.setCurrentItem(item, smoothScroll);
            return;
        }
        item = getOffsetAmount() + (item % getAdapter().getCount());
        super.setCurrentItem(item, smoothScroll);
    }

    @Override public int getCurrentItem()
    {
        if (getAdapter().getCount() == 0)
        {
            return super.getCurrentItem();
        }
        int position = super.getCurrentItem();
        if (getAdapter() instanceof InfinitePagerAdapter)
        {
            InfinitePagerAdapter infAdapter = (InfinitePagerAdapter) getAdapter();
            // Return the actual item position in the data backing InfinitePagerAdapter
            return (position % infAdapter.getRealCount());
        }
        else
        {
            return super.getCurrentItem();
        }
    }

    private int getOffsetAmount()
    {
        if (getAdapter().getCount() == 0)
        {
            return 0;
        }
        if (getAdapter() instanceof InfinitePagerAdapter)
        {
            InfinitePagerAdapter infAdapter = (InfinitePagerAdapter) getAdapter();
            // allow for 100 back cycles from the beginning
            // should be enough to create an illusion of infinity
            // warning: scrolling to very high values (1,000,000+) results in
            // strange drawing behaviour
            return infAdapter.getRealCount() * 100;
        }
        else
        {
            return 0;
        }
    }
}
