package com.kidoz.sdk.api.players;

import android.animation.Animator;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.graphics.Color;
import android.graphics.Point;
import android.os.Bundle;
import android.os.Looper;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.animation.AccelerateInterpolator;
import android.view.animation.DecelerateInterpolator;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.dialogs.WebDialog.VideoEnabledWebChromeClient;
import com.kidoz.sdk.api.dialogs.WebDialog.VideoEnabledWebView;
import com.kidoz.sdk.api.general.BaseDialog;
import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.enums.ContentType;
import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.players.video_player.VideoJavaScriptInterface;
import com.kidoz.sdk.api.players.video_player.VideoManager;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.KidozTopBar;
import com.kidoz.sdk.api.ui_views.loading_progress_view.LoadingProgressView;

/**
 * Created by roni on 10/13/2015.
 */
public class FullScreenViewDialog extends BaseDialog {

    private static final String ORIENTATION_STATE_SAVE_STATE_KEY = "ORIENTATION_STATE_SAVE_STATE_KEY_FS";
    private static final String ORIENTATION_MODIFY_SAVE_STATE_KEY = "ORIENTATION_MODIFY_SAVE_STATE_KEY_FS";
    private static final String CONTENT_ITEM_SAVE_STATE_KEY = "CONTENT_ITEM_SAVE_STATE_KEY_FS";
    private static final String TOP_BAR_HIDDEN_SAVE_STATE_KEY = "TOP_BAR_HIDDEN_SAVE_STATE_KEY_FS";

    private final String TAG = FullScreenViewDialog.class.getSimpleName();
    private final int BLACK_CURTAIN_HIDE_DELAY = 50;
    private final int TOP_BAR_ANIMATION_DURATION = 500;
    private final int TOP_BAR_ANIMATION_DELAY = 3000;

    private FrameLayout mContentView;
    private FrameLayout mCustomViewContainer;
    private KidozTopBar mKidozTopBar;
    private LoadingProgressView mLoadingProgressView;
    private View mBlackCurtain;

    // WebView related
    private VideoEnabledWebChromeClient mVideoEnabledWebChromeClient;

    // Video related
    private VideoEnabledWebView mVideoEnabledWebView;
    private VideoJavaScriptInterface mVideoJavaScriptInterface;
    private RelativeLayout mVideoLayer;
    private View mTransparentView;
    private ContentItem mContentItem;

    // Animation related
    private Utils.StaticHandler mStaticHandler;
    private boolean mIsTopBarHidden;
    private boolean mIsAnimationRunning;

    // Log events related
    private long mEventID;

    private boolean mAllowOrientationModify = false;
    private WidgetType mSourceWidgetType = WidgetType.WIDGET_TYPE_NONE;

    public FullScreenViewDialog(Context context, WidgetType sourceWidgetType, boolean allowModifyOrientation) {
        super(context, android.R.style.Theme_Black_NoTitleBar_Fullscreen, allowModifyOrientation);
        mAllowOrientationModify = allowModifyOrientation;
        mSourceWidgetType = sourceWidgetType;
        setCancelable(true);

        getWindow().setFlags(
                WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED,
                WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED);

        setOnDismissListener(new OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                mContentView.removeAllViews();
            }
        });
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        if (mAllowOrientationModify) {
            if (savedInstanceState != null && savedInstanceState.containsKey(ORIENTATION_STATE_SAVE_STATE_KEY)) {
                mCurrentOrientationState = savedInstanceState.getInt(ORIENTATION_STATE_SAVE_STATE_KEY);
            } else {
                if (mCallingActivity != null && mCallingActivity.isFinishing() == false) {
                    mCurrentOrientationState = mCallingActivity.getResources().getConfiguration().orientation;
                    mCallingActivity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR);
                }
            }
        }

        if (savedInstanceState != null) {
            if (savedInstanceState.containsKey(CONTENT_ITEM_SAVE_STATE_KEY)) {
                mContentItem = (ContentItem) savedInstanceState.getSerializable(CONTENT_ITEM_SAVE_STATE_KEY);
            }
            mIsTopBarHidden = savedInstanceState.getBoolean(TOP_BAR_HIDDEN_SAVE_STATE_KEY);
        }

        setOnDismissListener(new OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {

                mContentView.removeView(mVideoEnabledWebView);
                mStaticHandler.removeCallbacksAndMessages(null);
            }
        });

        initDialog();
        setContentView(mCustomViewContainer);
    }

    @Override
    public void onDetachedFromWindow() {
        super.onDetachedFromWindow();

        // Restore previous orientation
        if (mAllowOrientationModify) {
            restoreUserOriginalOrientation();
        }

        EventManager.getInstance(getContext()).activityEnd(getContext(), mSourceWidgetType, EventManager.LOG_CRITICAL_LEVEL, mEventID, false);
        EventManager.getInstance(getContext()).logEventWithDurationEnd(getContext(), mSourceWidgetType, 1, false);
    }

    @Override
    public Bundle onSaveInstanceState() {
        Bundle bundle = super.onSaveInstanceState();
        if (bundle != null) {
            bundle.putSerializable(ORIENTATION_STATE_SAVE_STATE_KEY, mCurrentOrientationState);
            bundle.putSerializable(CONTENT_ITEM_SAVE_STATE_KEY, mContentItem);
            bundle.putBoolean(ORIENTATION_MODIFY_SAVE_STATE_KEY, mAllowOrientationModify);
            bundle.putBoolean(TOP_BAR_HIDDEN_SAVE_STATE_KEY, mIsTopBarHidden);
        }
        return bundle;
    }

    private void initDialog() {
        mStaticHandler = new Utils.StaticHandler(Looper.getMainLooper());
        initRootView();
        initVideoWebView();
        initBlackCurtain();
        initTopBar();
        initLoadingView();
    }

    private void initBlackCurtain() {
        mBlackCurtain = new View(getContext());
        mBlackCurtain.setBackgroundColor(Color.BLACK);
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT);
        mCustomViewContainer.addView(mBlackCurtain, params);
    }

    private void initLoadingView() {
        mLoadingProgressView = new LoadingProgressView(getContext());
        mLoadingProgressView.setCircleColor(Color.parseColor("#ffffff"));
        mLoadingProgressView.stopLoadingAnimation();

        Point point = Utils.getScreenSize(getContext());
        int size = (int) (Math.min(point.x, point.y) * 0.25);
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(size, size);
        params.gravity = Gravity.CENTER;

        mCustomViewContainer.addView(mLoadingProgressView, params);
    }

    private void initRootView() {
        mCustomViewContainer = new FrameLayout(getContext());
        mCustomViewContainer.setId(Utils.generateViewId());
        mCustomViewContainer.setLayoutParams(new ViewGroup.MarginLayoutParams(ViewGroup.MarginLayoutParams.MATCH_PARENT, ViewGroup.MarginLayoutParams.MATCH_PARENT));

        mContentView = new FrameLayout(getContext());
        mContentView.setId(Utils.generateViewId());
        mContentView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT));
        mCustomViewContainer.addView(mContentView);
    }

    private void initTopBar() {
        mKidozTopBar = new KidozTopBar(getContext(), true);
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.WRAP_CONTENT);
        mKidozTopBar.setKidozTopBarListener(new KidozTopBar.KidozTopBarListener() {
            @Override
            public void onExitClicked(View view) {
                closeDialog();
            }
        });
        mKidozTopBar.setTitle("Video Player");
        mKidozTopBar.showParentalLockIcon(false);
        mCustomViewContainer.addView(mKidozTopBar, params);
    }


    private void initVideoWebView() {
        mVideoLayer = new RelativeLayout(getContext());
        mVideoLayer.setId(Utils.generateViewId());
        mVideoLayer.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT));
        mCustomViewContainer.addView(mVideoLayer);

        mVideoEnabledWebView = VideoManager.getInstance(getContext()).getIFrameVideo();
        if (getContext().getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            mVideoEnabledWebView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT, Gravity.CENTER));
        } else {
            mVideoEnabledWebView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT, Gravity.CENTER));
        }

        mVideoJavaScriptInterface = VideoManager.getInstance(getContext()).getVideoJavaScriptInterface();
        mVideoJavaScriptInterface.setJavascriptListener(new com.kidoz.sdk.api.players.video_player.VideoJavaScriptInterface.JavascriptListener() {
            @Override
            public void onVideoReady() {

            }

            @Override
            public void onError(final int errorCode) {
                mStaticHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        mLoadingProgressView.stopLoadingAnimation();
                        SDKLogger.printDebbugLog(TAG, ">>>>Error loading video, errorCode = " + String.valueOf(errorCode));
                    }
                });
            }

            @Override
            public void onStateChange(int state) {
                SDKLogger.printDebbugLog(TAG, ">>>State = " + state);
                switch (state) {
                    case 1: {
                        //Video is playing
                        mStaticHandler.postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                mLoadingProgressView.stopLoadingAnimation();
                                mBlackCurtain.setVisibility(View.GONE);
                            }
                        }, BLACK_CURTAIN_HIDE_DELAY);
                        break;
                    }
                    case 2: {
                        break;
                    }
                    default: {
                        //Video is not playing
                        mStaticHandler.post(new Runnable() {
                            @Override
                            public void run() {
                                mBlackCurtain.setVisibility(View.VISIBLE);
                            }
                        });
                    }
                }
            }
        });

        mVideoEnabledWebView.setOnConfigurationChangeListener(new VideoEnabledWebView.ConfigurationChangeListener() {
            @Override
            public void onConfigChange() {

            }
        });

        mContentView.addView(mVideoEnabledWebView);

        mTransparentView = new View(getContext());
        mTransparentView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT));
        mTransparentView.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                if (event.getAction() == MotionEvent.ACTION_DOWN) {
                    animateTopBarEnter();
                }
                return false;
            }
        });
        mCustomViewContainer.addView(mTransparentView);
    }


    @Override
    protected void onStop() {
        super.onStop();
        if (mVideoEnabledWebView != null) {
            mVideoEnabledWebView.pauseVideo();
        }
    }

    public void openDialog(ContentItem contentItem) {
        super.openDialog();
        mContentItem = contentItem;
        loadVideo();
        mKidozTopBar.setTitle(contentItem.getName());

        mEventID = EventManager.getInstance(getContext()).activityStart(getContext(), mSourceWidgetType, EventManager.LOG_CRITICAL_LEVEL, "YouTube Player", false);
        EventManager.getInstance(getContext()).logEventWithDuration(getContext(), mSourceWidgetType, EventManager.LOG_NORMAL_LEVEL, 1, EventManager.LOG_CLOSE_RESOURCE_LEVEL, EventParameters.CATEGORY_VIDEO_PLAYER, EventParameters.ACTION_VIDEO_VIEW, contentItem.getName(), contentItem.getId(), false);
    }

    private void loadVideo() {
        mLoadingProgressView.startLoadingAnimation();
        if (mVideoEnabledWebView != null) {
            String videoType = "0";
            if (mContentItem.getContentType() == ContentType.YOTUBE_VIDEO) {
                videoType = "0";
            } else if (mContentItem.getContentType() == ContentType.STREAMING_VIDEO) {
                videoType = "5";
            }
            SDKLogger.printDebbugLog(TAG, ">>>>loadVideo: Domain = " + mContentItem.getContentType().name() + "\nURL = " + mContentItem.getData() + "\nType = " + videoType);

            //            String videoURL = "http://www.w3schools.com/html/mov_bbb.mp4";
            String videoURL = mContentItem.getData();
            mVideoEnabledWebView.loadVideo(videoType, videoURL);
        }

        mStaticHandler.postDelayed(new Runnable() {
            @Override
            public void run() {
                animateTopBarExit();
            }
        }, TOP_BAR_ANIMATION_DELAY);
    }

    private void animateTopBarEnter() {
        if (mIsAnimationRunning == false && mIsTopBarHidden == true && getContext().getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            mIsAnimationRunning = true;
            mStaticHandler.removeCallbacks(null);
            GenAnimator.showTopBar(mKidozTopBar, mKidozTopBar.getHeight(), TOP_BAR_ANIMATION_DURATION, new DecelerateInterpolator(), new Animator.AnimatorListener() {
                @Override
                public void onAnimationStart(Animator animation) {

                }

                @Override
                public void onAnimationEnd(Animator animation) {
                    mIsAnimationRunning = false;
                    mIsTopBarHidden = false;
                    mStaticHandler.postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            animateTopBarExit();
                        }
                    }, TOP_BAR_ANIMATION_DELAY);
                }

                @Override
                public void onAnimationCancel(Animator animation) {

                }

                @Override
                public void onAnimationRepeat(Animator animation) {

                }
            });
        }
    }

    private void animateTopBarExit() {
        if (mIsAnimationRunning == false && mIsTopBarHidden == false && getContext().getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
            mIsAnimationRunning = true;
            mStaticHandler.removeCallbacks(null);
            GenAnimator.hideTopBar(mKidozTopBar, -mKidozTopBar.getHeight(), TOP_BAR_ANIMATION_DURATION, new AccelerateInterpolator(), new Animator.AnimatorListener() {
                @Override
                public void onAnimationStart(Animator animation) {

                }

                @Override
                public void onAnimationEnd(Animator animation) {
                    mIsAnimationRunning = false;
                    mIsTopBarHidden = true;
                }

                @Override
                public void onAnimationCancel(Animator animation) {

                }

                @Override
                public void onAnimationRepeat(Animator animation) {

                }
            });
        }
    }

    private void resetTopBarAnimation() {
        if (mIsTopBarHidden == true) {
            mStaticHandler.removeCallbacks(null);
            GenAnimator.showTopBar(mKidozTopBar, mKidozTopBar.getHeight(), 0, new DecelerateInterpolator(), new Animator.AnimatorListener() {
                @Override
                public void onAnimationStart(Animator animation) {

                }

                @Override
                public void onAnimationEnd(Animator animation) {
                    mIsTopBarHidden = false;
                }

                @Override
                public void onAnimationCancel(Animator animation) {

                }

                @Override
                public void onAnimationRepeat(Animator animation) {

                }
            });
        }
    }

    @Override
    protected void onDialogOrientationChange(boolean isPortrait) {
        if (isPortrait == true) {
            mVideoEnabledWebView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT, Gravity.CENTER));
            resetTopBarAnimation();
        } else {
            mVideoEnabledWebView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT, Gravity.CENTER));
            mStaticHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    animateTopBarExit();
                }
            }, TOP_BAR_ANIMATION_DELAY);
        }
        mVideoEnabledWebView.invalidate();
    }

    @Override
    protected void onDialogFocusChange(boolean isVisible) {
        if (isVisible == true) {
            mVideoEnabledWebView.resumeVideo();
        } else {
            mVideoEnabledWebView.pauseVideo();
        }
    }


}
