package com.kidoz.events;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.res.Configuration;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.DisplayMetrics;
import android.view.WindowManager;

import com.kidoz.sdk.api.general.enums.WidgetType;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.general.utils.Utils;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

/**
 * Created by orikam on 8/20/15.
 */
public class EventManager {
    private static final String TAG = EventManager.class.getSimpleName();

    public static int LOG_CRITICAL_LEVEL = 0;
    public static int LOG_NORMAL_LEVEL = 1;

    public static int LOG_CLOSE_ANY_EVENT_LEVEL = 0;
    public static int LOG_CLOSE_RESOURCE_LEVEL = 1;
    public static int LOG_CLOSE_ACTIVITY_LEVEL = 2;
    public static int LOG_CLOSE_SESSION_LEVEL = 3;

    private static final int NIGHT_TIME_HOUR_VALUE_START = 21;
    private static final int NIGHT_TIME_HOUR_VALUE_END = 8;
    private static final int DAY_TIME_AMOUNT = 10;
    private static final int NIGHT_TIME_AMOUNT_EXTRA = 50;
    private static final int THREE_G_CONNECTON_TYPE_AMOUNT = 10;
    private static final int WIFI_CONNECTON_TYPE_AMOUNT_EXTRA = 20;
    private static final int SCREEN_STATE_ON_AMOUNT = 10;
    private static final int SCREEN_STATE_OFF_AMOUNT_EXTRA = 20;
    private static final String OS_TYPE = "Android";
    private static final long DEFAULT_DELTA_TIME_BETWEEN_ACTIVITES_FOR_NEW_SESSION = 30 * 1000;//time in msec
    public static final String EVENTS_SHARED_PREFERENCES_KEY = "events_shared_preferences";
    public static final String EVENTS_LAST_SESSION_ID_KEY = "last_session_id";
    private static final String EVENTS_LAST_ACTIVITY_ID_KEY = "last_activity_id";
    private static final String EVENT_SERVER_ADDRESS_KEY = "event_server_address";
    private static final String DELTA_FOR_NEW_SESSION_KEY = "delta_for_new_session_key";
    public static final String LOG_LEVEL_KEY = "log_level";

    // Variables Defenitions
    public static int SDK_STYLE_VERSION_NUM = 0;

    private static EventManager mEventManager;
    private int mLogLevel;
    private long mSessionID;
    private long mActivityID;
    private String mActivityName;
    private Date mLastEventTimeStamp;
    private Date mLastActivityEndTime;
    private String mDeveloperID;
    private ArrayList<Event> mOpenedEventsList;
    private EventLogDatabaseManager mDatabaseManager;

    private static Utils.StaticHandler mSyncEventHandler;
    private EventSyncAsyncTask mEventSyncAsyncTask;
    private ArrayList<Event> mTempStoredEventsList = new ArrayList<Event>();

    private EventManager() {
        mOpenedEventsList = new ArrayList<Event>();
    }

    private boolean initialized = false;

    public static EventManager getInstance(Context context) {
        if (mEventManager == null) {


            mEventManager = new EventManager();
            mEventManager.init(context, null);
        }
        return mEventManager;
    }

    private EventLogDatabaseManager getEventDB(Context context) {
        if (mDatabaseManager == null) {
            mDatabaseManager = new EventLogDatabaseManager(context);
        }
        return mDatabaseManager;
    }


    /**
     * Get and initiate handler instance if needed
     */
    private Handler getMyHandler() {
        if (mSyncEventHandler == null) {
            mSyncEventHandler = new Utils.StaticHandler(Looper.getMainLooper()) {
                @Override
                public void handleMessage(Message message) {
                    // This is where you do your work in the UI thread.
                    // Your worker tells you in the message what to do.
                }
            };
        }
        return mSyncEventHandler;
    }


    /**
     * this function init the event manager.
     *
     * @param params
     */
    public void init(Context context, JSONObject params) {
        SharedPreferences eventsPref = context.getSharedPreferences(EVENTS_SHARED_PREFERENCES_KEY, 0);
        try {
            String mServerAdress;
            if (params != null) {
                mLogLevel = params.optInt(LOG_LEVEL_KEY, EventManager.LOG_CRITICAL_LEVEL);
                mServerAdress = params.optString("serveraddress", EVENT_SERVER_ADDRESS_KEY);
                mDeveloperID = params.optString(EventParameters.DEVELOPER_ID, "");
                mSessionID = params.optInt(EventParameters.SESSION_ID, 0);
                //todo: save new values to preference and reload them
                SharedPreferences.Editor editor = eventsPref.edit();
                editor.putInt(LOG_LEVEL_KEY, mLogLevel);
                editor.putString(EventParameters.DEVELOPER_ID, mDeveloperID);
                editor.apply();
            } else {
                mLogLevel = eventsPref.getInt(LOG_LEVEL_KEY, EventManager.LOG_CRITICAL_LEVEL);
                mServerAdress = eventsPref.getString("serveraddress", EVENT_SERVER_ADDRESS_KEY);
                mDeveloperID = eventsPref.getString(EventParameters.DEVELOPER_ID, "");
                mSessionID = -1;
            }

            mActivityID = 0;//eventsPref.getLong(EVENTS_LAST_ACTIVITY_ID_KEY, 0);

            if (mDatabaseManager == null) {
                mDatabaseManager = new EventLogDatabaseManager(context);
            }
            com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog("init Event manager");
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager init " + ex.getMessage());
        }
        initialized = true;
    }//end init function


    /**
     * This method return wherever a log can be sent.
     *
     * @return is can send log.
     */
    public boolean getIsNeedEventsSync(Context context) {
        if (getEventDB(context).getEventsTable().isDBempty()) {
            return false;
        } else {
            return true;
        }
    }

    public EventBulk getNextBulk(Context context) {
        EventBulk result = new EventBulk();
        result.setDeviceParamsJSONObject(getBulkDeviceParamsJsonObject(context));
        result.setEventArrayList(getEventDB(context).getEventsTable().getEvents(calculateBulkSize(context)));

        return result;
    }

    public void removeBulkFromDB(Context context, EventBulk eventBulk) {
        getEventDB(context).getEventsTable().removeLogEventsFromDB(eventBulk.getEventArrayList());
    }

    private int calculateBulkSize(Context context) {
        // The minimum bulk is 30, the maximum is 120.
        int result = DAY_TIME_AMOUNT + THREE_G_CONNECTON_TYPE_AMOUNT + SCREEN_STATE_ON_AMOUNT;
        Calendar calendar = Calendar.getInstance();
        int currentHour = calendar.get(Calendar.HOUR_OF_DAY);
        if (currentHour >= NIGHT_TIME_HOUR_VALUE_START || currentHour <= NIGHT_TIME_HOUR_VALUE_END) {
            result += NIGHT_TIME_AMOUNT_EXTRA;
        }

        /*if (BaseConnectionClient.getConnectionType(context) == ConnectivityManager.TYPE_WIFI)
        {
            result += WIFI_CONNECTON_TYPE_AMOUNT_EXTRA;
        }*/

        if (ScreenUtils.getIsScreenOff(context) == true) {
            result += SCREEN_STATE_OFF_AMOUNT_EXTRA;
        }

        return result;
    }

    /**
     * This method create the session start event.
     * the returned value is the id for the new activity. res lower than 0 means there was some error
     */
    public long activityStart(Context context, WidgetType widgetType, int logLevel, String activityName, boolean storeForLaterSending) {
        try {
            mLastEventTimeStamp = new Date();

            if (mLastActivityEndTime == null) {
                mLastActivityEndTime = new Date(0);
            }
            Date lastEventTime = mLastEventTimeStamp;
            mLastEventTimeStamp = new Date();
            Long timeDelta = mLastEventTimeStamp.getTime() - mLastActivityEndTime.getTime();

            mActivityID += 1;

            Event openedEvent = new Event();
            fillBaseEventParameters(openedEvent, EventParameters.CATEGORY_ACTIVITY, EventParameters.ACTION_ACTIVITY_END, activityName, widgetType);
            openedEvent.setLogLevel(EventManager.LOG_CRITICAL_LEVEL);
            openedEvent.setCloseLevel(EventManager.LOG_CLOSE_ACTIVITY_LEVEL);
            mOpenedEventsList.add(openedEvent);

            mActivityName = activityName;

            //send event to server
            internalLogEvent(context, widgetType, logLevel, null, EventParameters.CATEGORY_ACTIVITY, EventParameters.ACTION_ACTIVITY_START, activityName, storeForLaterSending);
            com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog("activity start event generated");
            return mActivityID;
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager activityStart " + ex.getMessage());
        }
        return -1;

    }

    /**
     * This method create the session end event. if the returned value lower than 0 then some error accourd
     */
    public int activityEnd(Context context, WidgetType widgetType, int logLevel, long activityID, boolean storeForLaterSending) {

        try {
            //close all open events
            for (Iterator<Event> iterator = mOpenedEventsList.iterator(); iterator.hasNext(); ) {
                Event opendEvent = iterator.next();
                if (opendEvent.getCloseLevel() <= LOG_CLOSE_ACTIVITY_LEVEL) {
                    long duration = System.currentTimeMillis() - opendEvent.getEventCreationTime();
                    opendEvent.addParameterToJsonObject(EventParameters.DURATION, duration);
                    internalLogEvent(context, widgetType, opendEvent.getLogLevel(), opendEvent, null, null, null, storeForLaterSending);
                    iterator.remove();
                }
            }//end iterator
            com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog("activity end event called");
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager activity end " + ex.getMessage());
        }
        mLastActivityEndTime = new Date();
        return 0;
    }//end activityEnd function


    /**
     * This method create basic event
     */
    public int logClickEvent(Context context, WidgetType widgetType, String action, String label, String itemID, int position, boolean storeForLaterSending) {
        Event event = new Event();
        event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
        event.addParameterToJsonObject(EventParameters.ITEM_INDEX, position);
        internalLogEvent(context, widgetType, EventManager.LOG_NORMAL_LEVEL, event, EventParameters.CATEGORY_CONTENT_CLICK, action, label, storeForLaterSending);

        return 0;
    }//end logEvent function

    /**
     * This method create basic event
     */
    public int logEvent(Context context, WidgetType widgetType, int logLevel, String category, String action, String label, boolean storeForLaterSending) {
        internalLogEvent(context, widgetType, logLevel, null, category, action, label, storeForLaterSending);
        return 0;
    }//end logEvent function

    /**
     * This method create event with duration. if a log entry with the same resource key ends
     * the previous event and start a new one. the return value is the resource id.
     */
    public int logEventWithDuration(Context context, WidgetType widgetType, int logLevel, int resourceID, int closeLevel, String category, String action, String label, String itemID, boolean storeForLaterSending) {
        try {
            for (Iterator<Event> iterator = mOpenedEventsList.iterator(); iterator.hasNext(); ) {
                Event opendEvent = iterator.next();
                if (opendEvent.getResourceID() == resourceID) {
                    long duration = System.currentTimeMillis() - opendEvent.getEventCreationTime();
                    opendEvent.addParameterToJsonObject(EventParameters.DURATION, duration);
                    opendEvent.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
                    internalLogEvent(context, widgetType, opendEvent.getLogLevel(), opendEvent, null, null, null, storeForLaterSending);
                    iterator.remove();
                }
            }//end iterator
            com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog("activity end event called");
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logEventWithDuration " + ex.getMessage());
        }
        Event openedEvent = new Event();
        fillBaseEventParameters(openedEvent, category, action, label, widgetType);
        openedEvent.setLogLevel(logLevel);
        openedEvent.setCloseLevel(closeLevel);
        openedEvent.setResourceID(resourceID);
        mOpenedEventsList.add(openedEvent);

        return 0;
    }//end logEvent function

    /**
     * This method ends the event with duration based on the resource ID
     */
    public int logEventWithDurationEnd(Context context, WidgetType widgetType, int resourceID, boolean storeForLaterSending) {
        try {
            for (Iterator<Event> iterator = mOpenedEventsList.iterator(); iterator.hasNext(); ) {
                Event opendEvent = iterator.next();
                if (opendEvent.getResourceID() == resourceID) {
                    long duration = System.currentTimeMillis() - opendEvent.getEventCreationTime();
                    opendEvent.addParameterToJsonObject(EventParameters.DURATION, duration);
                    internalLogEvent(context, widgetType, opendEvent.getLogLevel(), opendEvent, null, null, null, storeForLaterSending);
                    iterator.remove();
                }
            }//end iterator
            com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog("activity end event called");
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logEventWithDuration " + ex.getMessage());
        }
        return 0;
    }//end logEvent function

    public int logSponsoredContentImpressionEvent(Context context, WidgetType widgetType, String action, String label, String advertiserID, String itemID, int positionIndex, boolean storeForLaterSending) {
        try {
            Event event = new Event();
            event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
            event.addParameterToJsonObject(EventParameters.ADVERTISER_ID, advertiserID);
            event.addParameterToJsonObject(EventParameters.ITEM_INDEX, positionIndex);
            internalLogEvent(context, widgetType, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_SPONSORED_CONTENT, action, label, storeForLaterSending);
            return 0;
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logImpressionEvent " + ex.getMessage());
        }
        return -1;
    }

    public int logSponsoredContentClickEvent(Context context, WidgetType widgetType, String label, String advertiserID, String itemID, int positionIndex, boolean storeForLaterSending) {
        try {
            Event event = new Event();
            event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
            event.addParameterToJsonObject(EventParameters.ADVERTISER_ID, advertiserID);
            event.addParameterToJsonObject(EventParameters.ITEM_INDEX, positionIndex);
            internalLogEvent(context, widgetType, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_CLICK, label, storeForLaterSending);
            return 0;
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logImpressionEvent " + ex.getMessage());
        }
        return -1;
    }

    public int logSponsoredContentInstallEvent(Context context, WidgetType widgetType, String label, String advertiserID, String itemID, int positionIndex, boolean storeForLaterSending) {
        try {
            Event event = new Event();
            event.addParameterToJsonObject(EventParameters.ITEM_ID, itemID);
            event.addParameterToJsonObject(EventParameters.ADVERTISER_ID, advertiserID);
            event.addParameterToJsonObject(EventParameters.ITEM_INDEX, positionIndex);
            internalLogEvent(context, widgetType, EventManager.LOG_CRITICAL_LEVEL, event, EventParameters.CATEGORY_SPONSORED_CONTENT, EventParameters.ACTION_PROMOTED_APP_INSTALL_COMPLETE, label, storeForLaterSending);
            return 0;
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error on EventManager logImpressionEvent " + ex.getMessage());
        }
        return -1;
    }

    /**
     * This method is used to log basic event
     */
    public int LogEventWithIntField(Context context, WidgetType widgetType, int logLevel, String category, String action, String label, int intField, boolean storeForLaterSending) {
        Event event = new Event();
        event.addParameterToJsonObject(EventParameters.INT_FIELD, intField);
        return (internalLogEvent(context, widgetType, logLevel, event, category, action, label, storeForLaterSending));
    }


    /**
     * This method is used to log basic event
     */
    public int LogEvent(Context context, WidgetType widgetType, int logLevel, Event event, String category, String action, String label, boolean storeForLaterSending) {
        return (internalLogEvent(context, widgetType, logLevel, event, category, action, label, storeForLaterSending));
    }

    /**
     * This method create create the events and add it to que
     */
    private int internalLogEvent(Context context, WidgetType widgetType, int logLevel, Event event, String category, String action, String label, boolean storeForLaterSending) {
        Event tempEvent;

        if (logLevel <= mLogLevel) {
            if (event == null) {
                tempEvent = new Event();
            } else {
                tempEvent = event;
            }
            fillBaseEventParameters(tempEvent, category, action, label, widgetType);

            com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog("Event sent to DB: " + tempEvent.getJSONObject().toString());

            if (storeForLaterSending) {
                mTempStoredEventsList.add(tempEvent);
            } else {
                getEventDB(context).getEventsTable().insertLogEventToDB(tempEvent);
            }
        }

        return 0;
    }


    private int fillBaseEventParameters(Event event, String category, String action, String label, WidgetType widgetType) {
        if (category != null) {
            event.addParameterToJsonObject(EventParameters.CATEGORY, category);
        }
        if (action != null) {
            event.addParameterToJsonObject(EventParameters.ACTION, action);
        }
        if (label != null) {
            event.addParameterToJsonObject(EventParameters.LABEL, label);
        }
        event.addParameterToJsonObject(EventParameters.ACTIVITY_NAME, mActivityName);
        event.addParameterToJsonObject(EventParameters.ACTIVITY_ID, mActivityID);
        event.addParameterToJsonObject(EventParameters.SESSION_ID, mSessionID);

        if (widgetType != WidgetType.WIDGET_TYPE_NONE) {
            event.addParameterToJsonObject(EventParameters.WIDGET_TYPE, widgetType.getValue());
        }

        event.addParameterToJsonObject(EventParameters.SDK_STYLE_VERSION, SDK_STYLE_VERSION_NUM);

//      SDKLogger.printInfoLog("Event basic parameters added: " + event.getJSONObject().toString());

        return 0;
    }

    private JSONObject getBulkDeviceParamsJsonObject(Context context) {
        JSONObject result = new JSONObject();
        if (context != null) {
            try {
                if (mDeveloperID == null) {
                    SharedPreferences eventsPref = context.getSharedPreferences(EVENTS_SHARED_PREFERENCES_KEY, 0);
                    mDeveloperID = eventsPref.getString(EventParameters.DEVELOPER_ID, "");
                }

               /* String deviceID = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
                CRC32 crc = new CRC32();
                crc.update(deviceID.getBytes());*/

                result.put(EventParameters.DEVICE_HASH, Utils.generateUniqeDeviceID(context.getPackageName(), mDeveloperID));

                result.put(EventParameters.USER_REFFERAL, DeviceUtils.getDeviceReferral(context));

                result.put(EventParameters.KIDOZ_APP_PACKAGE_NAME, context.getPackageName());
                PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
                result.put(EventParameters.APP_VERSION, String.valueOf(pInfo.versionName));
                result.put(EventParameters.APP_ID, String.valueOf(pInfo.packageName));

                result.put(EventParameters.DEVELOPER_ID, mDeveloperID);
                result.put(EventParameters.EVENT_TYPE, EventParameters.EVENT_TYPE_NORMAL);
                result.put("EventVersion", 0);

                //todo: this data should be sent only once
                result.put(EventParameters.DEVICE_TYPE, String.valueOf(Build.DEVICE));
                result.put(EventParameters.DEVICE_BRAND, String.valueOf(Build.MANUFACTURER));
                result.put(EventParameters.DEVICE_MODEL, String.valueOf(Build.MODEL));
                result.put(EventParameters.SCREEN_TYPE, String.valueOf((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK)));
                result.put(EventParameters.SCREEN_SIZE, String.valueOf(ScreenUtils.getDeviceScreenSizeInInches(context)));

                WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
                DisplayMetrics metrics = new DisplayMetrics();
                windowManager.getDefaultDisplay().getMetrics(metrics);

                result.put(EventParameters.DPI_FACTOR, String.valueOf(metrics.densityDpi));
                result.put(EventParameters.SCREEN_WIDTH, String.valueOf(DeviceUtils.getScreenSize(context, true)));
                result.put(EventParameters.SCREEN_HEIGHT, String.valueOf(DeviceUtils.getScreenSize(context, false)));
                result.put(EventParameters.OS_TYPE, String.valueOf(OS_TYPE));
                result.put(EventParameters.OS_VERSION, String.valueOf(Build.VERSION.SDK_INT));
                //result.put(EventParameters.CONNECTION_TYPE, EventParameters.convertConnectionTypeToString(BaseConnectionClient.getConnectionType(context)));
                result.put(EventParameters.DEVICE_LANGUAGE, String.valueOf(Locale.getDefault().getLanguage()));
                result.put(EventParameters.TIME_ZONE, EventParameters.getCurrentTimezoneOffset());
                result.put(EventParameters.COUNTRY, String.valueOf(Locale.getDefault().getCountry()));
                result.put(EventParameters.EVENT_TYPE, EventParameters.EVENT_TYPE_DEVICE_INFO);

                result.put(EventParameters.ACTUAL_SDK_VERSION, ConstantDef.ACTUAL_SDK_VERSION);
            } catch (Exception ex) {
                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog("Error when trying to create device params: " + ex.getMessage());
            }
        }

        return result;
    }


    /**
     * This method simply start the SyncEventService.
     *
     * @param context
     */
    public void startEventsSync(final Context context) {
        getMyHandler().post(new Runnable() {
            @Override
            public void run() {
                if (mEventSyncAsyncTask == null || mEventSyncAsyncTask.isCancelled() == true || mEventSyncAsyncTask.getStatus() == AsyncTask.Status.FINISHED) {
                    mEventSyncAsyncTask = new EventSyncAsyncTask(context);
                    if (Build.VERSION.SDK_INT < 11) {
                        mEventSyncAsyncTask.execute();
                    } else {
                        mEventSyncAsyncTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
                    }
                }
            }
        });
    }

    /**
     * This method simply start the SyncEventService.
     *
     * @param context
     */
    public void startEventsSync(final Context context, int posDelayMillis) {
        getMyHandler().postDelayed(new Runnable() {
            @Override
            public void run() {
                if (mEventSyncAsyncTask == null || mEventSyncAsyncTask.isCancelled() == true || mEventSyncAsyncTask.getStatus() == AsyncTask.Status.FINISHED) {
                    mEventSyncAsyncTask = new EventSyncAsyncTask(context);
                    if (Build.VERSION.SDK_INT < 11) {
                        mEventSyncAsyncTask.execute();
                    } else {
                        mEventSyncAsyncTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
                    }
                }
            }
        }, posDelayMillis);
    }

    /**
     * Log stored tem events
     *
     * @return true if there where stored events , false otherwise
     */
    public boolean logAllTempStoredEvents(Context context) {
        if (mTempStoredEventsList != null && mTempStoredEventsList.isEmpty() == false) {
            for (int i = 0; i < mTempStoredEventsList.size(); i++) {
                getEventDB(context).getEventsTable().insertLogEventToDB(mTempStoredEventsList.get(i));
            }
            clearAllTempStoredEvents();
            return true;
        } else {
            return false;
        }
    }

    public void clearAllTempStoredEvents() {
        if (mTempStoredEventsList != null) {
            mTempStoredEventsList.clear();
        }
    }
}
