package com.kidoz.sdk.api.ui_views.panel_view;

import android.content.Context;
import android.graphics.Color;
import android.text.TextUtils;
import android.util.TypedValue;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.kidoz.sdk.api.general.custom_views.CustomCardView.KidozCardView;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.picasso_related.PicassoOk;
import com.kidoz.sdk.api.picasso_related.RoundedTransformation;
import com.kidoz.sdk.api.structure.ContentItem;

/**
 * This is the item that is shown inside a panel view.
 * Each item contains a background, a thumbnail, an icon and a name field.
 * The item size is constant and scaled automatically based on the device screen size and resolution.
 * Created by roni on 10/7/2015.
 */
public class PanelContentItem extends KidozCardView
{
    private RelativeLayout mContainer;
    private TextView mItemName;
    private ImageView mItemThumbnail;
    private ImageView mItemTypeIcon;
    private int mWidth;
    private int mHeight;
    private int mThumbnailCornersRadius;
    private int mThumbnailWidth;
    private int mThumbnailHeight;

    public PanelContentItem(Context context)
    {
        super(context);
        initParameters();
        initView();
    }

    private void initParameters()
    {
        float WIDTH = 517f / 1080f;
        float HEIGHT = 402f / 1920f;
        float SCALE_FACTOR = ScreenUtils.getScaleFactor(getContext());
        int screenWidth = Math.min(ScreenUtils.getScreenSize(getContext(), true), ScreenUtils.getScreenSize(getContext(), false));
        int screenHeight = Math.max(ScreenUtils.getScreenSize(getContext(), true), ScreenUtils.getScreenSize(getContext(), false));
        mWidth = (int) (517f * SCALE_FACTOR);
        mHeight = (int) (402f * SCALE_FACTOR);
    }

    private void initView()
    {
        initRootView();
        initContainer();
        initItemThumbnail();
        initItemTypeIcon();
        initItemName();
    }

    private void initRootView()
    {
        setContentPadding(0, 0, 0, 0);
        setRadius(com.kidoz.sdk.api.general.utils.Utils.dpTOpx(getContext(), 6));
        setCardElevation(com.kidoz.sdk.api.general.utils.Utils.dpTOpx(getContext(), 4));

        int margin = com.kidoz.sdk.api.general.utils.Utils.dpTOpx(getContext(), 3);
        KidozCardView.LayoutParams params = new KidozCardView.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        params.leftMargin = margin;
        params.topMargin = margin;
        params.rightMargin = margin;
        params.bottomMargin = margin;
        setLayoutParams(params);
    }

    private void initContainer()
    {
        mContainer = new RelativeLayout(getContext());
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(mWidth, mHeight);
        addView(mContainer, params);
    }

    private void initItemThumbnail()
    {
        float WIDTH = 504f / 517f;
        float HEIGHT = 260f / 402f;
        float MARGIN_TOP = 6.5f / 402f;

        mItemThumbnail = new ImageView(getContext());
        mItemThumbnail.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        mItemThumbnail.setScaleType(ImageView.ScaleType.FIT_XY);

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        params.width = (int) (mWidth * WIDTH);
        params.height = (int) (mHeight * HEIGHT);
        params.topMargin = (int) (mHeight * MARGIN_TOP);
        params.addRule(RelativeLayout.CENTER_HORIZONTAL);
        mContainer.addView(mItemThumbnail, params);
        mThumbnailCornersRadius = com.kidoz.sdk.api.general.utils.Utils.dpTOpx(getContext(), 5);
        mThumbnailWidth = params.width;
        mThumbnailHeight = params.height;
    }

    private void initItemTypeIcon()
    {
        float SIZE = 114f / 517f;
        float MARGIN_RIGHT = 33f / 517f;
        mItemTypeIcon = new ImageView(getContext());
        mItemTypeIcon.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        mItemTypeIcon.setScaleType(ImageView.ScaleType.FIT_CENTER);
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
        params.addRule(RelativeLayout.ALIGN_BOTTOM, mItemThumbnail.getId());
        params.width = (int) (mWidth * SIZE);
        params.height = (int) (mWidth * SIZE);
        params.rightMargin = (int) (mWidth * MARGIN_RIGHT);
        params.bottomMargin = (int) (-1 * mWidth * SIZE / 2);
        mContainer.addView(mItemTypeIcon, params);
    }

    private void initItemName()
    {
        float TEXT_SIZE = 44f / 402f;
        float MARGIN_HORIZONTAL = 25f / 517f;
        float MARGIN_TOP = 12.5f / 402f;
        mItemName = new TextView(getContext());
        mItemName.setTextSize(TypedValue.COMPLEX_UNIT_PX, mHeight * TEXT_SIZE);
        mItemName.setMaxLines(2);
        mItemName.setEllipsize(TextUtils.TruncateAt.END);
        mItemName.setTextColor(Color.parseColor("#484848"));
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        params.addRule(RelativeLayout.BELOW, mItemThumbnail.getId());
        params.addRule(RelativeLayout.LEFT_OF, mItemTypeIcon.getId());
        params.leftMargin = (int) (mWidth * MARGIN_HORIZONTAL);
        params.rightMargin = (int) (mWidth * MARGIN_HORIZONTAL);
        params.topMargin = (int) (mHeight * MARGIN_TOP);
        mContainer.addView(mItemName, params);
    }

    private void loadBackgroundColor(ContentItem contentItem)
    {
        setCardBackgroundColor(Color.WHITE);
        if (contentItem != null)
        {
            setCardBackgroundColor(contentItem.getPaliteBgColor());
        }
    }

    private void loadThumbnail(ContentItem contentItem)
    {
        mItemThumbnail.setImageBitmap(null);
        if (contentItem != null)
        {
            PicassoOk.getPicasso(getContext()).load(contentItem.getThumb()).transform(new RoundedTransformation(mThumbnailCornersRadius, 0)).resize(mThumbnailWidth, mThumbnailHeight).centerCrop().into(mItemThumbnail);
        }
    }

    private void loadItemTypeIcon(ContentItem contentItem)
    {
        mItemTypeIcon.setImageBitmap(null);
        if (contentItem != null)
        {
            PicassoOk.getPicasso(getContext()).load(contentItem.getSecThumb()).into(mItemTypeIcon);
        }
    }

    private void loadIteNameAndSetTextColor(ContentItem contentItem)
    {
        mItemName.setText(null);
        if (contentItem != null)
        {
            mItemName.setTextColor(contentItem.getPaliteTextColor());
            mItemName.setText(contentItem.getName());
        }
    }

    public void setData(ContentItem contentItem)
    {
        loadBackgroundColor(contentItem);
        loadThumbnail(contentItem);
        loadItemTypeIcon(contentItem);
        loadIteNameAndSetTextColor(contentItem);
    }
}
