package com.kidoz.sdk.api.ui_views.panel_view;

import android.animation.Animator;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.view.View;
import android.widget.ImageView;

import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.assets_handling.ImageAssetsUtils;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.picasso_related.PicassoOk;

import java.io.File;

/**
 * Created by roni on 10/7/2015.
 */
public class PanelButton extends ImageView
{
    private final String TAG = PanelButton.class.getSimpleName();
    private PanelButtonListener mPanelButtonListener;
    private PANEL_BUTTON_TYPE mButtonType;
    private PANEL_TYPE mPanelType;
    private File mImageFile;
    private Paint mPaint;
    private int mWidth;
    private int mHeight;
    private int mAnimationType;

    // Only parental lock button related
    private boolean mIsLockActive;
    private RectF mLockBodyRectF;
    private RectF mLockIconRectF;
    private RectF mWhiteBackgroundRoundedRectF;
    private RectF mWhiteBackgroundRectF;

    public PanelButton(Context context, PANEL_TYPE panelType, PANEL_BUTTON_TYPE panelButtonType, PanelButtonListener panelButtonListener)
    {
        super(context);
        mPanelType = panelType;
        mButtonType = panelButtonType;
        mPanelButtonListener = panelButtonListener;
        setScaleType(ScaleType.FIT_XY);
        initView();
    }

    private void initView()
    {
        initImageFile();
        initImageParameters();
        initPaints();
        initRectF();
        initOnClickAnimation();
    }

    private void initImageFile()
    {
        switch (mButtonType)
        {
            case HANDLE_OPEN:
            {
                mImageFile = ImageAssetsUtils.getAssetFile(getContext(), ImageAssetsUtils.APP_ASSET_TYPE.PANEL_BUTTON_HANDLE);
                break;
            }
            case HANDLE_CLOSE:
            {
                mImageFile = ImageAssetsUtils.getAssetFile(getContext(), ImageAssetsUtils.APP_ASSET_TYPE.PANEL_BUTTON_HANDLE);
                break;
            }
            case EXIT:
            {
                mImageFile = ImageAssetsUtils.getAssetFile(getContext(), ImageAssetsUtils.APP_ASSET_TYPE.PANEL_BUTTON_EXIT);
                break;
            }
            case ABOUT_KIDOZ:
            {
                mImageFile = ImageAssetsUtils.getAssetFile(getContext(), ImageAssetsUtils.APP_ASSET_TYPE.PANEL_BUTTON_ABOUT_KIDOZ);
                break;
            }
            case PARENTAL_LOCK:
            {
                mImageFile = ImageAssetsUtils.getAssetFile(getContext(), ImageAssetsUtils.APP_ASSET_TYPE.PANEL_BUTTON_PARENTAL_LOCK);
                break;
            }
        }
    }

    private void initPaints()
    {
        mPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mPaint.setStyle(Paint.Style.FILL_AND_STROKE);
        mPaint.setColor(Color.parseColor("#ffffff"));

        setLayerType(LAYER_TYPE_SOFTWARE, mPaint);
    }

    private void initRectF()
    {
        float left = (37f / 142f) * mWidth;
        float top = (58f / 142f) * mHeight;
        float right = (104f / 142f) * mWidth;
        float bottom = (108f / 142f) * mHeight;
        mLockBodyRectF = new RectF(left, top, right, bottom);

        left = (68f / 142f) * mWidth;
        top = (78f / 142f) * mHeight;
        right = (74f / 142f) * mWidth;
        bottom = (97f / 142f) * mHeight;
        mLockIconRectF = new RectF(left, top, right, bottom);

        switch (mPanelType)
        {
            case LEFT:
            {
                left = 0f;
                top = 0f;
                right = mWidth / 2;
                bottom = mHeight;
                mWhiteBackgroundRectF = new RectF(left, top, right, bottom);
                break;
            }
            case TOP:
            {
                left = 0f;
                top = 0f;
                right = mWidth;
                bottom = mHeight / 2;
                mWhiteBackgroundRectF = new RectF(left, top, right, bottom);
                break;
            }
            case RIGHT:
            {
                left = mWidth / 2;
                top = 0f;
                right = mWidth;
                bottom = mHeight;
                mWhiteBackgroundRectF = new RectF(left, top, right, bottom);
                break;
            }
            case BOTTOM:
            {
                left = 0f;
                top = mHeight / 2;
                right = mWidth;
                bottom = mHeight;
                mWhiteBackgroundRectF = new RectF(left, top, right, bottom);
                break;
            }
        }
        left = 0f;
        top = 0f;
        right = mWidth;
        bottom = mHeight;
        mWhiteBackgroundRoundedRectF = new RectF(left, top, right, bottom);
    }

    private void initImageParameters()
    {
        float scaleFactor = ScreenUtils.getScaleFactor(getContext());
        mWidth = 142;
        mHeight = 142;
        mAnimationType = 1;
        if (mImageFile != null && mImageFile.exists() == true)
        {
            String width = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), ImageAssetsUtils.PANEL_BUTTON_WIDTH, "142", Context.MODE_PRIVATE);
            String height = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), ImageAssetsUtils.PANEL_BUTTON_HEIGHT, "142", Context.MODE_PRIVATE);
            String animationType = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), ImageAssetsUtils.PANEL_BUTTON_ANIMATION, "1", Context.MODE_PRIVATE);

            try
            {
                mWidth = Integer.parseInt(width);
                mHeight = Integer.parseInt(height);
                mAnimationType = Integer.parseInt(animationType);
            }
            catch (Exception ex)
            {
                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to init image parameters: " + ex.getMessage());
            }
        }
        mWidth *= scaleFactor;
        mHeight *= scaleFactor;
    }

    private void initOnClickAnimation()
    {
        setOnClickListener(new OnClickListener()
        {
            @Override public void onClick(View view)
            {
                if (mPanelButtonListener != null)
                {
                    mPanelButtonListener.onClickStarted();
                }

                switch (mAnimationType)
                {
                    case 1:
                    {
                        GenAnimator.playPanelButtonClickAnimation(view, mPanelType, new Animator.AnimatorListener()
                        {
                            @Override public void onAnimationStart(Animator animation)
                            {

                            }

                            @Override public void onAnimationEnd(Animator animation)
                            {
                                if (mPanelButtonListener != null)
                                {
                                    mPanelButtonListener.onClickEnd();
                                }
                            }

                            @Override public void onAnimationCancel(Animator animation)
                            {

                            }

                            @Override public void onAnimationRepeat(Animator animation)
                            {

                            }
                        });
                        break;
                    }
                }
            }
        });
    }

    private void loadImage()
    {
        if (mImageFile.exists() == true)
        {
            if (getDrawable() == null)
            {
                PicassoOk.getPicasso(getContext()).load(mImageFile).into(this);
            }
        }
    }

    @Override protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec)
    {
        setMeasuredDimension(mWidth, mHeight);
    }

    @Override protected void onDraw(Canvas canvas)
    {
        super.onDraw(canvas);
        if (mImageFile.exists() == false)
        {
            // Draw background shape
            // Draw lock body

            float radius = (71f / 142f) * mWidth;
            float centerX = (71f / 142f) * mWidth;
            float centerY = (71f / 142f) * mHeight;

            mPaint.setColor(Color.parseColor("#ffffff"));
            radius = (20f / 142f) * mWidth;
            canvas.drawRect(mWhiteBackgroundRectF, mPaint);
            canvas.drawRoundRect(mWhiteBackgroundRoundedRectF, radius, radius, mPaint);

            switch (mButtonType)
            {
                case HANDLE_OPEN:
                {
                    // Draw the + shape
                    mPaint.setColor(Color.parseColor("#39b0c5"));
                    float left = (62f / 142f) * mWidth;
                    float top = (28f / 142f) * mHeight;
                    float right = (80f / 142f) * mWidth;
                    float bottom = (114f / 142f) * mHeight;
                    canvas.drawRect(left, top, right, bottom, mPaint);
                    canvas.restore();
                    canvas.save();
                    canvas.rotate(90f, centerX, centerY);
                    canvas.drawRect(left, top, right, bottom, mPaint);
                    canvas.restore();
                    break;
                }
                case HANDLE_CLOSE:
                {
                    // Draw the + shape
                    mPaint.setColor(Color.parseColor("#39b0c5"));
                    float left = (62f / 142f) * mWidth;
                    float top = (28f / 142f) * mHeight;
                    float right = (80f / 142f) * mWidth;
                    float bottom = (114f / 142f) * mHeight;
                    canvas.restore();
                    canvas.save();
                    canvas.rotate(45f, centerX, centerY);
                    canvas.drawRect(left, top, right, bottom, mPaint);
                    canvas.restore();
                    canvas.save();
                    canvas.rotate(-45f, centerX, centerY);
                    canvas.drawRect(left, top, right, bottom, mPaint);
                    canvas.restore();
                    break;
                }
                case EXIT:
                {
                    break;
                }
                case ABOUT_KIDOZ:
                {
                    // Draw the KIDOZ logo
                    mPaint.setColor(Color.parseColor("#d90000"));
                    radius = (22.5f / 142f) * mWidth;
                    centerX = (42.5f / 142f) * mWidth;
                    centerY = (82.5f / 142f) * mHeight;
                    canvas.drawCircle(centerX, centerY, radius, mPaint);

                    mPaint.setColor(Color.parseColor("#4aaed3"));
                    radius = (22.5f / 142f) * mWidth;
                    centerX = (99.5f / 142f) * mWidth;
                    centerY = (82.5f / 142f) * mHeight;
                    canvas.drawCircle(centerX, centerY, radius, mPaint);

                    mPaint.setColor(Color.parseColor("#f7da32"));
                    radius = (25f / 142f) * mWidth;
                    centerX = (71.5f / 142f) * mWidth;
                    centerY = (45.5f / 142f) * mHeight;
                    canvas.drawCircle(centerX, centerY, radius, mPaint);

                    mPaint.setColor(Color.parseColor("#f79e3d"));
                    radius = (22.5f / 142f) * mWidth;
                    centerX = (43.5f / 142f) * mWidth;
                    centerY = (58.5f / 142f) * mHeight;
                    canvas.drawCircle(centerX, centerY, radius, mPaint);

                    mPaint.setColor(Color.parseColor("#a7d125"));
                    radius = (22.5f / 142f) * mWidth;
                    centerX = (98.5f / 142f) * mWidth;
                    centerY = (58.5f / 142f) * mHeight;
                    canvas.drawCircle(centerX, centerY, radius, mPaint);

                    mPaint.setColor(Color.parseColor("#ffffff"));
                    radius = (25f / 142f) * mWidth;
                    centerX = (71f / 142f) * mWidth;
                    centerY = (74f / 142f) * mHeight;
                    canvas.drawCircle(centerX, centerY, radius, mPaint);
                    break;
                }
                case PARENTAL_LOCK:
                {
                    if (mIsLockActive == true)
                    {
                        // Lock active
                        mPaint.setColor(Color.parseColor("#39b0c5"));
                        radius = (29f / 142f) * mWidth;
                        centerX = (71f / 142f) * mWidth;
                        centerY = (51f / 142f) * mHeight;
                        canvas.drawCircle(centerX, centerY, radius, mPaint);

                        mPaint.setColor(Color.parseColor("#ffffff"));
                        radius = (17f / 142f) * mWidth;
                        centerX = (71f / 142f) * mWidth;
                        centerY = (51f / 142f) * mHeight;
                        canvas.drawCircle(centerX, centerY, radius, mPaint);
                    }
                    else
                    {
                        // Lock not active
                        mPaint.setColor(Color.parseColor("#fe9400"));
                        radius = (29f / 142f) * mWidth;
                        centerX = (71f / 142f) * mWidth;
                        centerY = (51f / 142f) * mHeight;
                        canvas.drawCircle(centerX, centerY, radius, mPaint);

                        centerX = (98f / 142f) * mWidth;
                        centerY = (52.5f / 142f) * mHeight;
                        canvas.save();
                        canvas.rotate(-45f, centerX, centerY);

                        float left = (72f / 142f) * mWidth;
                        float top = (36f / 142f) * mHeight;
                        float right = (124f / 142f) * mWidth;
                        float bottom = (69f / 142f) * mHeight;
                        mPaint.setColor(Color.parseColor("#ffffff"));
                        canvas.drawRect(left, top, right, bottom, mPaint);
                        canvas.restore();

                        mPaint.setColor(Color.parseColor("#ffffff"));
                        radius = (17f / 142f) * mWidth;
                        centerX = (71f / 142f) * mWidth;
                        centerY = (51f / 142f) * mHeight;
                        canvas.drawCircle(centerX, centerY, radius, mPaint);
                    }

                    // Draw lock body
                    mPaint.setColor(Color.parseColor("#39b0c5"));
                    radius = (10f / 142f) * mWidth;
                    canvas.drawRoundRect(mLockBodyRectF, radius, radius, mPaint);

                    mPaint.setColor(Color.parseColor("#ffffff"));
                    canvas.drawRect(mLockIconRectF, mPaint);

                    radius = (8f / 142f) * mWidth;
                    centerX = (71f / 142f) * mWidth;
                    centerY = (78f / 142f) * mHeight;
                    canvas.drawCircle(centerX, centerY, radius, mPaint);
                    break;
                }
            }
        }
        else
        {
            loadImage();
        }
    }

    public void setPanelButtonType(PANEL_BUTTON_TYPE panelButtonType)
    {
        mButtonType = panelButtonType;
        setImageDrawable(null);
        initImageFile();
        invalidate();
    }

    public interface PanelButtonListener
    {
        void onClickStarted();

        void onClickEnd();
    }

    public void setIsLockActive(boolean isLockActive)
    {
        mIsLockActive = isLockActive;
        invalidate();
    }
}
