package com.kidoz.sdk.api.players.web_player;

import android.app.Dialog;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.os.Bundle;
import android.os.Handler;
import android.support.v4.app.DialogFragment;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentTransaction;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.structure.ContentItem;


/**
 * Created by orikam on 7/15/15.
 */
public class WebPlayerFragmentDialog extends DialogFragment {
    private static final String TAG = WebPlayerFragmentDialog.class.getSimpleName();

    private static final int WEB_PLAYER_RESOURCE_ID = 1;

    private static final String CONTENT_ITEM_SAVE_STATE_KEY = "CONTENT_ITEM_SAVE_STATE_KEY";
    private static final String WEB_PLAYER_TYPE_SAVE_STATE_KEY = "WEB_PLAYER_TYPE_SAVE_STATE_KEY";
    private static final String ORIENTATION_STATE_SAVE_STATE_KEY = "ORIENTATION_STATE_SAVE_STATE_KEY";
    private static final String EVENT_ID_STATE_SAVE_STATE_KEY = "EVENT_ID_STATE_SAVE_STATE_KEY";

    private ContentItem mContentItem;
    private WebPlayerType mWebPlayerType;
    private WebPlayerView mWebPlayerView;

    private int mCurrentOrientationState = -1;
    private long mEventID;
    private boolean mAlllowOrientationMofify = false;

    public static WebPlayerFragmentDialog newInstance(ContentItem contentItem, WebPlayerType webPlayerType, boolean allowOrientationModification) {
        WebPlayerFragmentDialog f = new WebPlayerFragmentDialog();
        f.mContentItem = contentItem;
        f.mWebPlayerType = webPlayerType;
        f.mAlllowOrientationMofify = allowOrientationModification;
        return f;
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        int style = DialogFragment.STYLE_NO_FRAME;
        int theme = android.R.style.Theme_Black_NoTitleBar_Fullscreen;
        if (mWebPlayerType != null && mWebPlayerType == WebPlayerType.EXTERNAL_LINK) {
            theme = android.R.style.Theme_Translucent_NoTitleBar_Fullscreen;
        }
        setStyle(style, theme);

        if(mAlllowOrientationMofify) {
            if (savedInstanceState != null && savedInstanceState.containsKey(ORIENTATION_STATE_SAVE_STATE_KEY)) {
                mCurrentOrientationState = savedInstanceState.getInt(ORIENTATION_STATE_SAVE_STATE_KEY);
            } else {
                mCurrentOrientationState = getResources().getConfiguration().orientation;
                getActivity().setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR);
            }
        }

        setCancelable(true);
    }

    @Override
    public Dialog onCreateDialog(Bundle savedInstanceState) {
        Dialog dialog = super.onCreateDialog(savedInstanceState);
        //dialog.getWindow().getAttributes().windowAnimations = R.style.FullViewInOutAnimation;
        dialog.getWindow().requestFeature(Window.FEATURE_PROGRESS);
        return dialog;
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        if (savedInstanceState != null) {
            if (savedInstanceState.containsKey(CONTENT_ITEM_SAVE_STATE_KEY)) {
                mContentItem = (ContentItem) savedInstanceState.getSerializable(CONTENT_ITEM_SAVE_STATE_KEY);
            }
            mWebPlayerType = (WebPlayerType) savedInstanceState.getSerializable(WEB_PLAYER_TYPE_SAVE_STATE_KEY);
            mEventID = savedInstanceState.getLong(EVENT_ID_STATE_SAVE_STATE_KEY);
        }

        mWebPlayerView = new WebPlayerView(getActivity(), mWebPlayerType);
        if (mContentItem != null) {
            mWebPlayerView.loadWebUrl(mContentItem.getData(), mContentItem.getId(), mContentItem.getName());
        }

        mWebPlayerView.setOnCloseButtonClickListenr(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                closeDialog(getFragmentManager());
            }
        });

        mWebPlayerView.setOnRemoveViewRequestListener(new WebPlayerView.IOnRemoveViewRequestListener() {
            @Override
            public void onRemoveView() {
                closeDialog(getFragmentManager());
            }
        });
        return mWebPlayerView;
    }


    @Override
    public void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putSerializable(CONTENT_ITEM_SAVE_STATE_KEY, mContentItem);
        outState.putSerializable(WEB_PLAYER_TYPE_SAVE_STATE_KEY, mWebPlayerType);
        outState.putSerializable(ORIENTATION_STATE_SAVE_STATE_KEY, mCurrentOrientationState);
        outState.putSerializable(EVENT_ID_STATE_SAVE_STATE_KEY, mEventID);
    }

    /**
     * Show dialog view Set
     *
     * @param fragmentManager support fragment manager
     */
    public void openDialog(final FragmentManager fragmentManager) {
        new Handler().post(new Runnable() {
            public void run() {
                try {
                    FragmentTransaction ft = fragmentManager.beginTransaction();
                    ft.setTransition(FragmentTransaction.TRANSIT_FRAGMENT_OPEN);
                    Fragment prev = fragmentManager.findFragmentByTag(TAG);
                    if (prev != null) {
                        ft.remove(prev);
                    } else {
                        if (isAdded() == false) {
                            ft.addToBackStack(null);
                            show(fragmentManager, TAG);
                        }
                    }
                } catch (Exception ex) {
                    com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "Error when trying to open dialog: " + ex.getMessage());
                }
            }
        });
    }

    /**
     * Dismiss dialog view
     *
     * @param fragmentManager support fragment manager
     */
    public void closeDialog(FragmentManager fragmentManager) {
        try {
            FragmentTransaction ft = fragmentManager.beginTransaction();
            ft.setTransition(FragmentTransaction.TRANSIT_FRAGMENT_CLOSE);
            Fragment prev = fragmentManager.findFragmentByTag(TAG);
            if (prev != null) {
                ft.remove(prev);
                ft.commit();
            }

            // Restore previous orientation
            if(mAlllowOrientationMofify) {
                restoreUserOriginalOrientation();
            }

            if (this != null) {
                this.dismissAllowingStateLoss();
            }
        } catch (Exception ex) {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to close web player: " + ex.getMessage());
        }
    }

    @Override
    public void onDetach() {
        super.onDetach();
        if (mWebPlayerView != null) {
            try {
                mWebPlayerView.releseAndcloseView();
            } catch (Exception ex) {
                com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, "Error when trying to onDetach: " + ex.getMessage());
            }
        }
    }

    @Override
    public void onResume() {
        super.onResume();

        String activityName = "";
        String action = "";
        switch (mWebPlayerType) {
            case EXTERNAL_LINK:
                activityName = "Web Player External Link";
                action = EventParameters.ACTION_EXTERNAL_LINK_VIEW;
                break;
            case ONLINE_GAME:
                activityName = "Web Player Online Game";
                action = EventParameters.ACTION_GAME_VIEW;
                break;
            case WEB_BROWSER:
                activityName = "Web Player Web Browser";
                action = EventParameters.ACTION_WEBSITE_VIEW;
                break;
        }
        mEventID = EventManager.getInstance().activityStart(getActivity(), EventManager.LOG_CRITICAL_LEVEL, activityName);
        EventManager.getInstance().logEventWithDuration(getActivity(), EventManager.LOG_NORMAL_LEVEL, WEB_PLAYER_RESOURCE_ID,
                EventManager.LOG_CLOSE_RESOURCE_LEVEL, EventParameters.CATEGORY_WEB_PLAYER,
                action, mContentItem.getName(), mContentItem.getId());
    }

    @Override
    public void onPause() {
        EventManager.getInstance().activityEnd(getActivity(), EventManager.LOG_CRITICAL_LEVEL, mEventID);
        EventManager.getInstance().logEventWithDurationEnd(getActivity(), WEB_PLAYER_RESOURCE_ID);
        super.onPause();
    }


    /**
     * Restore user original orientation
     */
    private void restoreUserOriginalOrientation() {
        if (getActivity() != null && getActivity().isFinishing() == false) {
            switch (mCurrentOrientationState) {
                case Configuration.ORIENTATION_LANDSCAPE: {
                    getActivity().setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE);
                    break;
                }

                case Configuration.ORIENTATION_PORTRAIT: {
                    getActivity().setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT);
                    break;
                }
                case Configuration.ORIENTATION_UNDEFINED: {
                    getActivity().setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR);
                    break;
                }
            }
        }
    }
}
