package com.kidoz.sdk.api.players;

import android.animation.Animator;
import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.graphics.Color;
import android.graphics.Point;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.AccelerateInterpolator;
import android.view.animation.DecelerateInterpolator;
import android.webkit.WebSettings;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;

import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.dialogs.WebDialog.VideoEnabledWebChromeClient;
import com.kidoz.sdk.api.dialogs.WebDialog.VideoEnabledWebView;
import com.kidoz.sdk.api.general.BaseDialog;
import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.enums.ContentType;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.general.utils.YouTubeUtils;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.KidozTopBar;
import com.kidoz.sdk.api.ui_views.loading_progress_view.LoadingProgressView;

/**
 * Created by roni on 10/13/2015.
 */
public class FullScreenViewDialog extends BaseDialog
{

    private static final String ORIENTATION_STATE_SAVE_STATE_KEY = "ORIENTATION_STATE_SAVE_STATE_KEY";
    private static final String ORIENTATION_MODFY_SAVE_STATE_KEY = "ORIENTATION_MODFY_SAVE_STATE_KEY";

    private final String TAG = FullScreenViewDialog.class.getSimpleName();
    private final int BLACK_CURTAIN_HIDE_DELAY = 50;
    private final int TOP_BAR_ANIMATION_DURATION = 500;
    private final int TOP_BAR_ANIMATION_DELAY = 3000;

    private FrameLayout mContentView;
    private FrameLayout mCustomViewContainer;
    private View mCustomView;
    private KidozTopBar mKidozTopBar;
    private LoadingProgressView mLoadingProgressView;
    private View mBlackCurtain;

    // WebView related
    private VideoEnabledWebChromeClient mVideoEnabledWebChromeClient;

    // Video related
    private VideoEnabledWebView mVideoEnabledWebView;
    private RelativeLayout mVideoLayer;
    private View mTransparentView;
    private ContentItem mContentItem;

    // Animation related
    private Handler mHandler;
    private boolean mIsTopBarHidden;
    private boolean mIsAnimationRunning;

    // Log events related
    private long mEventID;

    private boolean mAlllowOrientationModify = false;

    public FullScreenViewDialog(Context context, boolean allowModifyOrientation)
    {
        super(context, android.R.style.Theme_Black_NoTitleBar_Fullscreen, allowModifyOrientation);
        setCancelable(true);
    }

    @Override protected void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        if (mAlllowOrientationModify)
        {
            if (savedInstanceState != null && savedInstanceState.containsKey(ORIENTATION_STATE_SAVE_STATE_KEY))
            {
                mCurrentOrientationState = savedInstanceState.getInt(ORIENTATION_STATE_SAVE_STATE_KEY);
            }
            else
            {
                if (mCallingActivity instanceof Activity && mCallingActivity.isFinishing() == false)
                {
                    mCurrentOrientationState = mCallingActivity.getResources().getConfiguration().orientation;
                    mCallingActivity.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR);
                }
            }
        }

        initDialog();
        setContentView(mCustomViewContainer);

        setOnDismissListener(new OnDismissListener()
        {
            @Override public void onDismiss(DialogInterface dialog)
            {
                // Restore previous orientation
                if (mAlllowOrientationModify)
                {
                    restoreUserOriginalOrientation();
                }
            }
        });
    }


    @Override public Bundle onSaveInstanceState()
    {
        Bundle bundle = super.onSaveInstanceState();
        if (bundle != null)
        {
            bundle.putSerializable(ORIENTATION_STATE_SAVE_STATE_KEY, mCurrentOrientationState);

            bundle.putBoolean(ORIENTATION_MODFY_SAVE_STATE_KEY, mAlllowOrientationModify);
        }
        return bundle;
    }

    private void initDialog()
    {
        mHandler = new Handler();
        initRootView();
        initVideoWebView();
        initBlackCurtain();
        initTopBar();
        initLoadingView();
    }

    private void initBlackCurtain()
    {
        mBlackCurtain = new View(getContext());
        mBlackCurtain.setBackgroundColor(Color.BLACK);
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT);
        mCustomViewContainer.addView(mBlackCurtain, params);
    }

    private void initLoadingView()
    {
        mLoadingProgressView = new LoadingProgressView(getContext());
        mLoadingProgressView.setCircleColor(Color.parseColor("#ffffff"));
        mLoadingProgressView.stopLoadingAnimation();

        Point point = Utils.getScreenSize(getContext());
        int size = (int) (Math.min(point.x, point.y) * 0.25);
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(size, size);
        params.gravity = Gravity.CENTER;

        mCustomViewContainer.addView(mLoadingProgressView, params);
    }

    private void initRootView()
    {
        mCustomViewContainer = new FrameLayout(getContext());
        mCustomViewContainer.setId(Utils.generateViewId());
        mCustomViewContainer.setLayoutParams(new ViewGroup.MarginLayoutParams(ViewGroup.MarginLayoutParams.MATCH_PARENT, ViewGroup.MarginLayoutParams.MATCH_PARENT));

        mContentView = new FrameLayout(getContext());
        mContentView.setId(Utils.generateViewId());
        mContentView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT));
        mCustomViewContainer.addView(mContentView);
    }

    private void initTopBar()
    {
        mKidozTopBar = new KidozTopBar(getContext(), true);
        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.WRAP_CONTENT);
        mKidozTopBar.setKidozTopBarListener(new KidozTopBar.KidozTopBarListener()
        {
            @Override public void onExitClicked(View view)
            {
                closeDialog();
            }
        });
        mKidozTopBar.setTitle("Video Player");
        mKidozTopBar.showParentalLockIcon(false);
        mCustomViewContainer.addView(mKidozTopBar, params);
    }


    private void initVideoWebView()
    {
        mVideoLayer = new RelativeLayout(getContext());
        mVideoLayer.setId(Utils.generateViewId());
        mVideoLayer.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT));
        mCustomViewContainer.addView(mVideoLayer);

        mVideoEnabledWebView = new VideoEnabledWebView(getContext());
        if (getContext().getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE)
        {
            mVideoEnabledWebView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT, Gravity.CENTER));
        }
        else
        {
            mVideoEnabledWebView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.WRAP_CONTENT, Gravity.CENTER));
        }
        mVideoEnabledWebView.getSettings().setAppCacheEnabled(true);
        if (Build.VERSION.SDK_INT < 18)
        {
            mVideoEnabledWebView.getSettings().setAppCacheMaxSize(Long.MAX_VALUE);
        }
        mVideoEnabledWebView.getSettings().setAppCachePath(getContext().getCacheDir().toString() + "/VideoCache");
        mVideoEnabledWebView.getSettings().setCacheMode(WebSettings.LOAD_DEFAULT);
        mVideoEnabledWebView.getSettings().setJavaScriptEnabled(true);
        mVideoEnabledWebView.getSettings().setUseWideViewPort(true);
        mVideoEnabledWebView.getSettings().setLoadWithOverviewMode(true);
        mVideoEnabledWebView.getSettings().setSupportZoom(false);
        mVideoEnabledWebView.getSettings().setPluginState(WebSettings.PluginState.ON);
        mVideoEnabledWebView.addJavascriptInterface(new VideoJavaScriptInterface(), "VideoJavaScriptInterface");
        if (Build.VERSION.SDK_INT >= 11)
        {
            mVideoEnabledWebView.setLayerType(View.LAYER_TYPE_HARDWARE, null);
        }

        mVideoEnabledWebChromeClient = new VideoEnabledWebChromeClient(mContentView, mVideoLayer, null, mVideoEnabledWebView)
        {

        };
        if (Build.VERSION.SDK_INT > 16)
        {
            mVideoEnabledWebView.getSettings().setMediaPlaybackRequiresUserGesture(false);
        }
        mVideoEnabledWebView.setWebChromeClient(mVideoEnabledWebChromeClient);
        mVideoEnabledWebView.setVerticalScrollBarEnabled(false);
        mVideoEnabledWebView.setHorizontalScrollBarEnabled(false);
        mContentView.addView(mVideoEnabledWebView);

        mTransparentView = new View(getContext());
        mTransparentView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT));
        mTransparentView.setOnTouchListener(new View.OnTouchListener()
        {
            @Override public boolean onTouch(View v, MotionEvent event)
            {
                if (event.getAction() == MotionEvent.ACTION_DOWN)
                {
                    animateTopBarEnter();
                }
                return false;
            }
        });
        mCustomViewContainer.addView(mTransparentView);
    }

    @Override protected void onStop()
    {
        super.onStop();
        if (mVideoEnabledWebView != null)
        {
            mVideoEnabledWebView.stopLoading();
            mVideoEnabledWebView.removeAllViews();
            mVideoEnabledWebView.destroy();
            mVideoEnabledWebView = null;
        }
    }

    public void openDialog(ContentItem contentItem)
    {
        super.openDialog();
        mContentItem = contentItem;
        loadVideo();
        mKidozTopBar.setTitle(contentItem.getName());

        mEventID = EventManager.getInstance().activityStart(getContext(), EventManager.LOG_CRITICAL_LEVEL, "YouTube Player");
        EventManager.getInstance().logEventWithDuration(getContext(), EventManager.LOG_NORMAL_LEVEL, 1, EventManager.LOG_CLOSE_RESOURCE_LEVEL, EventParameters.CATEGORY_VIDEO_PLAYER, EventParameters.ACTION_VIDEO_VIEW, contentItem.getName(), contentItem.getId());
    }

    private void loadVideo()
    {
        mLoadingProgressView.startLoadingAnimation();
        if (mVideoEnabledWebView != null)
        {
            mVideoEnabledWebView.loadUrl("http://d28lrrc51wcjkk.cloudfront.net/video_player/youtube_js.html?VideoID=" + YouTubeUtils.extractYouTubeVideoIDString(mContentItem.getData()) + "&VideoSource=" + ContentType.getValueFromType(mContentItem.getContentType()));
        }
        mHandler.postDelayed(new Runnable()
        {
            @Override public void run()
            {
                animateTopBarExit();
            }
        }, TOP_BAR_ANIMATION_DELAY);
    }

    @Override public void closeDialog()
    {
        EventManager.getInstance().activityEnd(getContext(), EventManager.LOG_CRITICAL_LEVEL, mEventID);
        EventManager.getInstance().logEventWithDurationEnd(getContext(), 1);

        super.closeDialog();
    }

    private void animateTopBarEnter()
    {
        if (mIsAnimationRunning == false && mIsTopBarHidden == true && getContext().getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE)
        {
            mIsAnimationRunning = true;
            mHandler.removeCallbacks(null);
            GenAnimator.showTopBar(mKidozTopBar, mKidozTopBar.getHeight(), TOP_BAR_ANIMATION_DURATION, new DecelerateInterpolator(), new Animator.AnimatorListener()
            {
                @Override public void onAnimationStart(Animator animation)
                {

                }

                @Override public void onAnimationEnd(Animator animation)
                {
                    mIsAnimationRunning = false;
                    mIsTopBarHidden = false;
                    mHandler.postDelayed(new Runnable()
                    {
                        @Override public void run()
                        {
                            animateTopBarExit();
                        }
                    }, TOP_BAR_ANIMATION_DELAY);
                }

                @Override public void onAnimationCancel(Animator animation)
                {

                }

                @Override public void onAnimationRepeat(Animator animation)
                {

                }
            });
        }
    }

    private void animateTopBarExit()
    {
        if (mIsAnimationRunning == false && mIsTopBarHidden == false && getContext().getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE)
        {
            mIsAnimationRunning = true;
            mHandler.removeCallbacks(null);
            GenAnimator.hideTopBar(mKidozTopBar, -mKidozTopBar.getHeight(), TOP_BAR_ANIMATION_DURATION, new AccelerateInterpolator(), new Animator.AnimatorListener()
            {
                @Override public void onAnimationStart(Animator animation)
                {

                }

                @Override public void onAnimationEnd(Animator animation)
                {
                    mIsAnimationRunning = false;
                    mIsTopBarHidden = true;
                }

                @Override public void onAnimationCancel(Animator animation)
                {

                }

                @Override public void onAnimationRepeat(Animator animation)
                {

                }
            });
        }
    }

    private void resetTopBarAnimation()
    {
        if (mIsTopBarHidden == true)
        {
            mHandler.removeCallbacks(null);
            GenAnimator.showTopBar(mKidozTopBar, mKidozTopBar.getHeight(), 0, new DecelerateInterpolator(), new Animator.AnimatorListener()
            {
                @Override public void onAnimationStart(Animator animation)
                {

                }

                @Override public void onAnimationEnd(Animator animation)
                {
                    mIsTopBarHidden = false;
                }

                @Override public void onAnimationCancel(Animator animation)
                {

                }

                @Override public void onAnimationRepeat(Animator animation)
                {

                }
            });
        }
    }

    @Override protected void onDialogOrientationChange(boolean isPortrait)
    {
        if (isPortrait == true)
        {
            mVideoEnabledWebView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.WRAP_CONTENT, Gravity.CENTER));
            resetTopBarAnimation();
        }
        else
        {
            mVideoEnabledWebView.setLayoutParams(new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT, Gravity.CENTER));
            mHandler.postDelayed(new Runnable()
            {
                @Override public void run()
                {
                    animateTopBarExit();
                }
            }, TOP_BAR_ANIMATION_DELAY);
        }
        mVideoEnabledWebView.invalidate();
    }

    @Override protected void onDialogVisibilityChange(boolean isVisible)
    {
        if (isVisible == true)
        {
            mVideoEnabledWebView.loadUrl("javascript:resumeYoutube()");
        }
        else
        {
            mVideoEnabledWebView.loadUrl("javascript:pauseYoutube()");
        }
    }

    class VideoJavaScriptInterface
    {
        @android.webkit.JavascriptInterface public void onVideoReady()
        {

        }

        @android.webkit.JavascriptInterface public void onError(final int errorCode)
        {
            mHandler.post(new Runnable()
            {
                @Override public void run()
                {
                    mLoadingProgressView.stopLoadingAnimation();
                    com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>Error loading video, errorCode = " + String.valueOf(errorCode));
                }
            });
        }

        @android.webkit.JavascriptInterface public void onStateChange(final int state)
        {
            SDKLogger.printDebbugLog(TAG, ">>>State = " + state);
            switch (state)
            {
                case 1:
                {
                    //Video is playing
                    mHandler.postDelayed(new Runnable()
                    {
                        @Override public void run()
                        {
                            mLoadingProgressView.stopLoadingAnimation();
                            mBlackCurtain.setVisibility(View.GONE);
                        }
                    }, BLACK_CURTAIN_HIDE_DELAY);
                    break;
                }
                case 2:
                {
                    break;
                }
                default:
                {
                    //Video is not playing
                    mHandler.post(new Runnable()
                    {
                        @Override public void run()
                        {
                            mBlackCurtain.setVisibility(View.VISIBLE);
                        }
                    });
                }
            }
        }
    }

}
