package com.kidoz.sdk.api;


import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;
import android.view.WindowManager;

import com.kidoz.events.DeviceUtils;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.general.assets_handling.ImageAssetsUtils;
import com.kidoz.sdk.api.general.assets_handling.SoundAssetsUtils;
import com.kidoz.sdk.api.general.database.DatabaseManager;
import com.kidoz.sdk.api.general.utils.PropertiesObj;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.SdkSoundManager;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;

import org.json.JSONObject;

public class KidozSDK
{
    private static final String TAG = KidozSDK.class.getSimpleName();

    private static boolean isInitialized = false;
    private static String mPublicherKey = null;
    private static String mSecurityKey = null;

    private static boolean isDeveloperLoggingON = false;

    /**
     * Initialize Kidoz SDK
     *
     * @param context      activity context
     * @param publicher_id valid publisher id
     * @param token        valid security token
     * @throws RuntimeException in case of invalid or missing publisher_id or security token
     */
    public static void initialize(Context context, String publicher_id, String token)
    {
        try
        {
            DeviceUtils.getGoogleAdvertisingID(context);
            SharedPreferences eventsPref = context.getSharedPreferences(EventManager.EVENTS_SHARED_PREFERENCES_KEY, 0);
            long sessionID = eventsPref.getLong(EventManager.EVENTS_LAST_SESSION_ID_KEY, 0);
            sessionID += 1;
            SharedPreferences.Editor editor = eventsPref.edit();
            editor.putLong(EventManager.EVENTS_LAST_SESSION_ID_KEY, sessionID);
            editor.commit();

            SdkSoundManager.initSoundPool();

            SharedPreferencesUtils.saveSharedPreferencesData(context, "PUBLISHER_ID", publicher_id);
            JSONObject eventParams = new JSONObject();
            eventParams.put(EventParameters.DEVELOPER_ID, publicher_id);
            eventParams.put(EventManager.LOG_LEVEL_KEY, EventManager.LOG_NORMAL_LEVEL);
            eventParams.put(EventParameters.SESSION_ID, sessionID);
            EventManager.getInstance().init(context, eventParams);
            EventManager.getInstance().setSyncAlarm(context);
            EventManager.getInstance().logEvent(context, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_SDK_INIT, publicher_id);
            EventManager.getInstance().logEvent(context, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SESSION, EventParameters.ACTION_SESSION_START, publicher_id);
            EventManager.getInstance().startSync(context);
            initFrameRate(context);
        }
        catch (Exception ex)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to create device params: " + ex.getMessage());
        }
        if (publicher_id == null || token == null)
        {
            throw new RuntimeException("Missing valid publisher id or security token!");
        }
        else
        {
            mPublicherKey = publicher_id;
            mSecurityKey = token;
            SdkAPIManager.init(context, publicher_id, token,isDeveloperLoggingON);
            isInitialized = true;
        }
        validateSDKConfiguration(context);

        DeviceUtils.generateGoogleAdvertisingIDInBackground(context);

        if(isDeveloperLoggingON) {
            Log.d(SDKLogger.GENERAL_TAG,"Kidoz SDK has been successfully Initialized !");
        }
    }

    private static SdkAPIManager getApiManager(Context context)
    {
        SdkAPIManager.init(context, mPublicherKey, mSecurityKey,isDeveloperLoggingON);
        return SdkAPIManager.getSdkApiInstance(context);
    }

    /**
     * Check if Kidoz SDK has been initialised
     *
     * @return initialized
     */
    public static boolean isInitialised()
    {
        if (isInitialized == false)
        {
            Log.e("[KIDOZ SDK]", "Kidoz SDK need to be initialized !");
        }
        return isInitialized;
    }

    /**
     * Validate SDK configuration (Configuration retrieved from remote server)
     */
    private static void validateSDKConfiguration(final Context context)
    {
        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>validateSDKConfiguration");

        if (context != null)
        {
            getApiManager(context).validateSDKConfiguration(context, new ApiResultCallback<PropertiesObj>() {
                @Override
                public void onServerResult(ResultData<?> result) {
                    if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true && result.getData() != null && result.getData() instanceof PropertiesObj) {
                        final PropertiesObj serverConfig = (PropertiesObj) result.getData();
                        PropertiesObj appConfig = DatabaseManager.getInstance(context).getConfigTable().loadAppProperties();
                        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>Server config version = " + serverConfig.getConfigVersion());
                        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>App config version = " + (appConfig == null ? "NULL" : appConfig.getConfigVersion()));

                        if (appConfig == null || appConfig.getConfigVersion() < serverConfig.getConfigVersion()) {
                            com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>Going to download resources...");
                            // Need to update configuration (graphics, strings etc...)
                            getApiManager(context).getSDKResources(context, new ApiResultCallback<Boolean>() {
                                @Override
                                public void onServerResult(ResultData<?> result) {
                                    if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true && result.getData() != null && result.getData() instanceof Boolean) {
                                        if ((Boolean) result.getData() == true) {
                                            com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>All resources downloaded");
                                            SharedPreferencesUtils.removeSharedPreferencesData(context, ImageAssetsUtils.IMAGE_ASSETS_SYNCED_FLAG);
                                            SharedPreferencesUtils.removeSharedPreferencesData(context, ImageAssetsUtils.PARENTAL_LOCK_ASSETS_SYNCED_FLAG);
                                            SharedPreferencesUtils.removeSharedPreferencesData(context, SoundAssetsUtils.SOUND_ASSETS_SYNCED_FLAG);

                                            DatabaseManager.getInstance(context).getConfigTable().insertAppProperties(serverConfig);
                                        }
                                    }
                                }

                                @Override
                                public void onFailed() {

                                }
                            });
                        }
                    }
                }

                @Override
                public void onFailed() {

                }
            });
        }
    }

    private static void initFrameRate(Context context)
    {
        float frameRate = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay().getRefreshRate();
        com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>frameRate = " + String.valueOf(frameRate));
    }

    /**
     * Set developer logging enabled
     * (Must be set before initialization of the SDK)
     *
     * @param enabled
     */
    public static void setLoggingEnabled(boolean enabled) {
        isDeveloperLoggingON = enabled;
    }
}
