package com.kidoz.sdk.api;

import android.content.Context;
import android.graphics.Point;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;
import android.widget.RelativeLayout;

import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.ui_views.AnimatableGifImageView;
import com.kidoz.sdk.api.ui_views.custom_drawables.KidozLogoDrawable;

public class FeedButton extends RelativeLayout {

    private static final String TAG = FeedButton.class.getSimpleName();

    private static final float BUTTON_SIZE_AS_SCREEN_PORTION = 0.25f;
    private AnimatableGifImageView mAnimatableGifImageView;
    private FeedView mFeedView;

    private View mRootView;
    private Context mContext;

    public FeedButton(Context context) {
        super(context);
        initView(context,-1);
    }

    /**
     * Initiate button view
     *
     * @param size button size
     */
    public FeedButton(Context context,int size) {
        super(context);
        initView(context,size);
    }

    /**
     * Initiate button view
     */
    public FeedButton(Context context, AttributeSet attrs) {
        super(context, attrs);
        initView(context,-1);
    }

    public FeedButton(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initView(context,-1);
    }

    /**
     * Initiate view
     */
    private void initView(Context context,int initSize) {

        mContext = context;
        mRootView = this;

        int size = getDefaultButtonSize(context);

        if(initSize != -1) {
            size = initSize;
        }

        mAnimatableGifImageView = new AnimatableGifImageView(context);

        RelativeLayout.LayoutParams animatableImageParams = new RelativeLayout.LayoutParams(size, size);
        addView(mAnimatableGifImageView, animatableImageParams);

        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(size,size);
        setLayoutParams(params);

        KidozLogoDrawable kidozLogoDrawable = new KidozLogoDrawable();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            mAnimatableGifImageView.setBackground(kidozLogoDrawable);
        } else {
            mAnimatableGifImageView.setBackgroundDrawable(kidozLogoDrawable);
        }

        if (isInEditMode() == false) {
            mAnimatableGifImageView.setAnimatedGif(Utils.prepareCorrectUri("https://kidoz-cdn.s3.amazonaws.com/sdk/btn_animation.gif"), new AnimatableGifImageView.IOnAnimatedGifReadyListener() {
                @Override
                public void onGifReady() {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                        mAnimatableGifImageView.setBackground(null);
                    } else {
                        mAnimatableGifImageView.setBackgroundDrawable(null);
                    }
                }
            });

            initFeedView(context);
        }
    }


    /**
     * Initiate button click listener
     */
    private void initClickListener() {
        setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mFeedView != null) {
                    GenAnimator.clickItemAnimation(mRootView, 100, new GenAnimator.ViewAnimationListener() {
                        @Override
                        public void onAnimationEnd() {
                            mFeedView.showView();
                        }

                        @Override
                        public void onAnimationStart() {

                        }
                    });
                }
            }
        });
    }

    /**
     * Initiate interstitial view
     */
    private void initFeedView(Context context) {
        mFeedView = new FeedView.Builder(context).build();
        initClickListener();
    }

    /**
     * Get the inner interstitial view object
     *
     * @return interstitial view object
     */
    public FeedView getFeedView() {
        if(mFeedView == null) {
            if(mContext == null) {
                mContext = getContext();
            }
            initFeedView(mContext);
        }
        return mFeedView;
    }

    /**
     * Set button size in pixels
     *
     * @param sizePx size in pixels
     */
    public void setFeedButtonSize(int sizePx) {
        mAnimatableGifImageView.getLayoutParams().height = sizePx;
        mAnimatableGifImageView.getLayoutParams().width = sizePx;

        getLayoutParams().height = sizePx;
        getLayoutParams().width = sizePx;
    }

    /**
     * Set button size in DP
     *
     * @param sizeDp size in pixels
     */
    public void setFeedButtonSizeDp(int sizeDp) {

        int h = Utils.dpTOpx(getResources(),sizeDp);
        int w = Utils.dpTOpx(getResources(),sizeDp);

        mAnimatableGifImageView.getLayoutParams().height = h;
        mAnimatableGifImageView.getLayoutParams().width = w;

        getLayoutParams().height = h;
        getLayoutParams().width = w;
    }

    /** Get default button size
     *
     * @return button size in pixels
     */
    public static int getDefaultButtonSize(Context context) {
        Point point = Utils.getScreenSize(context);
        return (int) (Math.min(point.x, point.y) * BUTTON_SIZE_AS_SCREEN_PORTION);
    }
}
