package com.kidoz.sdk.api.ui_views;

import android.animation.Animator;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.view.View;
import android.widget.ImageView;

import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.utils.AppAssetsUtils;
import com.kidoz.sdk.api.general.utils.ScreenUtils;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.picasso_related.PicassoOk;

import java.io.File;

/**
 * Created by roni on 9/24/2015.
 */
public class ExitButton extends ImageView
{
    private final String TAG = ExitButton.class.getSimpleName();

    private boolean mIsUseBackButton;

    private File mImageFile;
    private float mScaleFactor;
    private int mWidth;
    private int mHeight;
    private int mAnimationType;

    private Paint mWhitePaint;
    private Paint mBluePaint;
    private RectF mRectF;

    private ExitButtonListener mExitButtonListener;

    public ExitButton(Context context, ExitButtonListener exitButtonListener, boolean isUseBackButton)
    {
        super(context);
        mExitButtonListener = exitButtonListener;
        mIsUseBackButton = isUseBackButton;
        setScaleType(ScaleType.FIT_XY);
        initView();
    }

    private void initView()
    {
        initImageFile();
        initImageParameters();
        initPaints();
        initRectF();
        initOnClickAnimation();
    }

    private void initOnClickAnimation()
    {
        setOnClickListener(new OnClickListener()
        {
            @Override public void onClick(View view)
            {
                if (mExitButtonListener != null)
                {
                    mExitButtonListener.onClickStarted();
                }

                switch (mAnimationType)
                {
                    case 1:
                    {
                        GenAnimator.playStandUpAnimation(view, new Animator.AnimatorListener()
                        {
                            @Override public void onAnimationStart(Animator animation)
                            {

                            }

                            @Override public void onAnimationEnd(Animator animation)
                            {
                                if (mExitButtonListener != null)
                                {
                                    mExitButtonListener.onClickEnd();
                                }
                            }

                            @Override public void onAnimationCancel(Animator animation)
                            {

                            }

                            @Override public void onAnimationRepeat(Animator animation)
                            {

                            }
                        });
                        break;
                    }
                }
            }
        });
    }

    private void initImageFile()
    {
        if (mIsUseBackButton == false)
        {
            mImageFile = AppAssetsUtils.getAssetFile(getContext(), AppAssetsUtils.APP_ASSET_TYPE.EXIT_BUTTON);
        }
        else
        {
            mImageFile = AppAssetsUtils.getAssetFile(getContext(), AppAssetsUtils.APP_ASSET_TYPE.BACK_BUTTON);
        }
    }

    private void loadImage()
    {
        if (mImageFile.exists() == true)
        {
            if (getDrawable() == null)
            {
                PicassoOk.getPicasso(getContext()).load(mImageFile).into(this);
            }
        }
    }

    private void initImageParameters()
    {
        mScaleFactor = ScreenUtils.getScaleFactor(getContext());
        mWidth = 178;
        mHeight = 128;
        mAnimationType = 1;
        if (mImageFile != null && mImageFile.exists() == true)
        {
            String width = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), AppAssetsUtils.EXIT_BUTTON_WIDTH, "178", Context.MODE_PRIVATE);
            String height = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), AppAssetsUtils.EXIT_BUTTON_HEIGHT, "128", Context.MODE_PRIVATE);
            String animationType = SharedPreferencesUtils.loadSharedPreferencesData(getContext(), AppAssetsUtils.EXIT_BUTTON_ANIMATION_TYPE, "1", Context.MODE_PRIVATE);

            try
            {
                mWidth = Integer.parseInt(width);
                mHeight = Integer.parseInt(height);
                mAnimationType = Integer.parseInt(animationType);
            }
            catch (Exception ex)
            {
                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to init image parameters: " + ex.getMessage());
            }
        }
        mWidth *= mScaleFactor;
        mHeight *= mScaleFactor;
    }

    private void initPaints()
    {
        mWhitePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mWhitePaint.setStyle(Paint.Style.FILL_AND_STROKE);
        mWhitePaint.setColor(Color.parseColor("#ffffff"));

        mBluePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mBluePaint.setStyle(Paint.Style.FILL_AND_STROKE);
        mBluePaint.setColor(Color.parseColor("#04a0e1"));

        setLayerType(LAYER_TYPE_SOFTWARE, mWhitePaint);
    }

    private void initRectF()
    {
        float left = (65f / 178f) * mWidth;
        float top = (56f / 128f) * mHeight;
        float right = (124f / 178f) * mWidth;
        float bottom = (72f / 128f) * mHeight;

        if (mIsUseBackButton == true)
        {
            left = (73f / 178f) * mWidth;
            top = (59f / 128f) * mHeight;
            right = (114f / 178f) * mWidth;
            bottom = (69f / 128f) * mHeight;
        }
        mRectF = new RectF(left, top, right, bottom);
    }

    @Override protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec)
    {
        setMeasuredDimension(mWidth, mHeight);
    }

    @Override protected void onDraw(Canvas canvas)
    {
        super.onDraw(canvas);
        if (mImageFile.exists() == false)
        {
            // Draw left line
            float left = (33f / 178f) * mWidth;
            float top = (0f / 128f) * mHeight;
            float right = (35f / 178f) * mWidth;
            float bottom = (49f / 128f) * mHeight;
            canvas.drawRect(left, top, right, bottom, mWhitePaint);

            // Draw right line
            left = (162f / 178f) * mWidth;
            top = (0f / 128f) * mHeight;
            right = (164f / 178f) * mWidth;
            bottom = (82f / 128f) * mHeight;
            canvas.drawRect(left, top, right, bottom, mWhitePaint);

            // Draw bottom left circle
            float radius = (42f / 128f) * mHeight;
            float centerX = (42f / 178f) * mWidth;
            float centerY = (86f / 128f) * mHeight;
            canvas.drawCircle(centerX, centerY, radius, mWhitePaint);

            // Draw the bottom right circle
            radius = (26f / 128f) * mHeight;
            centerX = (152f / 178f) * mWidth;
            centerY = (102f / 128f) * mHeight;
            canvas.drawCircle(centerX, centerY, radius, mWhitePaint);

            // Draw the background circle
            radius = (49.4f / 128f) * mHeight;
            centerX = (101.41f / 178f) * mWidth;
            centerY = (66.64f / 128f) * mHeight;
            canvas.drawCircle(centerX, centerY, radius, mWhitePaint);

            // Draw the background rectangle
            left = (42f / 178f) * mWidth;
            top = (101f / 128f) * mHeight;
            right = (153f / 178f) * mWidth;
            bottom = (128f / 128f) * mHeight;
            canvas.drawRect(left, top, right, bottom, mWhitePaint);

            // Draw the x circle
            mWhitePaint.setShadowLayer(2.0f, 0.0f, 3.0f, 0xFF7c7c7c);
            radius = (42f / 128f) * mHeight;
            centerX = (93f / 178f) * mWidth;
            centerY = (64f / 128f) * mHeight;
            canvas.save();
            canvas.rotate(45f, centerX, centerY);
            canvas.drawCircle(centerX, centerY, radius, mWhitePaint);
            mWhitePaint.setShadowLayer(0.0f, 0.0f, 0.0f, 0x00000000);
            canvas.restore();

            if (mIsUseBackButton == false)
            {
                // Draw the x shape
                radius = (7f / 178f) * mWidth;
                centerX = (94.52f / 178f) * mWidth;
                centerY = (64.02f / 128f) * mHeight;
                canvas.save();
                canvas.rotate(45f, centerX, centerY);
                canvas.drawRoundRect(mRectF, radius, radius, mBluePaint);
                canvas.restore();
                canvas.save();
                canvas.rotate(-45f, centerX, centerY);
                canvas.drawRoundRect(mRectF, radius, radius, mBluePaint);
                canvas.restore();
            }
            else
            {
                // Draw the back shape
                radius = (5f / 178f) * mWidth;
                centerX = (73f / 178f) * mWidth;
                centerY = (64f / 128f) * mHeight;
                canvas.rotate(30f, centerX, centerY);
                canvas.drawRoundRect(mRectF, radius, radius, mBluePaint);
                canvas.restore();
                canvas.save();
                canvas.rotate(-30f, centerX, centerY);
                canvas.drawRoundRect(mRectF, radius, radius, mBluePaint);
                canvas.restore();

                radius = (5f / 128f) * mHeight;
                centerX = (76f / 178f) * mWidth;
                centerY = (64f / 128f) * mHeight;
                canvas.drawCircle(centerX, centerY, radius, mBluePaint);
            }
        }
        else
        {
            loadImage();
        }
    }

    public interface ExitButtonListener
    {
        public void onClickStarted();

        public void onClickEnd();
    }
}
