package com.kidoz.sdk.api;

import android.app.Activity;
import android.content.Context;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import com.kidoz.sdk.api.interfaces.IOnFeedViewEventListener;
import com.kidoz.sdk.api.ui_views.panel_view.HANDLE_POSITION;
import com.kidoz.sdk.api.ui_views.panel_view.PANEL_TYPE;
import com.unity3d.player.UnityPlayer;

/**
 * Helper class for Unity vs android comunication handling
 */
public final class KidozUnityBridge {

    private static KidozUnityBridge mKidozUnity;

    private Context mContext;
    private FeedView mFeedView;
    private PanelView mPanelView;
    private FeedButton mFeedButton;

    private FrameLayout mContainer;
    private boolean isContianerAdded = false;

    private String mGameObject;
    private String mShowMethodName;
    private String mDismissMethodName;

    public KidozUnityBridge(Context context) {
        mContext = context;
        mContainer = new FrameLayout(mContext);
    }

    public static KidozUnityBridge getInstance(Context context) {
        if (mKidozUnity == null) {
            mKidozUnity = new KidozUnityBridge(context);
        }
        return mKidozUnity;
    }

    /**
     * Add Feed Button to view (The button will be added with default size)
     *
     * @param x the x coordinate of the button
     * @param y the y coordinate of the button
     */
    public void addFeedButton(final int x, final int y) {
        addFeedButton(x, y,-1);
    }

    /**
     * Add Feed Button to view
     *
     * @param x the x coordinate of the button
     * @param y the y coordinate of the button
     * @param size button size
     */
    public void addFeedButton(final int x, final int y,int size) {
        addContainerIfNedded();

        if (mFeedButton == null) {
            if(size == -1) {
                mFeedButton = new FeedButton(mContext);
            } else {
                mFeedButton = new FeedButton(mContext,size);
            }

            mFeedButton.getFeedView().setOnFeedViewEventListener(new IOnFeedViewEventListener() {
                @Override
                public void onDismissView() {
                    if (mGameObject != null && mDismissMethodName != null) {
                        UnityPlayer.UnitySendMessage(mGameObject, mDismissMethodName, "Show");
                    }
                }

                @Override
                public void onReadyToShow() {
                    if (mGameObject != null && mShowMethodName != null) {
                        UnityPlayer.UnitySendMessage(mGameObject, mShowMethodName, "Dismiss");
                    }
                }
            });
        }

        if (mContext instanceof Activity) {
            ((Activity) mContext).runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (isChildViewExistsInContainer(mFeedButton) == false) {
                        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.WRAP_CONTENT);
                        params.setMargins(x, y, 0, 0);
                        mContainer.addView(mFeedButton, params);
                    }
                }
            });
        }
    }

    /**
     * Remove feed button from view
     */
    public void removeFeedButton() {
        ((Activity) mContext).runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (mFeedButton != null) {
                    mContainer.removeView(mFeedButton);
                }
            }
        });
    }

    /**
     * Change feed button visibility state
     */
    public void changeFeedButtonVisibility(final boolean visible) {
        ((Activity) mContext).runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (mFeedButton != null) {
                    if (visible) {
                        mFeedButton.setVisibility(View.VISIBLE);
                    } else {
                        mFeedButton.setVisibility(View.INVISIBLE);
                    }
                }
            }
        });
    }

    /**
     * Check if main container has this view as a child
     */
    private boolean isChildViewExistsInContainer(View view) {
        boolean exists = false;
        if (mContainer != null && mContainer.getChildCount() > 0) {
            for (int i = 0; i < mContainer.getChildCount(); i++) {
                if (mContainer.getChildAt(i) == view) {
                    exists = true;
                    break;
                }
            }
        }
        return exists;
    }

    /**
     * Adds views holder container to a main activity view
     */
    private void addContainerIfNedded() {
        ((Activity) mContext).runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (isContianerAdded == false) {
                    if (mContext instanceof Activity) {
                        ((Activity) mContext).addContentView(mContainer, new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT));
                        isContianerAdded = true;
                    }
                }
            }
        });
    }

    /**
     * Show feed view
     */
    public void showFeedView() {
        if (mFeedView == null) {
            mFeedView = new FeedView.Builder(mContext).build();
            mFeedView.setOnFeedViewEventListener(new IOnFeedViewEventListener() {
                @Override
                public void onDismissView() {
                    if (mGameObject != null && mDismissMethodName != null) {
                        UnityPlayer.UnitySendMessage(mGameObject, mDismissMethodName, "Dismiss");
                    }
                }

                @Override
                public void onReadyToShow() {
                    if (mGameObject != null && mShowMethodName != null) {
                        UnityPlayer.UnitySendMessage(mGameObject, mShowMethodName, "Show");
                    }
                }
            });
        }

        ((Activity) mContext).runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (mFeedView != null) {
                    mFeedView.showView();
                }
            }
        });
    }

    /**
     * Dismiss feed view
     */
    public void dismissFeedView() {
        ((Activity) mContext).runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (mFeedView != null) {
                    mFeedView.dismissView();
                }
            }
        });
    }

    /**
     * Set callback for listening feed view dismiss and show events
     */
    public void setFeedViewEventListeners(String gameObject, String showMethodName, String dismissMethodName) {
        mGameObject = gameObject;
        mShowMethodName = showMethodName;
        mDismissMethodName = dismissMethodName;
    }

    /**
     * Add PanelView to view
     */
    public void addPanelView(PANEL_TYPE panel_type,HANDLE_POSITION handle_position) {

        addContainerIfNedded();

        if (mPanelView == null) {
            mPanelView = new PanelView(mContext);
        }

        mPanelView.setPanelConfiguration(panel_type,handle_position);

        if(mContext instanceof Activity) {
            ((Activity) mContext).runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if(isChildViewExistsInContainer(mPanelView) == false) {
                        FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT);
                        mContainer.addView(mFeedButton, params);
                    }
                }
            });
        }
    }


    /**
     * Change feed button visibility state
     */
    public void changePanelVisibility(final boolean visible) {
        ((Activity) mContext).runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (mPanelView != null) {
                    if (visible) {
                        mPanelView.setVisibility(View.VISIBLE);
                    } else {
                        mPanelView.setVisibility(View.INVISIBLE);
                    }
                }
            }
        });
    }

    /**
     * Remove feed button from view
     */
    public void removePanelView() {
        ((Activity) mContext).runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (mPanelView != null) {
                    mContainer.removeView(mPanelView);
                }
            }
        });
    }


    /**
     * Get feed button default size
     *
     * @return default button size in pixels
     */
    public int getFeedButtonDefaultSize(Context context) {
        return FeedButton.getDefaultButtonSize(context);
    }
}
