package com.kidoz.events;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;

import org.json.JSONObject;

import java.util.ArrayList;

/**
 * Created by orikam on 8/30/15.
 */
public class EventsTable
{
    private static final String TAG = EventsTable.class.getName();
    private static final String TABLE_NAME = "KIDOZ_EVENTS_TABLE";
    private static final String COLUMN_ROW_ID = "COLUMN_ROW_ID";
    private static final String COLUMN_JSON_OBJECT = "COLUMN_JSON_OBJECT";

    public static final String CREATE_KIDOZ_EVENTS_TABLE = "CREATE TABLE " + TABLE_NAME + "(" + COLUMN_ROW_ID + " TEXT PRIMARY KEY, " + COLUMN_JSON_OBJECT + " TEXT)";
    private EventLogDatabaseManager mDBmanager;

    private final Object mSyncObject = new Object();

    public EventsTable(EventLogDatabaseManager databaseManager)
    {
        mDBmanager = databaseManager;
    }

    public static void onCreate(SQLiteDatabase database)
    {
        database.execSQL(CREATE_KIDOZ_EVENTS_TABLE);
    }

    public static void onUpgrade(SQLiteDatabase database, int oldVersion, int newVersion)
    {
        if (newVersion > oldVersion)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog(TAG, "KidozEventsTable onUpgrade: oldVersion = " + oldVersion + " and newVersion = " + newVersion);
            if (oldVersion < 2)
            {
                onCreate(database);
            }
        }
    }

    public synchronized void inserLogEventToDB(Event Event)
    {
        synchronized (mSyncObject)
        {
            if (Event != null)
            {
                int count = 0;
                int rowID = -1;
                SQLiteDatabase db = null;
                try
                {
                    db = mDBmanager.openDatabase();
                    db.beginTransaction();

                    ContentValues values = new ContentValues();
                    values.put(COLUMN_ROW_ID, EncoderUtils.encryptWithMD5(Event.getJSONObject().toString()));
                    values.put(COLUMN_JSON_OBJECT, Event.getJSONObject().toString());
                    rowID = (int) db.insertWithOnConflict(TABLE_NAME, null, values, SQLiteDatabase.CONFLICT_IGNORE);
                    if (rowID != -1)
                    {
                        count++;
                    }
                    db.yieldIfContendedSafely();
                    db.setTransactionSuccessful();
                }
                catch (Exception e) {

                }
                finally
                {
                    if(db != null)
                        db.endTransaction();
                    mDBmanager.closeDatabase();
                }
                com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog("Insert [" + count + "] log events to local db");
                com.kidoz.sdk.api.general.utils.SDKLogger.printDebbugLog(TAG, ">>>>> New log Event: " + Event.getJSONObject().toString());
            }
        }
    }

    /**
     * This method return arraylist of type KidozEvent. the number of items is limit according to given int. this method always return the 'limit' first rows.
     *
     * @param limit
     */
    public synchronized ArrayList<Event> getEvents(int limit)
    {
        int count = 0;
        ArrayList<Event> result = new ArrayList<Event>();
        synchronized (mSyncObject)
        {
            try {
                SQLiteDatabase db = mDBmanager.openDatabase();
                Cursor cursor = db.query(false, TABLE_NAME, null, null, null, null, null, null, String.valueOf(limit));
                if (cursor != null)
                {
                    count = cursor.getCount();
                    if (cursor.getCount() > 0)
                    {
                        cursor.moveToFirst();
                        do
                        {
                            Event nextEvent = new Event();
                            JSONObject nextJSONObject = new JSONObject();
                            try
                            {
                                nextJSONObject = new JSONObject(cursor.getString(cursor.getColumnIndex(COLUMN_JSON_OBJECT)));
                            }
                            catch (Exception ex)
                            {
                                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(TAG, "Error when trying to load json object: " + ex.getMessage());
                            }
                            nextEvent.setRowID(cursor.getString(cursor.getColumnIndex(COLUMN_ROW_ID)));
                            nextEvent.setJsonObject(nextJSONObject);
                            result.add(nextEvent);
                        }
                        while (cursor.moveToNext());
                    }
                    cursor.close();
                }
            } catch (Exception e) {

            }
            finally {
                mDBmanager.closeDatabase();
            }
            com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog(">>Get [" + count + "] oldest log events from local db, Limit = " + limit);
        }
        return result;
    }

    public synchronized void removeLogEventsFromDB(ArrayList<Event> eventArrayList)
    {
        synchronized (mSyncObject)
        {
            int count = 0;
            if (eventArrayList != null)
            {
                SQLiteDatabase db = null;
                try
                {
                    db = mDBmanager.openDatabase();
                    db.beginTransaction();
                    for (Event nextEvent : eventArrayList)
                    {
                        int res = db.delete(TABLE_NAME, COLUMN_ROW_ID + "=?", new String[] { "" + nextEvent.getRowID() });
                        if (res != -1)
                        {
                            count++;
                        }
                    }
                    db.setTransactionSuccessful();
                }
                finally
                {
                    if(db != null)
                        db.endTransaction();
                    mDBmanager.closeDatabase();
                }
                com.kidoz.sdk.api.general.utils.SDKLogger.printInfoLog("Remove [" + count + "] oldest log events from local db");
            }
        }
    }

    public synchronized boolean isDBempty()
    {
        boolean result = true;
        synchronized (mSyncObject)
        {

            try {
                SQLiteDatabase db = mDBmanager.openDatabase();
                Cursor cursor = db.rawQuery("SELECT count(*) FROM " + TABLE_NAME, null);
                if (cursor != null)
                {
                    if (cursor.moveToFirst())
                    {
                        if (cursor.getInt(0) > 0)
                        {
                            result = false;
                        }
                    }
                    cursor.close();
                }

            } catch (Exception e) {

            }
            finally
            {
                mDBmanager.closeDatabase();
            }
        }
        return result;
    }
}
