package com.kidoz.sdk.api.structure;

import android.content.Context;
import android.graphics.Bitmap;

import com.kidoz.sdk.api.dialogs.ParentalLockDialog;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.picasso_related.PicassoOk;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;

/**
 * Created by roni on 8/23/2015.
 */
public class SDKConfigurationData
{
    private static final String TAG = SDKConfigurationData.class.getSimpleName();
    private final String SDK_CONFIGURATION_VERSION = "SDK_CONFIGURATION_VERSION";

    private String mConfigurationVersion;
    private ParentalLockData mParentalLockData;

    public SDKConfigurationData(String data)
    {
        if (data != null)
        {
            try
            {
                JSONObject resultJSONObject = new JSONObject(data);
                if (resultJSONObject.has("data") == true && resultJSONObject.isNull("data") == false)
                {
                    JSONObject jSONObject = resultJSONObject.getJSONObject("data");
                    if (jSONObject != null)
                    {
                        if (jSONObject.has("configuration_version") == true && jSONObject.isNull("configuration_version") == false)
                        {
                            mConfigurationVersion = jSONObject.getString("configuration_version");
                        }
                        if (jSONObject.has("parental_lock_data") == true && jSONObject.isNull("parental_lock_data") == false)
                        {
                            mParentalLockData = new ParentalLockData(jSONObject.getJSONObject("parental_lock_data"));
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                SDKLogger.printErrorLog(TAG, "Error creating sdk configuration data: " + ex.getMessage());
            }
        }
    }

    public String getConfigurationVersion()
    {
        return mConfigurationVersion;
    }

    public void setConfigurationVersion(String configurationVersion)
    {
        mConfigurationVersion = configurationVersion;
    }

    public ParentalLockData getParentalLockData()
    {
        return mParentalLockData;
    }


    public static void saveConfigurationFile(Context context, JSONArray jSONArray)
    {
        String configurationFileName = "ConfigurationFile";
        try
        {
            File dir = context.getFilesDir();
            File file = new File(dir, configurationFileName);
            if (file.exists() == true)
            {
                file.delete();
            }

            FileOutputStream fileOutputStream = context.openFileOutput(configurationFileName, Context.MODE_PRIVATE);
            fileOutputStream.write(jSONArray.toString().getBytes());
            fileOutputStream.close();
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to create configuration file: " + ex.getMessage());
        }
    }

    public static void saveLockIconToFS(Context context, Bitmap bitmap, String lockIconName)
    {
        try
        {
            FileOutputStream fileOutputStream = context.openFileOutput(lockIconName, Context.MODE_PRIVATE);
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, fileOutputStream);
            fileOutputStream.close();
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to create configuration file: " + ex.getMessage());
        }
    }


    public static JSONArray loadConfigurationFile(Context context)
    {
        JSONArray result = null;
        String configurationFileName = "ConfigurationFile";
        StringBuffer stringBuffer = new StringBuffer();
        try
        {
            FileInputStream fileInputStream = context.openFileInput(configurationFileName);
            InputStreamReader inputStreamReader = new InputStreamReader(fileInputStream);
            BufferedReader bufferedReader = new BufferedReader(inputStreamReader);
            String nextLine = bufferedReader.readLine();
            while (nextLine != null)
            {
                stringBuffer.append(nextLine);
                nextLine = bufferedReader.readLine();
            }
            inputStreamReader.close();
            result = new JSONArray(stringBuffer.toString());
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to load configuration file: " + ex.getMessage());
        }
        return result;
    }

    public void updateConfigurationIfNeeded(Context context)
    {
        try
        {
            // 1. Create saved configuration version if any.
            String savedConfigurationVersionAsString = SharedPreferencesUtils.loadSharedPreferencesData(context, SDK_CONFIGURATION_VERSION);
            int savedConfigurationVersion = -1;
            int newConfigurationVersion = 0;

            if (savedConfigurationVersionAsString != null)
            {
                savedConfigurationVersion = Integer.parseInt(savedConfigurationVersionAsString);
            }
            if (mConfigurationVersion != null)
            {
                newConfigurationVersion = Integer.parseInt(mConfigurationVersion);
            }
            // 2. If new configuration is higher then saved version - need to update.
            if (newConfigurationVersion > savedConfigurationVersion)
            {
                // 2.0 Save the configuration version.
                SharedPreferencesUtils.saveSharedPreferencesData(context, SDK_CONFIGURATION_VERSION, mConfigurationVersion);

                // 2.1 Update parental lock status.
                if (mParentalLockData.getIsActivated() == true)
                {
                    SharedPreferencesUtils.saveSharedPreferencesData(context, ParentalLockDialog.PARENTAL_LOCK_ACTIVE_KEY, ParentalLockDialog.PARENTAL_LOCK_ACTIVE_KEY);
                }
                else
                {
                    SharedPreferencesUtils.removeSharedPreferencesData(context, ParentalLockDialog.PARENTAL_LOCK_ACTIVE_KEY);
                }
                // 2.2 Save the new lock icons.
                ArrayList<LockIconData> lockIconArrayList = mParentalLockData.getLockIconArrayList();
                if (lockIconArrayList != null && lockIconArrayList.isEmpty() == false)
                {
                    JSONArray jSONArray = new JSONArray();
                    for (LockIconData nextLockIconData : lockIconArrayList)
                    {
                        // 2.2.1 Save the file data to FS.
                        JSONObject nextJSONObject = new JSONObject();
                        nextJSONObject.put("LockIconURL", nextLockIconData.getIconURL());
                        nextJSONObject.put("LockIconName", nextLockIconData.getName());
                        nextJSONObject.put("LockIconOnFS", "Lock_Icon_" + nextLockIconData.getName() + ".jpg");
                        jSONArray.put(nextJSONObject);

                        // 2.2.2 Download the actual image and save to FS.
                        Bitmap nextBitmap = PicassoOk.getPicasso(context).load(nextLockIconData.getIconURL()).get();
                        saveLockIconToFS(context, nextBitmap, "Lock_Icon_" + nextLockIconData.getName() + ".jpg");
                    }
                    SDKConfigurationData.saveConfigurationFile(context, jSONArray);
                }
            }
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to updateConfigurationIfNeeded: " + ex.getMessage());
        }
    }

    public static void prepareLockIcons(Context context)
    {
        JSONArray jSONArray = SDKConfigurationData.loadConfigurationFile(context);
        if (jSONArray != null && jSONArray.length() > 0)
        {
            for (int lockIconIndex = 0; lockIconIndex < jSONArray.length(); lockIconIndex++)
            {
                try
                {
                    JSONObject nextJSONObject = jSONArray.getJSONObject(lockIconIndex);
                    LockIconData nextLockIconData = new LockIconData(context, nextJSONObject);
                    File dir = context.getFilesDir();
                    File imageFile = new File(dir, nextLockIconData.getIconURL());
                    PicassoOk.getPicasso(context).load(imageFile).fetch();
                }
                catch (Exception ex)
                {
                    SDKLogger.printErrorLog(TAG, "Error when trying to prepareLockIcons:" + ex.getMessage());
                }
            }
        }
    }
}
