package com.kidoz.sdk.api.server_connect;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Environment;

import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.squareup.okhttp.Cache;
import com.squareup.okhttp.Call;
import com.squareup.okhttp.Callback;
import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.RequestBody;
import com.squareup.okhttp.Response;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.CookieHandler;
import java.net.CookieManager;
import java.net.CookiePolicy;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.TimeUnit;


/**
 * {@code BaseConnectionClient} Responsible for making a connection to a remote server via different approach methods SOAP and REST
 */
@SuppressLint("NewApi") public class BaseConnectionClient
{
    public static boolean isNetworkAvailable(Context context)
    {
        ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
        return activeNetworkInfo != null && activeNetworkInfo.isConnected();
    }

    public static int getConnectionType(Context context)
    {
        int reuslt = ConnectivityManager.TYPE_WIFI;
        if (context != null)
        {
            ConnectivityManager connManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo nInfo = connManager.getActiveNetworkInfo();
            if (nInfo != null)
            {
                if (nInfo.isAvailable())
                {
                    reuslt = nInfo.getType();
                }
            }
        }
        return reuslt;
    }

    // Main server access Url
    protected static String MAIN_SERVER_URL = "";
    protected static String KIDOZ_ANALITYCS_SERVER_URL = "";

    protected final static String TAG = BaseConnectionClient.class.getSimpleName();

    protected enum CONNECTION_TYPE
    {
        POST, GET
    }

    private static final OkHttpClient client = new OkHttpClient();
    private static final MediaType JSON = MediaType.parse("application/x-www-form-urlencoded; charset=utf-8");

    /**
     * Initiate http conn ection client
     *
     * @param enableCache enable http cache
     */
    protected static void initConnectionClient(boolean enableCache)
    {
        client.setConnectTimeout(10, TimeUnit.SECONDS);
        client.setWriteTimeout(10, TimeUnit.SECONDS);
        client.setReadTimeout(20, TimeUnit.SECONDS);

        if (enableCache == true)
        {
            File file = Environment.getDownloadCacheDirectory();
            if (file != null)
            {
                file = new File(file.getAbsolutePath() + File.separator + "SDKhttpCacheDir");
                if (file != null)
                {
                    if (file.exists() == false)
                    {
                        file.mkdirs();
                    }

                    int cacheSize = 10 * 1024 * 1024; // 10 MiB
                    Cache cache = new Cache(file, cacheSize);
                    client.setCache(cache);
                }
            }
        }
        initCookieManager();
    }


    /**
     * Initiate cookie manager
     */
    private static void initCookieManager()
    {
        CookieManager cookieManager = new CookieManager();
        cookieManager.setCookiePolicy(CookiePolicy.ACCEPT_ALL);
        CookieHandler.setDefault(cookieManager);
    }


    /**
     * Make Post connection // Uses MORE preferable HTTP connection library
     *
     * @param url           url
     * @param contentValues http parameters
     * @param requestTag    connection request tag
     */
    public static Call makePostConnection(String url, ContentValues contentValues, String requestTag) throws IOException
    {
        // 'method_name' => 'validateSDKConfiguration',\n  'request_parameters' => '{ "version": "1", "publisher_id": 1, "package_id": 1003249 }',\n  'key' => 'zidok',\n  'hash' => '2343 2c 423x23xd23sd3223',\n)  \n        Call call = null;
        Call result = null;
        if (contentValues != null && contentValues.size() > 0)
        {
            StringBuilder stringBuilder = new StringBuilder();
            Set<String> set = contentValues.keySet();
            Iterator<String> iter = set.iterator();
            if (iter.hasNext())
            {
                String val = iter.next();
                stringBuilder.append(val + "=" + contentValues.getAsString(val));
            }
            while (iter.hasNext())
            {
                String val = iter.next();
                stringBuilder.append("&");
                stringBuilder.append(val + "=" + contentValues.getAsString(val));
            }
            if (url != null && url.equals("") == false && stringBuilder != null)
            {
                SDKLogger.printDebbugLog(TAG, ">>>>stringBuilder = " + stringBuilder.toString());
                RequestBody body = RequestBody.create(JSON, stringBuilder.toString());
                Request request = new Request.Builder().url(url).post(body).tag(requestTag).build();
                result = client.newCall(request);
            }
            else
            {
                com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n Invalid Server Url or Request parameters JSON !");
            }
        }
        return result;
    }

    /**
     * Make GET connection
     *
     * @param url           url
     * @param contentValues http parameters
     * @param requestTag    connection request tag
     */
    public static Call makeGetConnection(String url, ContentValues contentValues, String requestTag) throws IOException
    {

        Call call = null;

        StringBuilder sb = new StringBuilder(url);
        if (contentValues != null && contentValues.size() > 0)
        {
            sb.append("?");

            Set<String> set = contentValues.keySet();
            Iterator<String> iter = set.iterator();
            if (iter.hasNext())
            {
                String val = iter.next();
                sb.append(val + "=" + contentValues.getAsString(val));
            }
            while (iter.hasNext())
            {
                String val = iter.next();
                sb.append("&");
                sb.append(val + "=" + contentValues.getAsString(val));
            }
        }
        Request request = new Request.Builder().url(sb.toString()).tag(requestTag).build();

        call = client.newCall(request);

        return call;
    }


    /**
     * Cancel current running request connection
     *
     * @param tag connection request tag
     */
    private static void cancelRequest(String tag)
    {
        client.cancel(tag);
    }

    public static OkHttpClient getConnectionClient()
    {
        return client;
    }


    /**
     * Get gif image stream
     *
     * @param url        url
     * @param requestTag connection request tag
     */
    public static void getImageStreamAsync(String url, String requestTag, final IOnImageRequestCallback requestCallback)
    {
        Request request = new Request.Builder().url(url).tag(requestTag).build();
        client.newCall(request).enqueue(new Callback()
        {

            @Override public void onFailure(Request request, IOException e)
            {

            }

            @Override public void onResponse(Response response) throws IOException
            {

                requestCallback.onImageRecieved(response.body().byteStream());
            }
        });
    }

    /**
     * Get gif image stream
     *
     * @param url        url
     * @param requestTag connection request tag
     */
    public static Call getImageStream(String url, String requestTag) throws IOException
    {
        Request request = new Request.Builder().url(url).tag(requestTag).build();
        return client.newCall(request);
    }


    public interface IOnImageRequestCallback
    {
        public void onImageRecieved(InputStream inputStream) throws IOException;
    }
}

