package com.kidoz.events;

import android.annotation.SuppressLint;
import android.app.ActivityManager;
import android.app.AppOpsManager;
import android.content.ContentResolver;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.content.res.Configuration;
import android.database.Cursor;
import android.graphics.Point;
import android.hardware.Camera;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraManager;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Environment;
import android.provider.MediaStore;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;
import android.view.Surface;
import android.view.WindowManager;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.net.NetworkInterface;
import java.nio.channels.FileChannel;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

/**
 * Created by orikam on 8/30/15.
 */
@SuppressLint("NewApi")
public class DeviceUtils {
    private static final String TAG = DeviceUtils.class.getSimpleName();
    private static final String STOP_STATE_FLAG_SHARED_PREFERENCES = "STOP_STATE_FLAG_SHARED_PREFERENCES";

    /**
     * @return The application version - used in combination with referral
     */
    public static String getApplicationVersion(Context context) {
        String result = "";
        try {
            result = String.valueOf((context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionCode));
        } catch (Exception ex) {
            EventLogger.printErrorLog(TAG, "Error when trying to get application version: " + ex.getMessage());
        }
        return result;
    }

    public static String getDeviceReferral(Context context) {
        String publisherID = SharedPreferencesUtils.loadSharedPreferencesData(context, "PUBLISHER_ID");
        String result = "SDK_";
        if (context != null) {
            result += context.getPackageName();
            result += "_";
            if (publisherID != null) {
                result += publisherID;
            }
        }
        EventLogger.printDebbugLog(TAG, "Device Referral>> " + result);
        return result;
    }


    /**
     * @param context
     * @return the default launcher package name or null if no default is set
     */
    public static String getCurrentDefaultLauncherPackageName(Context context) {
        String result = null;
        Intent intent = new Intent(Intent.ACTION_MAIN);
        intent.addCategory(Intent.CATEGORY_HOME);
        ResolveInfo resolveInfo = context.getPackageManager().resolveActivity(intent, PackageManager.MATCH_DEFAULT_ONLY);
        if (resolveInfo != null) {
            result = resolveInfo.activityInfo.packageName;
            if (result.equals("android") == true) {
                result = null;
            }
        }
        return result;
    }

    public static void setDeviceLanguage(Context context, String languageCode) {
        if (context != null && languageCode != null) {
            String[] selectedLocale = languageCode.split("_");
            Locale locale = null;
            if (selectedLocale != null && selectedLocale.length > 1) {
                locale = new Locale(selectedLocale[0], selectedLocale[1]);
            } else if (selectedLocale != null && selectedLocale.length > 0) {
                locale = new Locale(selectedLocale[0]);
            } else {
                return;
            }
            Locale.setDefault(locale);
            Configuration config = context.getResources().getConfiguration();
            config.locale = locale;
            if (context != null) {
                context.getResources().updateConfiguration(config, context.getResources().getDisplayMetrics());
            }
        }
    }

    public static String getWiFiActivityClassName(Context context) {
        Intent wifiIntent = new Intent(Settings.ACTION_WIRELESS_SETTINGS);
        final PackageManager packageManager = context.getPackageManager();

        List<ResolveInfo> galleryActivities = packageManager.queryIntentActivities(wifiIntent, PackageManager.MATCH_DEFAULT_ONLY);
        if (galleryActivities != null && galleryActivities.size() > 0) {
            if (galleryActivities.get(0).activityInfo != null) {
                return galleryActivities.get(0).activityInfo.name;
            }
        }

        return null;
    }

    public static Intent getLaunchIntentForPackageName(Context context, String packageName) {
        if (context != null) {
            return context.getPackageManager().getLaunchIntentForPackage(packageName);
        } else {
            return null;
        }

    }

    /**
     * Get external storage file path
     *
     * @return file path
     */
    public static File getWritableExternalStorageFilePath() {
        File path = null;
        final String state = Environment.getExternalStorageState();

        if (Environment.MEDIA_MOUNTED.equals(state) || !Environment.MEDIA_MOUNTED_READ_ONLY.equals(state)) { // we
            // can
            // write
            // the
            // External
            // Storage...
            path = Environment.getExternalStorageDirectory();
        } else {
            path = null;
        }
        return path;
    }

    /**
     * Get screen size
     *
     * @return file path
     */
    @SuppressLint("NewApi")
    public static int getScreenSize(Context context, boolean isWidth) {
        int result = 0;
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = windowManager.getDefaultDisplay();
        if (isWidth) {
            if (Build.VERSION.SDK_INT >= 19) {
                Point point = new Point();
                display.getRealSize(point);
                result = point.x;
            } else {
                result = display.getWidth();
            }
        } else {
            if (Build.VERSION.SDK_INT >= 19) {
                Point point = new Point();
                display.getRealSize(point);
                result = point.y;
            } else {
                result = display.getHeight();
            }
        }
        return result;
    }

    /**
     * Get screen size
     *
     * @return file path
     */
    @SuppressLint("NewApi")
    public static Point getScreenSize(Context context) {
        Point size = new Point();
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = windowManager.getDefaultDisplay();

        if (Build.VERSION.SDK_INT >= 19) {
            display.getRealSize(size);
        } else {
            size.x = display.getWidth();
            size.y = display.getHeight();
        }

        return size;
    }

    /**
     * Get device unique Id
     *
     * @param context
     * @return device id as string representation
     */
    public static String getDeviceUniqueID(Context context) {
        String result = null;
        final TelephonyManager telephonyManager = (TelephonyManager) ((ContextWrapper) context).getBaseContext().getSystemService(Context.TELEPHONY_SERVICE);
        result = telephonyManager.getDeviceId();
        return result;
    }


    /**
     * Get network ip address
     *
     * @param interfaceName interface name
     * @return return MAC address
     */
    public static String getMACAddress(String interfaceName) {
        String result = null;
        try {
            List<NetworkInterface> interfaces = Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface intf : interfaces) {
                if (interfaceName != null) {
                    if (!intf.getName().equalsIgnoreCase(interfaceName)) {
                        continue;
                    }
                }
                byte[] mac = intf.getHardwareAddress();
                if (mac != null) {
                    StringBuilder buf = new StringBuilder();
                    for (int idx = 0; idx < mac.length; idx++) {
                        buf.append(String.format("%02X:", mac[idx]));
                    }
                    if (buf.length() > 0) {
                        buf.deleteCharAt(buf.length() - 1);
                    }
                    result = buf.toString();
                }
            }
        } catch (Exception ex) {

        }
        return result;
    }

    /**
     * Write object to internal storage
     *
     * @param context current context
     * @param key     object key
     * @param object  object to save
     */
    public static void writeObjectToInternalStorage(Context context, String key, Object object) {
        FileOutputStream fos;
        try {
            fos = context.openFileOutput(key, Context.MODE_PRIVATE);
            ObjectOutputStream oos = new ObjectOutputStream(fos);
            oos.writeObject(object);
            oos.flush();
            oos.close();
            fos.flush();
            fos.close();

        } catch (Exception e) {
            EventLogger.printErrorLog(TAG, "Error Writing OBJECT to interlal application storage !");
        }
    }

    /**
     * Read object from internal storage
     *
     * @param context current context
     * @param key     object key
     * @return object
     */
    public static Object readObjectFromInternalStorage(Context context, String key) {
        FileInputStream fis;
        Object object = null;
        try {
            fis = context.openFileInput(key);
            ObjectInputStream ois = new ObjectInputStream(fis);
            object = ois.readObject();
            ois.close();
            fis.close();
        } catch (Exception e) {
            EventLogger.printErrorLog(TAG, "Error Reading OBJECT from interlal application storage !");
        }
        return object;
    }

    /**
     * Delete object from internal storage
     *
     * @param context current context
     * @param key     object key
     * @return object
     */
    public static boolean removeObjectFromInternalStorage(Context context, String key) {
        boolean res = false;
        try {
            res = context.deleteFile(key);
        } catch (Exception e) {
            EventLogger.printErrorLog(TAG, "Error Deleteing OBJECT from interlal application storage !");
        }
        return res;
    }

    /**
     * Write object to a path
     *
     * @param filePath object key
     * @param object
     */
    public static void writeObject(File filePath, Object object) {
        try {

            if (filePath.exists()) {
                filePath.delete();
            }

            if (!filePath.exists()) {
                filePath.createNewFile();
            }

            FileOutputStream fos = new FileOutputStream(filePath);
            ObjectOutputStream out = new ObjectOutputStream(fos);
            out.writeObject(object);
            out.flush();
            out.close();
            fos.flush();
            fos.close();

        } catch (Exception ex) {
            EventLogger.printErrorLog("Error Writing object to External Cache folder \n " + ex.getMessage());
        }
    }

    /**
     * Write object to a path
     *
     * @param filePath object key
     * @return object
     */
    public static Object readObject(File filePath) {
        Object object = null;
        try {
            if (filePath.exists()) {
                // create an ObjectInputStream for the file we created before
                FileInputStream fos = new FileInputStream(filePath);
                ObjectInputStream ois = new ObjectInputStream(fos);
                object = ois.readObject();
                ois.close();
                fos.close();
            }
        } catch (Exception ex) {
            EventLogger.printErrorLog("Error Reading object from External Cache folder \n " + ex.getMessage());
        }
        return object;
    }

    /**
     * Checks if external storage is available for read and write
     *
     * @return true or false
     */
    public static boolean isExternalStorageWritable() {
        String state = Environment.getExternalStorageState();
        if (Environment.MEDIA_MOUNTED.equals(state)) {
            return true;
        }
        return false;
    }

    /**
     * Checks if external storage is available to at least read
     *
     * @return true or false
     */
    public static boolean isExternalStorageReadable() {
        String state = Environment.getExternalStorageState();
        if (Environment.MEDIA_MOUNTED.equals(state) || Environment.MEDIA_MOUNTED_READ_ONLY.equals(state)) {
            return true;
        }
        return false;
    }

    /**
     * Get or create the location of backgrounds images inside the system
     * pictures folder
     *
     * @return true or false
     */
    public static File getBackgroundsFolderLocationPath() {
        File outputDir = null;

        if (isExternalStorageWritable()) {
            File photoDir = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_PICTURES);
            outputDir = new File(photoDir, "Backgrounds");
            if (!outputDir.exists()) {
                if (!outputDir.mkdirs()) {
                    EventLogger.printErrorLog(TAG, "Failed to create directory " + outputDir.getAbsolutePath());
                    outputDir = null;
                }
            }
        }
        return outputDir;
    }

    /**
     * Get device dafault orientation
     *
     * @return int (orientation as in <code>Configuration</code> )
     */
    public static int getDeviceDefaultOrientation(Context context) {
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);

        Configuration config = context.getResources().getConfiguration();

        int rotation = windowManager.getDefaultDisplay().getRotation();

        if (((rotation == Surface.ROTATION_0 || rotation == Surface.ROTATION_180) && config.orientation == Configuration.ORIENTATION_LANDSCAPE) || ((rotation == Surface.ROTATION_90 || rotation == Surface.ROTATION_270) && config.orientation == Configuration.ORIENTATION_PORTRAIT)) {
            return Configuration.ORIENTATION_LANDSCAPE;
        } else {
            return Configuration.ORIENTATION_PORTRAIT;
        }
    }

    /**
     * Get path from uri
     *
     * @param pictureUri picture Uri
     * @return extracted path
     */
    public static String getPicturePathFromUri(Context context, Uri pictureUri) {
        String result = null;
        if (pictureUri != null) {
            String[] proj = {MediaStore.Images.Media.DATA};
            ContentResolver contentResolver = context.getContentResolver();
            Cursor cursor = contentResolver.query(pictureUri, proj, null, null, null);
            int column_index = cursor.getColumnIndexOrThrow(MediaStore.Images.Media.DATA);
            cursor.moveToFirst();
            result = cursor.getString(column_index);
        }
        return result;
    }

    /**
     * Get device DPI factor
     *
     * @return device DPI factor
     */
    public static double getDeviceDPIFactor(Context context) {
        double result = 1;
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        DisplayMetrics metrics = new DisplayMetrics();
        windowManager.getDefaultDisplay().getMetrics(metrics);

        switch (metrics.densityDpi) {
            case DisplayMetrics.DENSITY_LOW: {
                result = 0.75;
                break;
            }
            case DisplayMetrics.DENSITY_MEDIUM: {
                result = 1;
                break;
            }
            case DisplayMetrics.DENSITY_HIGH: {
                result = 1.5;
                break;
            }
            case DisplayMetrics.DENSITY_XHIGH: {
                result = 2;
                break;
            }
            case DisplayMetrics.DENSITY_XXHIGH: {
                result = 3;
                break;
            }
            case DisplayMetrics.DENSITY_XXXHIGH: {
                result = 4;
                break;
            }
        }

        return result;
    }

    /**
     * Check if device usages stats option enabled
     * (Should be used only from LOLLIPOP and up) API greater than 21
     */
    public static boolean isDeviceUsageStatsEnabled(Context context) {
        try {
            PackageManager packageManager = context.getPackageManager();
            ApplicationInfo applicationInfo = packageManager.getApplicationInfo(context.getPackageName(), 0);
            AppOpsManager appOpsManager = (AppOpsManager) context.getSystemService(Context.APP_OPS_SERVICE);
            int mode = appOpsManager.checkOpNoThrow(AppOpsManager.OPSTR_GET_USAGE_STATS, applicationInfo.uid, applicationInfo.packageName);
            return (mode == AppOpsManager.MODE_ALLOWED);

        } catch (PackageManager.NameNotFoundException e) {
            return false;
        }
    }

    /**
     * Get is KIDOZ package set as default launcher
     */
    public static boolean getIsKidozSetAsDefaultLauncher(Context context) {
        boolean result = false;
        String currentLauncherPackageName = getCurrentDefaultLauncherPackageName(context);
        if (currentLauncherPackageName != null && currentLauncherPackageName.equals(context.getPackageName()) == true) {
            result = true;
        }

        return result;
    }

    /**
     * copies content from source file to destination file
     *
     * @param sourceFile
     * @param destFile
     * @throws IOException
     */
    public static void copyFile(File sourceFile, File destFile) throws IOException {
        try {
            if (!sourceFile.exists()) {
                return;
            }

            FileChannel source = null;
            FileChannel destination = null;
            FileInputStream fis = new FileInputStream(sourceFile);
            source = fis.getChannel();
            FileOutputStream fos = new FileOutputStream(destFile);
            destination = fos.getChannel();
            if (destination != null && source != null) {
                destination.transferFrom(source, 0, source.size());
            }
            if (source != null) {
                source.close();
            }
            if (destination != null) {
                destination.close();
            }

            fis.close();
            fos.flush();
            fos.close();
        } catch (Exception e) {

        }

    }

    /**
     * This method return false if the current camera is back camera or if there
     * is an error. It's return true if the current camera facing front (selfy
     * mode).
     *
     * @param context
     * @param cameraID
     * @return
     */
    public static boolean getIsCurrentCameraFacingFront(Context context, int cameraID) {
        boolean result = false;
        if (Build.VERSION.SDK_INT < 21) {
            Camera.CameraInfo cameraInfo = new Camera.CameraInfo();
            Camera.getCameraInfo(cameraID, cameraInfo);
            if (cameraInfo.facing == Camera.CameraInfo.CAMERA_FACING_FRONT) {
                result = true;
            }
        } else {
            CameraManager cameraManager = (CameraManager) context.getSystemService(Context.CAMERA_SERVICE);
            try {
                CameraCharacteristics cameraCharacteristics = cameraManager.getCameraCharacteristics(String.valueOf(cameraID));
                int lensFacing = cameraCharacteristics.get(CameraCharacteristics.LENS_FACING);
                if (lensFacing == CameraCharacteristics.LENS_FACING_FRONT) {
                    result = true;
                }
            } catch (Exception ex) {
                EventLogger.printErrorLog(TAG, "Error when trying to get camera specs: " + ex.getMessage());
            }
        }
        return result;
    }

    /**
     * Check if some user manifest permission exists
     *
     * @param context
     */
    public static boolean isManifetsPermissionExists(Context context, String permissison) {
        int result = context.checkCallingOrSelfPermission(permissison);
        return result == PackageManager.PERMISSION_GRANTED;
    }

    /**
     * This method return wherever an app is installed for a given INTENT FILTER
     *
     * @return is app installed
     */
    public static boolean getIsAppInstalledByIntentFilter(Context context, String appIntentFilter) {
        boolean result = false;
        if (appIntentFilter != null) {
            Intent appIntent = new Intent(appIntentFilter);
            final PackageManager packageManager = context.getPackageManager();
            List<ResolveInfo> resultActivities = packageManager.queryIntentActivities(appIntent, PackageManager.MATCH_DEFAULT_ONLY);
            if (resultActivities != null && resultActivities.size() > 0) {
                result = true;
            }
        }
        return result;
    }

    /**
     * This method return packagename by given INTENT FILTER
     *
     * @return is app installed
     */
    public static String getActivityNameByIntentFilter(Context context, String appIntentFilter) {
        String name = null;
        if (appIntentFilter != null) {
            Intent appIntent = new Intent(appIntentFilter);
            final PackageManager packageManager = context.getPackageManager();
            List<ResolveInfo> resultActivities = packageManager.queryIntentActivities(appIntent, PackageManager.MATCH_DEFAULT_ONLY);
            if (resultActivities != null && resultActivities.size() > 0) {
                name = resultActivities.get(0).resolvePackageName;
            }
        }
        return name;
    }


    /**
     * This method return wherever an app is installed for a given Package Name
     *
     * @return is app installed
     */
    public static boolean getIsAppInstalledByPckgName(Context context, String packageName) {
        if (context.getPackageManager().getLaunchIntentForPackage(packageName) != null) {
            return true;
        } else {
            return false;
        }
    }


    /**
     * Copy one folder content to the other
     *
     * @param sourceLocation source folder path
     * @param targetLocation target folder path
     */
    // If targetLocation does not exist, it will be created.
    public void copyDirectory(File sourceLocation, File targetLocation) throws IOException {

        if (sourceLocation.isDirectory()) {
            if (!targetLocation.exists() && !targetLocation.mkdirs()) {
                throw new IOException("Cannot create dir " + targetLocation.getAbsolutePath());
            }

            String[] children = sourceLocation.list();
            for (int i = 0; i < children.length; i++) {
                copyDirectory(new File(sourceLocation, children[i]), new File(targetLocation, children[i]));
            }
        } else {

            // make sure the directory we plan to store the recording in exists
            File directory = targetLocation.getParentFile();
            if (directory != null && !directory.exists() && !directory.mkdirs()) {
                throw new IOException("Cannot create dir " + directory.getAbsolutePath());
            }

            InputStream in = new FileInputStream(sourceLocation);
            OutputStream out = new FileOutputStream(targetLocation);

            // Copy the bits from instream to outstream
            byte[] buf = new byte[1024];
            int len;
            while ((len = in.read(buf)) > 0) {
                out.write(buf, 0, len);
            }
            in.close();
            out.flush();
            out.close();
        }
    }

    // For to Delete the directory inside list of files and inner Directory
    public static boolean deleteFolder(File dir) {
        if (dir.isDirectory()) {
            String[] children = dir.list();
            for (int i = 0; i < children.length; i++) {
                boolean success = deleteFolder(new File(dir, children[i]));
                if (!success) {
                    return false;
                }
            }
        }

        // The directory is now empty so delete it
        return dir.delete();
    }


    /**
     * Clear user application data
     */
    public static void clearApplicationData(Context context) {
        try {
            if (Build.VERSION_CODES.KITKAT <= Build.VERSION.SDK_INT) {
                ((ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE)).clearApplicationUserData(); // note: it has a return value!
            } else {
                File cache = context.getCacheDir();
                File appDir = new File(cache.getParent());
                if (appDir.exists()) {
                    String[] children = appDir.list();
                    for (String s : children) {
                        if (!s.equals("lib")) {
                            deleteDir(new File(appDir, s));
                            Log.i("TAG", "**************** File /data/data/APP_PACKAGE/" + s + " DELETED *******************");
                        }
                    }
                }
            }
        } catch (Exception e) {

        }
    }

    /**
     * Delete directory
     *
     * @param dir diractory to delete
     */
    private static boolean deleteDir(File dir) {
        if (dir != null && dir.isDirectory()) {
            String[] children = dir.list();
            for (int i = 0; i < children.length; i++) {
                boolean success = deleteDir(new File(dir, children[i]));
                if (!success) {
                    return false;
                }
            }
        }
        return dir.delete();
    }

    /**
     * Retrieve Google advertising Id
     *
     * @param context
     */
    public static String getGoogleAdvertisingID(Context context) {
        String GOOGLE_ADVERTISING_ID_KEY = "GOOGLE_ADVERTISING_ID_KEY";
        String result = SharedPreferencesUtils.loadSharedPreferencesData(context, GOOGLE_ADVERTISING_ID_KEY);
        if (result == null) {
            generateGoogleAdvertisingIDInBackground(context);
        }
        SDKLogger.printDebbugLog(TAG, ">>>>Google advertising ID = " + result);
        return result;
    }

    /**
     * Generate google advertising id async
     *
     * @param context
     */
    private static synchronized void generateGoogleAdvertisingIDInBackground(final Context context) {
        if (context != null) {

            AsyncTask<Void, Void, Void> registerInBackground = new AsyncTask<Void, Void, Void>() {
                @Override
                protected Void doInBackground(Void... params) {
                    String GOOGLE_ADVERTISING_ID_KEY = "GOOGLE_ADVERTISING_ID_KEY";
                    try {
                        AdvertisingIdClient.Info info = AdvertisingIdClient.getAdvertisingIdInfo(context);
                        String googleAdvertisingID = info.getId();
                        SharedPreferencesUtils.saveSharedPreferencesData(context, GOOGLE_ADVERTISING_ID_KEY, googleAdvertisingID);
                    } catch (Exception ex) {
                        SDKLogger.printErrorLog(TAG, "Error when trying to get google advertising ID: " + ex.getMessage());
                    }
                    return null;
                }
            };
            registerInBackground.execute(null, null, null);
        }
    }
}
