package com.kidoz.sdk.api.server_connect;

import android.content.ContentValues;
import android.content.Context;

import com.kidoz.events.DeviceUtils;
import com.kidoz.sdk.api.analytics.EventsBulk;
import com.kidoz.sdk.api.general.enums.UiStyleType;
import com.kidoz.sdk.api.general.utils.ConstantDef;
import com.kidoz.sdk.api.general.utils.EncoderUtils;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.structure.ContentData;
import com.kidoz.sdk.api.structure.SDKConfigurationData;
import com.squareup.okhttp.Call;
import com.squareup.okhttp.Response;

import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Locale;
import java.util.Set;


public class SdkAPIManager extends BaseAPIManager
{
    private static final String TAG = SdkAPIManager.class.getName();

    private String mPublicherId = null;
    private String mSecurityToken = null;
    private String mPackageName;

    public SdkAPIManager(Context context, String pId, String sKey)
    {
        MAIN_SERVER_URL = "http://sdk.kidoz.net/api/sdk";
        KIDOZ_ANALITYCS_SERVER_URL = "http://54.197.245.43/parents/KidozRestMobile.php";

        initConnectionClient(false);
        mPublicherId = pId;
        mSecurityToken = sKey;
        mPackageName = context.getPackageName();
    }

    //SDK ApiApi Functions
    private static final String JSON = "JSON";
    private static final String REQUEST_FORMAT = "resFormat";
    private static final String WEB_SERVICE_REQUEST_PARAMETERS = "request_parameters";
    private static final String GET_CONTENT = "getContent";
    private static final String VALIDATE_SDK_CONFIGURATION = "validateSDKConfiguration";
    private static final String SEND_KIDOZ_EVENTS_TO_SERVER = "SendKidozEventsToServer";

    /**
     * Load content
     *
     * @param styleType      content view style
     * @param resultCallback result callabck
     */
    public void loadContent(Context context, UiStyleType styleType, final ApiResultCallback<ContentData> resultCallback)
    {
        ContentValues contentValues = new ContentValues();
        contentValues.put("method_name", GET_CONTENT);
        contentValues.put("publisher_id", mPublicherId);
        contentValues.put("package_id", mPackageName);
        contentValues.put("device_lang", Locale.getDefault().getLanguage());
        contentValues.put("style_id", styleType.getValue());
        contentValues.put("device_hash", com.kidoz.sdk.api.general.utils.Utils.generateUniqeDeviceID(mPackageName, mPublicherId));
        contentValues.put("google_id", DeviceUtils.getGoogleAdvertisingID(context));
        contentValues.put("version", ConstantDef.SDK_VERSION);

        long ts = System.currentTimeMillis() / 1000;
        contentValues.put("ts", ts);
        contentValues.put("hash", generateRequestHash(contentValues, ts, mSecurityToken));

        startServerConnection(context, CONNECTION_TYPE.POST, SdkRequestType.LOAD_SDK_CONTENT, contentValues, 2, resultCallback);
    }

    public void validateSDKConfiguration(Context context, ApiResultCallback<SDKConfigurationData> resultCallback)
    {
        ContentValues contentValues = new ContentValues();
        contentValues.put("method_name", VALIDATE_SDK_CONFIGURATION);
        try
        {
            JSONObject requestParameters = new JSONObject();
            requestParameters.put("publisher_id", mPublicherId);
            requestParameters.put("package_id", mPackageName);
            requestParameters.put("device_hash", com.kidoz.sdk.api.general.utils.Utils.generateUniqeDeviceID(mPackageName, mPublicherId));
            requestParameters.put("version", ConstantDef.SDK_VERSION);
            contentValues.put(WEB_SERVICE_REQUEST_PARAMETERS, requestParameters.toString());
            long ts = System.currentTimeMillis() / 1000;
            contentValues.put("ts", String.valueOf(ts));
            contentValues.put("hash", generateRequestHash(contentValues, ts, mSecurityToken));
            startServerConnection(context, CONNECTION_TYPE.POST, SdkRequestType.VALIDATE_SDK, contentValues, 1, resultCallback);
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to validateSDK: " + ex.getMessage());
            resultCallback.onFailed();
        }
    }


    /**
     * Send kidoz events to server
     *
     * @param eventsJsonBulk
     */
    public ResultData<Boolean> sendKidozEventsToServerNotAsync(String eventsJsonBulk)
    {

        ContentValues contentValues = new ContentValues();
        contentValues.put("methodName", SEND_KIDOZ_EVENTS_TO_SERVER);
        contentValues.put("KidozEventsLogAsJson", eventsJsonBulk);
        contentValues.put(REQUEST_FORMAT, JSON);

        ResultData<Boolean> resultCallback = null;

        Call call = null;
        try
        {
            call = BaseConnectionClient.makePostConnection(KIDOZ_ANALITYCS_SERVER_URL, contentValues, "analyticsRequest");
            if (call != null)
            {

                Response response = call.execute();
                if (response != null)
                {
                    if (response.isSuccessful())
                    {
                        try
                        {
                            String responce = StreamToStringConverter.readStream(new BufferedInputStream(response.body().byteStream()), null, false);
                            if (responce != null)
                            {
                                resultCallback = EventsBulk.parseResultSuccsesStatus(responce);
                            }

                        }
                        catch (IOException e)
                        {
                            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n Unable to convertByte Stream to String! [ Kidoz Analytics Data Sent ]  : \n\n" + e.getMessage());
                        }
                    }
                    else
                    {
                        com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n Server connectivity Error! [ Kidoz Analytics Data Sent ]  Code : " + response.code());
                    }
                }
            }
        }
        catch (IOException e)
        {
            com.kidoz.sdk.api.general.utils.SDKLogger.printErrorLog(" \n IO Exception On [ Kidoz Analytics Data Sent ] \n " + e.getMessage());
        }
        return resultCallback;
    }

    /**
     * Generate security hash for a server request
     */
    private String generateRequestHash(ContentValues contentValues, long ts, String token)
    {
        Set<String> set = contentValues.keySet();
        ArrayList<String> list = new ArrayList<String>();
        list.addAll(set);
        Collections.sort(list);

        StringBuilder stringBuilder = new StringBuilder();
        for (int i = 0; i < list.size(); i++)
        {
            stringBuilder.append(contentValues.getAsString(list.get(i)));
        }
        stringBuilder.append(ts);
        stringBuilder.append(token);

        return EncoderUtils.SHA1(stringBuilder.toString());
    }

}


/*
class StringComparator implements Comparator<String> {
    @Override
    public int compare(String o1, String o2) {
        return o1.compareTo(o2);
    }
}*/
